"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InnerJoin = void 0;
exports.createJoinTermSource = createJoinTermSource;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _es_term_source = require("../sources/es_term_source");
var _style_util = require("../styles/vector/style_util");
var _constants = require("../../../common/constants");
var _table_source = require("../sources/table_source");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createJoinTermSource(descriptor) {
  if (!descriptor) {
    return;
  }
  if (descriptor.type === _constants.SOURCE_TYPES.ES_TERM_SOURCE && descriptor.indexPatternId !== undefined && descriptor.term !== undefined) {
    return new _es_term_source.ESTermSource(descriptor);
  } else if (descriptor.type === _constants.SOURCE_TYPES.TABLE_SOURCE) {
    return new _table_source.TableSource(descriptor);
  }
}
class InnerJoin {
  constructor(joinDescriptor, leftSource) {
    (0, _defineProperty2.default)(this, "_descriptor", void 0);
    (0, _defineProperty2.default)(this, "_rightSource", void 0);
    (0, _defineProperty2.default)(this, "_leftField", void 0);
    this._descriptor = joinDescriptor;
    this._rightSource = createJoinTermSource(this._descriptor.right);
    this._leftField = joinDescriptor.leftField ? leftSource.createField({
      fieldName: joinDescriptor.leftField
    }) : undefined;
  }
  hasCompleteConfig() {
    return this._leftField && this._rightSource ? this._rightSource.hasCompleteConfig() : false;
  }
  getJoinFields() {
    return this._rightSource ? this._rightSource.getRightFields() : [];
  }

  // Source request id must be static and unique because the re-fetch logic uses the id to locate the previous request.
  // Elasticsearch sources have a static and unique id so that requests can be modified in the inspector.
  // Using the right source id as the source request id because it meets the above criteria.
  getSourceDataRequestId() {
    return `join_source_${this._rightSource.getId()}`;
  }
  getSourceMetaDataRequestId() {
    return `${this.getSourceDataRequestId()}_${_constants.META_DATA_REQUEST_ID_SUFFIX}`;
  }
  getSourceFormattersDataRequestId() {
    return `${this.getSourceDataRequestId()}_${_constants.FORMATTERS_DATA_REQUEST_ID_SUFFIX}`;
  }
  getLeftField() {
    if (!this._leftField) {
      throw new Error('Cannot get leftField from InnerJoin with incomplete config');
    }
    return this._leftField;
  }
  joinPropertiesToFeature(feature, propertiesMap) {
    if (!feature.properties || !this._leftField || !this._rightSource) {
      return false;
    }
    const rightMetricFields = this._rightSource.getRightFields();
    // delete feature properties added by previous join
    for (let j = 0; j < rightMetricFields.length; j++) {
      const metricPropertyKey = rightMetricFields[j].getName();
      delete feature.properties[metricPropertyKey];

      // delete all dynamic properties for metric field
      const stylePropertyPrefix = (0, _style_util.getComputedFieldNamePrefix)(metricPropertyKey);
      Object.keys(feature.properties).forEach(featurePropertyKey => {
        if (featurePropertyKey.length >= stylePropertyPrefix.length && featurePropertyKey.substring(0, stylePropertyPrefix.length) === stylePropertyPrefix) {
          delete feature.properties[featurePropertyKey];
        }
      });
    }
    const joinKey = this.getJoinKey(feature);
    if (joinKey !== null && propertiesMap.has(joinKey)) {
      Object.assign(feature.properties, propertiesMap.get(joinKey));
      return true;
    } else {
      return false;
    }
  }
  getJoinKey(feature) {
    const joinKey = feature.properties && this._leftField ? feature.properties[this._leftField.getName()] : undefined;
    return joinKey === undefined || joinKey === null ? null : joinKey.toString();
  }
  getRightJoinSource() {
    if (!this._rightSource) {
      throw new Error('Cannot get rightSource from InnerJoin with incomplete config');
    }
    return this._rightSource;
  }
  toDescriptor() {
    return this._descriptor;
  }
  async getTooltipProperties(properties, executionContext) {
    return await this.getRightJoinSource().getTooltipProperties(properties, executionContext);
  }
  getIndexPatternIds() {
    return this.getRightJoinSource().getIndexPatternIds();
  }
  getQueryableIndexPatternIds() {
    return this.getRightJoinSource().getQueryableIndexPatternIds();
  }
  getWhereQuery() {
    return this.getRightJoinSource().getWhereQuery();
  }
}
exports.InnerJoin = InnerJoin;