"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addLayer = addLayer;
exports.addLayerWithoutDataSync = addLayerWithoutDataSync;
exports.addPreviewLayers = addPreviewLayers;
exports.clearLayerAttribution = clearLayerAttribution;
exports.cloneLayer = cloneLayer;
exports.createLayerGroup = createLayerGroup;
exports.hideAllLayers = hideAllLayers;
exports.moveLayerToBottom = moveLayerToBottom;
exports.moveLayerToLeftOfTarget = moveLayerToLeftOfTarget;
exports.promotePreviewLayers = promotePreviewLayers;
exports.removeLayer = removeLayer;
exports.removePreviewLayers = removePreviewLayers;
exports.removeSelectedLayer = removeSelectedLayer;
exports.removeTrackedLayerStateForSelectedLayer = removeTrackedLayerStateForSelectedLayer;
exports.replaceLayerList = replaceLayerList;
exports.rollbackToTrackedLayerStateForSelectedLayer = rollbackToTrackedLayerStateForSelectedLayer;
exports.setAreTilesLoaded = setAreTilesLoaded;
exports.setFirstPreviewLayerToSelectedLayer = setFirstPreviewLayerToSelectedLayer;
exports.setHiddenLayers = setHiddenLayers;
exports.setJoinsForLayer = setJoinsForLayer;
exports.setLayerAttribution = setLayerAttribution;
exports.setLayerParent = setLayerParent;
exports.setLayerQuery = setLayerQuery;
exports.setLayerVisibility = setLayerVisibility;
exports.setSelectedLayer = setSelectedLayer;
exports.showAllLayers = showAllLayers;
exports.showThisLayerOnly = showThisLayerOnly;
exports.toggleLayerVisible = toggleLayerVisible;
exports.trackCurrentLayerState = trackCurrentLayerState;
exports.ungroupLayer = ungroupLayer;
exports.updateDisableTooltips = updateDisableTooltips;
exports.updateFittableFlag = updateFittableFlag;
exports.updateLabelsOnTop = updateLabelsOnTop;
exports.updateLayerAlpha = updateLayerAlpha;
exports.updateLayerById = updateLayerById;
exports.updateLayerLabel = updateLayerLabel;
exports.updateLayerLocale = updateLayerLocale;
exports.updateLayerMaxZoom = updateLayerMaxZoom;
exports.updateLayerMinZoom = updateLayerMinZoom;
exports.updateLayerOrder = updateLayerOrder;
exports.updateLayerStyle = updateLayerStyle;
exports.updateLayerStyleForSelectedLayer = updateLayerStyleForSelectedLayer;
exports.updateMetaFromTiles = updateMetaFromTiles;
exports.updateSourceProp = updateSourceProp;
exports.updateSourceProps = updateSourceProps;
var _map_selectors = require("../selectors/map_selectors");
var _ui = require("../reducers/ui");
var _non_serializable_instances = require("../reducers/non_serializable_instances");
var _ui_actions = require("./ui_actions");
var _map_action_constants = require("./map_action_constants");
var _data_request_actions = require("./data_request_actions");
var _constants = require("../../common/constants");
var _licensed_features = require("../licensed_features");
var _ui_selectors = require("../selectors/ui_selectors");
var _layer_group = require("../classes/layers/layer_group");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function trackCurrentLayerState(layerId) {
  return {
    type: _map_action_constants.TRACK_CURRENT_LAYER_STATE,
    layerId
  };
}
function rollbackToTrackedLayerStateForSelectedLayer() {
  return async (dispatch, getState) => {
    const layerId = (0, _map_selectors.getSelectedLayerId)(getState());
    await dispatch({
      type: _map_action_constants.ROLLBACK_TO_TRACKED_LAYER_STATE,
      layerId
    });

    // Ensure updateStyleMeta is triggered
    // syncDataForLayer may not trigger endDataLoad if no re-fetch is required
    dispatch((0, _data_request_actions.updateStyleMeta)(layerId));
    dispatch((0, _data_request_actions.syncDataForLayerId)(layerId, false));
  };
}
function removeTrackedLayerStateForSelectedLayer() {
  return (dispatch, getState) => {
    const layerId = (0, _map_selectors.getSelectedLayerId)(getState());
    dispatch({
      type: _map_action_constants.REMOVE_TRACKED_LAYER_STATE,
      layerId
    });
  };
}
function replaceLayerList(newLayerList) {
  return (dispatch, getState) => {
    const isMapReady = (0, _map_selectors.getMapReady)(getState());
    if (!isMapReady) {
      dispatch({
        type: _map_action_constants.CLEAR_WAITING_FOR_MAP_READY_LAYER_LIST
      });
    } else {
      (0, _map_selectors.getLayerListRaw)(getState()).forEach(({
        id
      }) => {
        dispatch(removeLayerFromLayerList(id));
      });
    }
    newLayerList.forEach(layerDescriptor => {
      dispatch(addLayer(layerDescriptor));
    });
  };
}
function updateLayerById(layerDescriptor) {
  return async (dispatch, getState) => {
    dispatch({
      type: _map_action_constants.UPDATE_LAYER,
      layer: layerDescriptor
    });
    await dispatch((0, _data_request_actions.syncDataForLayerId)(layerDescriptor.id, false));
    if ((0, _map_selectors.getMapSettings)(getState()).autoFitToDataBounds) {
      dispatch((0, _data_request_actions.autoFitToBounds)());
    }
  };
}
function cloneLayer(layerId) {
  return async (dispatch, getState) => {
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layer) {
      return;
    }
    (await layer.cloneDescriptor()).forEach(layerDescriptor => {
      dispatch(addLayer(layerDescriptor));
      if (layer.getParent()) {
        dispatch(moveLayerToLeftOfTarget(layerDescriptor.id, layerId));
      }
    });
  };
}
function addLayer(layerDescriptor) {
  return async (dispatch, getState) => {
    const isMapReady = (0, _map_selectors.getMapReady)(getState());
    if (!isMapReady) {
      dispatch({
        type: _map_action_constants.ADD_WAITING_FOR_MAP_READY_LAYER,
        layer: layerDescriptor
      });
      return;
    }
    dispatch({
      type: _map_action_constants.ADD_LAYER,
      layer: layerDescriptor
    });
    dispatch((0, _data_request_actions.syncDataForLayerId)(layerDescriptor.id, false));
    const layer = (0, _map_selectors.createLayerInstance)(layerDescriptor, []); // custom icons not needed, layer instance only used to get licensed features
    const features = await layer.getLicensedFeatures();
    features.forEach(_licensed_features.notifyLicensedFeatureUsage);
  };
}
function addLayerWithoutDataSync(layerDescriptor) {
  return {
    type: _map_action_constants.ADD_LAYER,
    layer: layerDescriptor
  };
}
function addPreviewLayers(layerDescriptors) {
  return dispatch => {
    dispatch(removePreviewLayers());
    layerDescriptors.forEach(layerDescriptor => {
      dispatch(addLayer({
        ...layerDescriptor,
        __isPreviewLayer: true
      }));

      // Auto open layer legend to increase legend discoverability
      if (layerDescriptor.style && (hasByValueStyling(layerDescriptor.style) || layerDescriptor.style.type === _constants.LAYER_STYLE_TYPE.HEATMAP)) {
        dispatch((0, _ui_actions.showTOCDetails)(layerDescriptor.id));
      }
    });
  };
}
function removePreviewLayers() {
  return (dispatch, getState) => {
    (0, _map_selectors.getLayerList)(getState()).forEach(layer => {
      if (layer.isPreviewLayer()) {
        if ((0, _layer_group.isLayerGroup)(layer)) {
          dispatch(ungroupLayer(layer.getId()));
        }
        dispatch(removeLayer(layer.getId()));
      }
    });
  };
}
function promotePreviewLayers() {
  return (dispatch, getState) => {
    (0, _map_selectors.getLayerList)(getState()).forEach(layer => {
      if (layer.isPreviewLayer()) {
        dispatch({
          type: _map_action_constants.UPDATE_LAYER_PROP,
          id: layer.getId(),
          propName: '__isPreviewLayer',
          newValue: false
        });
      }
    });
  };
}
function setLayerVisibility(layerId, makeVisible) {
  return (dispatch, getState) => {
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layer) {
      return;
    }
    if ((0, _layer_group.isLayerGroup)(layer)) {
      layer.getChildren().forEach(childLayer => {
        dispatch(setLayerVisibility(childLayer.getId(), makeVisible));
      });
    }

    // If the layer visibility is already what we want it to be, do nothing
    if (layer.isVisible() === makeVisible) {
      return;
    }
    dispatch({
      type: _map_action_constants.SET_LAYER_VISIBILITY,
      layerId,
      visibility: makeVisible
    });

    // if the current-state is invisible, we also want to sync data
    // e.g. if a layer was invisible at start-up, it won't have any data loaded
    if (makeVisible) {
      dispatch((0, _data_request_actions.syncDataForLayerId)(layerId, false));
    }
  };
}
function toggleLayerVisible(layerId) {
  return (dispatch, getState) => {
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layer) {
      return;
    }
    const makeVisible = !layer.isVisible();
    dispatch(setLayerVisibility(layerId, makeVisible));
  };
}
function hideAllLayers() {
  return (dispatch, getState) => {
    (0, _map_selectors.getLayerList)(getState()).forEach((layer, index) => {
      if (!layer.isBasemap(index)) {
        dispatch(setLayerVisibility(layer.getId(), false));
      }
    });
  };
}
function showAllLayers() {
  return (dispatch, getState) => {
    (0, _map_selectors.getLayerList)(getState()).forEach((layer, index) => {
      dispatch(setLayerVisibility(layer.getId(), true));
    });
  };
}
function showThisLayerOnly(layerId) {
  return (dispatch, getState) => {
    (0, _map_selectors.getLayerList)(getState()).forEach((layer, index) => {
      if (layer.isBasemap(index) || layer.getId() === layerId) {
        return;
      }

      // hide all other layers
      dispatch(setLayerVisibility(layer.getId(), false));
    });

    // show target layer after hiding all other layers
    // since hiding layer group will hide its children
    const targetLayer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (targetLayer) {
      dispatch(setLayerVisibility(layerId, true));
    }
  };
}
function setSelectedLayer(layerId) {
  return async (dispatch, getState) => {
    const oldSelectedLayer = (0, _map_selectors.getSelectedLayerId)(getState());
    if (oldSelectedLayer) {
      await dispatch(rollbackToTrackedLayerStateForSelectedLayer());
    }
    if (layerId) {
      dispatch(trackCurrentLayerState(layerId));
      // Reset draw mode only if setting a new selected layer
      if ((0, _ui_selectors.getDrawMode)(getState()) !== _constants.DRAW_MODE.NONE) {
        dispatch((0, _ui_actions.setDrawMode)(_constants.DRAW_MODE.NONE));
      }
    }
    dispatch({
      type: _map_action_constants.SET_SELECTED_LAYER,
      selectedLayerId: layerId
    });
  };
}
function setFirstPreviewLayerToSelectedLayer() {
  return async (dispatch, getState) => {
    const firstPreviewLayer = (0, _map_selectors.getLayerList)(getState()).find(layer => {
      return layer.isPreviewLayer();
    });
    if (firstPreviewLayer) {
      dispatch(setSelectedLayer(firstPreviewLayer.getId()));
    }
  };
}
function updateLayerOrder(newLayerOrder) {
  return {
    type: _map_action_constants.UPDATE_LAYER_ORDER,
    newLayerOrder
  };
}
function updateMetricsProp(layerId, value) {
  return async (dispatch, getState) => {
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    const previousFields = await layer.getFields();
    dispatch({
      type: _map_action_constants.UPDATE_SOURCE_PROP,
      layerId,
      propName: 'metrics',
      value
    });
    await dispatch(updateStyleProperties(layerId, previousFields));
  };
}
function updateSourcePropWithoutSync(layerId, propName, value, newLayerType) {
  return async (dispatch, getState) => {
    if (propName === 'metrics') {
      if (newLayerType) {
        throw new Error('May not change layer-type when modifying metrics source-property');
      }
      return await dispatch(updateMetricsProp(layerId, value));
    }
    dispatch({
      type: _map_action_constants.UPDATE_SOURCE_PROP,
      layerId,
      propName,
      value
    });
    if (newLayerType) {
      dispatch(updateLayerType(layerId, newLayerType));
    }
    if (propName === 'scalingType') {
      // get joins from layer descriptor instead of layer.getJoins()
      // 1) IVectorLayer implementations may return empty array when descriptor has joins
      // 2) getJoins returns instances and descriptors are needed.
      const layerDescriptor = (0, _map_selectors.getLayerDescriptor)(getState(), layerId);
      const joins = layerDescriptor.joins ? layerDescriptor.joins : [];
      if (value === _constants.SCALING_TYPES.CLUSTERS && joins.length) {
        // Blended scaling type does not support joins
        // It is not possible to display join metrics when showing clusters
        dispatch({
          type: _map_action_constants.SET_JOINS,
          layerId,
          joins: []
        });
        await dispatch(updateStyleProperties(layerId));
      } else if (value === _constants.SCALING_TYPES.MVT) {
        if (joins.length > 1) {
          // Maplibre feature-state join uses promoteId and there is a limit to one promoteId
          // Therefore, Vector tile scaling supports only one join
          dispatch({
            type: _map_action_constants.SET_JOINS,
            layerId,
            joins: [joins[0]]
          });
        }
        // update style props regardless of updating joins
        // Allow style to clean-up data driven style properties with join fields that do not support feature-state.
        await dispatch(updateStyleProperties(layerId));
      }
    }
  };
}
function updateSourceProp(layerId, propName, value, newLayerType) {
  return async dispatch => {
    await dispatch(updateSourcePropWithoutSync(layerId, propName, value, newLayerType));
    dispatch((0, _data_request_actions.syncDataForLayerId)(layerId, false));
  };
}
function updateSourceProps(layerId, sourcePropChanges) {
  return async dispatch => {
    // Using for loop to ensure update completes before starting next update
    for (let i = 0; i < sourcePropChanges.length; i++) {
      const {
        propName,
        value,
        newLayerType
      } = sourcePropChanges[i];
      await dispatch(updateSourcePropWithoutSync(layerId, propName, value, newLayerType));
    }
    dispatch((0, _data_request_actions.syncDataForLayerId)(layerId, false));
  };
}
function updateLayerType(layerId, newLayerType) {
  return (dispatch, getState) => {
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layer || layer.getType() === newLayerType) {
      return;
    }
    dispatch((0, _data_request_actions.clearDataRequests)(layer));
    clearInspectorAdapters(layer, (0, _non_serializable_instances.getInspectorAdapters)(getState()));
    dispatch({
      type: _map_action_constants.UPDATE_LAYER_PROP,
      id: layerId,
      propName: 'type',
      newValue: newLayerType
    });
  };
}
function updateLayerLabel(id, newLabel) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'label',
    newValue: newLabel
  };
}
function updateLayerLocale(id, locale) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'locale',
    newValue: locale
  };
}
function setLayerAttribution(id, attribution) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'attribution',
    newValue: attribution
  };
}
function clearLayerAttribution(id) {
  return {
    type: _map_action_constants.CLEAR_LAYER_PROP,
    id,
    propName: 'attribution'
  };
}
function updateLayerMinZoom(id, minZoom) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'minZoom',
    newValue: minZoom
  };
}
function updateLayerMaxZoom(id, maxZoom) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'maxZoom',
    newValue: maxZoom
  };
}
function updateLayerAlpha(id, alpha) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'alpha',
    newValue: alpha
  };
}
function updateLabelsOnTop(id, areLabelsOnTop) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'areLabelsOnTop',
    newValue: areLabelsOnTop
  };
}
function updateFittableFlag(id, includeInFitToBounds) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'includeInFitToBounds',
    newValue: includeInFitToBounds
  };
}
function updateDisableTooltips(id, disableTooltips) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id,
    propName: 'disableTooltips',
    newValue: disableTooltips
  };
}
function setLayerQuery(id, query) {
  return dispatch => {
    dispatch({
      type: _map_action_constants.UPDATE_LAYER_PROP,
      id,
      propName: 'query',
      newValue: query
    });
    dispatch((0, _data_request_actions.syncDataForLayerId)(id, false));
  };
}
function setLayerParent(id, parent) {
  return (dispatch, getState) => {
    dispatch({
      type: _map_action_constants.UPDATE_LAYER_PROP,
      id,
      propName: 'parent',
      newValue: parent
    });
    if (parent) {
      // Open parent layer details. Without opening parent details, layer disappears from legend and this confuses users
      dispatch((0, _ui_actions.showTOCDetails)(parent));
    }
  };
}
function removeSelectedLayer() {
  return (dispatch, getState) => {
    const state = getState();
    const layerId = (0, _map_selectors.getSelectedLayerId)(state);
    if (layerId) {
      dispatch(removeLayer(layerId));
    }
  };
}
function removeLayer(layerId) {
  return async (dispatch, getState) => {
    const state = getState();
    const selectedLayerId = (0, _map_selectors.getSelectedLayerId)(state);
    if (layerId === selectedLayerId) {
      dispatch((0, _ui_actions.updateFlyout)(_ui.FLYOUT_STATE.NONE));
      await dispatch(setSelectedLayer(null));
    }
    dispatch(removeLayerFromLayerList(layerId));
  };
}
function removeLayerFromLayerList(layerId) {
  return (dispatch, getState) => {
    const layerGettingRemoved = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layerGettingRemoved) {
      return;
    }
    layerGettingRemoved.getInFlightRequestTokens().forEach(requestToken => {
      dispatch((0, _non_serializable_instances.cancelRequest)(requestToken));
    });
    clearInspectorAdapters(layerGettingRemoved, (0, _non_serializable_instances.getInspectorAdapters)(getState()));
    dispatch({
      type: _map_action_constants.REMOVE_LAYER,
      id: layerId
    });
    // Clean up draw state if needed
    const editState = (0, _map_selectors.getEditState)(getState());
    if (layerId === (editState === null || editState === void 0 ? void 0 : editState.layerId)) {
      dispatch((0, _ui_actions.setDrawMode)(_constants.DRAW_MODE.NONE));
    }
    const openTOCDetails = (0, _ui_selectors.getOpenTOCDetails)(getState());
    if (openTOCDetails.includes(layerId)) {
      dispatch((0, _ui_actions.hideTOCDetails)(layerId));
    }
    if ((0, _layer_group.isLayerGroup)(layerGettingRemoved)) {
      layerGettingRemoved.getChildren().forEach(childLayer => {
        dispatch(removeLayerFromLayerList(childLayer.getId()));
      });
    }
  };
}
function updateStyleProperties(layerId, previousFields) {
  return async (dispatch, getState) => {
    const targetLayer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!targetLayer) {
      return;
    }
    const style = targetLayer.getCurrentStyle();
    if (!style || style.getType() !== _constants.LAYER_STYLE_TYPE.VECTOR) {
      return;
    }
    if (!('getFields' in targetLayer)) {
      return;
    }
    const nextFields = await targetLayer.getFields(); // take into account all fields, since labels can be driven by any field (source or join)
    const {
      hasChanges,
      nextStyleDescriptor
    } = await style.getDescriptorWithUpdatedStyleProps(nextFields, (0, _map_selectors.getMapColors)(getState()), previousFields);
    if (hasChanges && nextStyleDescriptor) {
      dispatch(updateLayerStyle(layerId, nextStyleDescriptor));
    }
  };
}
function updateLayerStyle(layerId, styleDescriptor) {
  return dispatch => {
    dispatch({
      type: _map_action_constants.UPDATE_LAYER_STYLE,
      layerId,
      style: {
        ...styleDescriptor
      }
    });

    // Auto open layer legend to increase legend discoverability
    if (hasByValueStyling(styleDescriptor)) {
      dispatch((0, _ui_actions.showTOCDetails)(layerId));
    }

    // Ensure updateStyleMeta is triggered
    // syncDataForLayer may not trigger endDataLoad if no re-fetch is required
    dispatch((0, _data_request_actions.updateStyleMeta)(layerId));

    // Style update may require re-fetch, for example ES search may need to retrieve field used for dynamic styling
    dispatch((0, _data_request_actions.syncDataForLayerId)(layerId, false));
  };
}
function updateLayerStyleForSelectedLayer(styleDescriptor) {
  return (dispatch, getState) => {
    const selectedLayerId = (0, _map_selectors.getSelectedLayerId)(getState());
    if (!selectedLayerId) {
      return;
    }
    dispatch(updateLayerStyle(selectedLayerId, styleDescriptor));
  };
}
function setJoinsForLayer(layer, joins) {
  return async dispatch => {
    const previousFields = await layer.getFields();
    dispatch({
      type: _map_action_constants.SET_JOINS,
      layerId: layer.getId(),
      joins
    });
    await dispatch(updateStyleProperties(layer.getId(), previousFields));
    dispatch((0, _data_request_actions.syncDataForLayerId)(layer.getId(), false));
  };
}
function setHiddenLayers(hiddenLayerIds) {
  return (dispatch, getState) => {
    const isMapReady = (0, _map_selectors.getMapReady)(getState());
    if (!isMapReady) {
      dispatch({
        type: _map_action_constants.SET_WAITING_FOR_READY_HIDDEN_LAYERS,
        hiddenLayerIds
      });
    } else {
      (0, _map_selectors.getLayerListRaw)(getState()).forEach(layer => dispatch(setLayerVisibility(layer.id, !hiddenLayerIds.includes(layer.id))));
    }
  };
}
function setAreTilesLoaded(layerId, areTilesLoaded) {
  return {
    type: _map_action_constants.UPDATE_LAYER_PROP,
    id: layerId,
    propName: '__areTilesLoaded',
    newValue: areTilesLoaded
  };
}
function updateMetaFromTiles(layerId, mbMetaFeatures) {
  return async (dispatch, getState) => {
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layer) {
      return;
    }
    dispatch({
      type: _map_action_constants.UPDATE_LAYER_PROP,
      id: layerId,
      propName: '__metaFromTiles',
      newValue: mbMetaFeatures
    });
    await dispatch((0, _data_request_actions.updateStyleMeta)(layerId));
  };
}
function clearInspectorAdapters(layer, adapters) {
  if ((0, _layer_group.isLayerGroup)(layer) || !layer.getSource().isESSource()) {
    return;
  }
  if (adapters.vectorTiles) {
    adapters.vectorTiles.removeLayer(layer.getId());
  }
  if (adapters.requests && 'getValidJoins' in layer) {
    const vectorLayer = layer;
    adapters.requests.resetRequest(layer.getSource().getId());
    vectorLayer.getValidJoins().forEach(join => {
      adapters.requests.resetRequest(join.getRightJoinSource().getId());
    });
  }
}
function hasByValueStyling(styleDescriptor) {
  return styleDescriptor.type === _constants.LAYER_STYLE_TYPE.VECTOR && Object.values(styleDescriptor.properties).some(styleProperty => {
    return (styleProperty === null || styleProperty === void 0 ? void 0 : styleProperty.type) === _constants.STYLE_TYPE.DYNAMIC;
  });
}
function createLayerGroup(draggedLayerId, combineLayerId) {
  return (dispatch, getState) => {
    const group = _layer_group.LayerGroup.createDescriptor({});
    const combineLayerDescriptor = (0, _map_selectors.getLayerDescriptor)(getState(), combineLayerId);
    if (combineLayerDescriptor !== null && combineLayerDescriptor !== void 0 && combineLayerDescriptor.parent) {
      group.parent = combineLayerDescriptor.parent;
    }
    dispatch({
      type: _map_action_constants.ADD_LAYER,
      layer: group
    });
    // Move group to left of combine-layer
    dispatch(moveLayerToLeftOfTarget(group.id, combineLayerId));
    dispatch((0, _ui_actions.showTOCDetails)(group.id));
    dispatch(setLayerParent(draggedLayerId, group.id));
    dispatch(setLayerParent(combineLayerId, group.id));

    // Move dragged-layer to left of combine-layer
    dispatch(moveLayerToLeftOfTarget(draggedLayerId, combineLayerId));
  };
}
function ungroupLayer(layerId) {
  return (dispatch, getState) => {
    const layer = (0, _map_selectors.getLayerList)(getState()).find(findLayer => findLayer.getId() === layerId);
    if (!layer || !(0, _layer_group.isLayerGroup)(layer)) {
      return;
    }
    layer.getChildren().forEach(childLayer => {
      dispatch(setLayerParent(childLayer.getId(), layer.getParent()));
    });
  };
}
function moveLayerToLeftOfTarget(moveLayerId, targetLayerId) {
  return (dispatch, getState) => {
    const layers = (0, _map_selectors.getLayerList)(getState());
    const moveLayerIndex = layers.findIndex(layer => layer.getId() === moveLayerId);
    const targetLayerIndex = layers.findIndex(layer => layer.getId() === targetLayerId);
    if (moveLayerIndex === -1 || targetLayerIndex === -1) {
      return;
    }
    const moveLayer = layers[moveLayerIndex];
    const newIndex = moveLayerIndex > targetLayerIndex ?
    // When layer is moved to the right, new left sibling index is to the left of destination
    targetLayerIndex + 1 :
    // When layer is moved to the left, new left sibling index is the destination index
    targetLayerIndex;
    const newOrder = [];
    for (let i = 0; i < layers.length; i++) {
      newOrder.push(i);
    }
    newOrder.splice(moveLayerIndex, 1);
    newOrder.splice(newIndex, 0, moveLayerIndex);
    dispatch(updateLayerOrder(newOrder));
    if ((0, _layer_group.isLayerGroup)(moveLayer)) {
      moveLayer.getChildren().forEach(childLayer => {
        dispatch(moveLayerToLeftOfTarget(childLayer.getId(), targetLayerId));
      });
    }
  };
}
function moveLayerToBottom(moveLayerId) {
  return (dispatch, getState) => {
    const layers = (0, _map_selectors.getLayerList)(getState());
    const moveLayerIndex = layers.findIndex(layer => layer.getId() === moveLayerId);
    if (moveLayerIndex === -1) {
      return;
    }
    const moveLayer = layers[moveLayerIndex];
    const newIndex = 0;
    const newOrder = [];
    for (let i = 0; i < layers.length; i++) {
      newOrder.push(i);
    }
    newOrder.splice(moveLayerIndex, 1);
    newOrder.splice(newIndex, 0, moveLayerIndex);
    dispatch(updateLayerOrder(newOrder));
    if ((0, _layer_group.isLayerGroup)(moveLayer)) {
      moveLayer.getChildren().forEach(childLayer => {
        dispatch(moveLayerToBottom(childLayer.getId()));
      });
    }
  };
}