"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSuggestions = getSuggestions;
exports.getTopSuggestionForField = getTopSuggestionForField;
exports.getVisualizeFieldSuggestions = getVisualizeFieldSuggestions;
exports.switchToSuggestion = switchToSuggestion;
var _public = require("@kbn/expression-xy-plugin/public");
var _lens_ui_errors = require("../../lens_ui_errors");
var _add_layer = require("./config_panel/add_layer");
var _state_management = require("../../state_management");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This function takes a list of available data tables and a list of visualization
 * extensions and creates a ranked list of suggestions which contain a pair of a data table
 * and a visualization.
 *
 * Each suggestion represents a valid state of the editor and can be applied by creating an
 * action with `toSwitchAction` and dispatching it
 */
function getSuggestions({
  datasourceMap,
  datasourceStates,
  visualizationMap,
  activeVisualization,
  subVisualizationId,
  visualizationState,
  field,
  visualizeTriggerFieldContext,
  activeData,
  dataViews,
  mainPalette
}) {
  const datasources = Object.entries(datasourceMap).filter(([datasourceId]) => datasourceStates[datasourceId] && !datasourceStates[datasourceId].isLoading);
  const layerTypesMap = datasources.reduce((memo, [datasourceId, datasource]) => {
    const datasourceState = datasourceStates[datasourceId].state;
    if (!activeVisualization || !datasourceState) {
      return memo;
    }
    const layers = datasource.getLayers(datasourceState);
    for (const layerId of layers) {
      const type = (0, _add_layer.getLayerType)(activeVisualization, visualizationState, layerId);
      memo[layerId] = type;
    }
    return memo;
  }, {});
  const isLayerSupportedByVisualization = (layerId, supportedTypes) => {
    var _layerTypesMap$layerI;
    return supportedTypes.includes((_layerTypesMap$layerI = layerTypesMap[layerId]) !== null && _layerTypesMap$layerI !== void 0 ? _layerTypesMap$layerI : _public.LayerTypes.DATA);
  };

  // Collect all table suggestions from available datasources
  const datasourceTableSuggestions = datasources.flatMap(([datasourceId, datasource]) => {
    const datasourceState = datasourceStates[datasourceId].state;
    let dataSourceSuggestions;
    try {
      // context is used to pass the state from location to datasource
      if (visualizeTriggerFieldContext) {
        // used for navigating from VizEditor to Lens
        if ('isVisualizeAction' in visualizeTriggerFieldContext) {
          dataSourceSuggestions = datasource.getDatasourceSuggestionsForVisualizeCharts(datasourceState, visualizeTriggerFieldContext.layers, dataViews.indexPatterns);
        } else {
          // used for navigating from Discover to Lens
          dataSourceSuggestions = datasource.getDatasourceSuggestionsForVisualizeField(datasourceState, visualizeTriggerFieldContext.dataViewSpec.id, visualizeTriggerFieldContext.fieldName, dataViews.indexPatterns);
        }
      } else if (field) {
        dataSourceSuggestions = datasource.getDatasourceSuggestionsForField(datasourceState, field, layerId => isLayerSupportedByVisualization(layerId, [_public.LayerTypes.DATA]),
        // a field dragged to workspace should added to data layer
        dataViews.indexPatterns);
      } else {
        dataSourceSuggestions = datasource.getDatasourceSuggestionsFromCurrentState(datasourceState, dataViews.indexPatterns, layerId => isLayerSupportedByVisualization(layerId, [_public.LayerTypes.DATA]), activeData);
      }
    } catch (error) {
      (0, _lens_ui_errors.showMemoizedErrorNotification)(error);
      return [];
    }
    return dataSourceSuggestions.map(suggestion => ({
      ...suggestion,
      datasourceId
    }));
  });
  // Pass all table suggestions to all visualization extensions to get visualization suggestions
  // and rank them by score
  return Object.entries(visualizationMap).flatMap(([visualizationId, visualization]) => {
    const supportedLayerTypes = visualization.getSupportedLayers().map(({
      type
    }) => type);
    return datasourceTableSuggestions.filter(datasourceSuggestion => {
      const filteredCount = datasourceSuggestion.keptLayerIds.filter(layerId => isLayerSupportedByVisualization(layerId, supportedLayerTypes)).length;
      // make it pass either suggestions with some ids left after filtering
      // or suggestion with already 0 ids before the filtering (testing purposes)
      return filteredCount || filteredCount === datasourceSuggestion.keptLayerIds.length;
    }).flatMap(datasourceSuggestion => {
      var _activeVisualization$;
      const table = datasourceSuggestion.table;
      const currentVisualizationState = visualizationId === (activeVisualization === null || activeVisualization === void 0 ? void 0 : activeVisualization.id) ? visualizationState : undefined;
      const palette = mainPalette || (activeVisualization === null || activeVisualization === void 0 ? void 0 : (_activeVisualization$ = activeVisualization.getMainPalette) === null || _activeVisualization$ === void 0 ? void 0 : _activeVisualization$.call(activeVisualization, visualizationState));
      return getVisualizationSuggestions(visualization, table, visualizationId, {
        ...datasourceSuggestion,
        keptLayerIds: datasourceSuggestion.keptLayerIds.filter(layerId => isLayerSupportedByVisualization(layerId, supportedLayerTypes))
      }, currentVisualizationState, subVisualizationId, palette, visualizeTriggerFieldContext && 'isVisualizeAction' in visualizeTriggerFieldContext, activeData);
    });
  }).sort((a, b) => b.score - a.score);
}
function getVisualizeFieldSuggestions({
  datasourceMap,
  datasourceStates,
  visualizationMap,
  visualizeTriggerFieldContext,
  dataViews
}) {
  const activeVisualization = (visualizationMap === null || visualizationMap === void 0 ? void 0 : visualizationMap[Object.keys(visualizationMap)[0]]) || null;
  const suggestions = getSuggestions({
    datasourceMap,
    datasourceStates,
    visualizationMap,
    activeVisualization,
    visualizationState: undefined,
    visualizeTriggerFieldContext,
    dataViews
  });
  if (visualizeTriggerFieldContext && 'isVisualizeAction' in visualizeTriggerFieldContext) {
    var _visualization$getSug;
    const allSuggestions = suggestions.filter(s => s.visualizationId === visualizeTriggerFieldContext.type);
    const visualization = visualizationMap[visualizeTriggerFieldContext.type] || null;
    return visualization === null || visualization === void 0 ? void 0 : (_visualization$getSug = visualization.getSuggestionFromConvertToLensContext) === null || _visualization$getSug === void 0 ? void 0 : _visualization$getSug.call(visualization, {
      suggestions: allSuggestions,
      context: visualizeTriggerFieldContext
    });
  }
  // suggestions for visualizing textbased languages
  if (visualizeTriggerFieldContext && 'query' in visualizeTriggerFieldContext) {
    if (visualizeTriggerFieldContext.query) {
      return suggestions.find(s => s.datasourceId === 'textBased');
    }
  }
  if (suggestions.length) {
    return suggestions.find(s => s.visualizationId === (activeVisualization === null || activeVisualization === void 0 ? void 0 : activeVisualization.id)) || suggestions[0];
  }
}

/**
 * Queries a single visualization extensions for a single datasource suggestion and
 * creates an array of complete suggestions containing both the target datasource
 * state and target visualization state along with suggestion meta data like score,
 * title and preview expression.
 */
function getVisualizationSuggestions(visualization, table, visualizationId, datasourceSuggestion, currentVisualizationState, subVisualizationId, mainPalette, isFromContext, activeData) {
  try {
    return visualization.getSuggestions({
      table,
      state: currentVisualizationState,
      keptLayerIds: datasourceSuggestion.keptLayerIds,
      subVisualizationId,
      mainPalette,
      isFromContext,
      activeData
    }).map(({
      state,
      ...visualizationSuggestion
    }) => ({
      ...visualizationSuggestion,
      visualizationId,
      visualizationState: state,
      keptLayerIds: datasourceSuggestion.keptLayerIds,
      datasourceState: datasourceSuggestion.state,
      datasourceId: datasourceSuggestion.datasourceId,
      columns: table.columns.length,
      changeType: table.changeType
    }));
  } catch (e) {
    (0, _lens_ui_errors.showMemoizedErrorNotification)(e);
    return [];
  }
}
function switchToSuggestion(dispatchLens, suggestion, options) {
  dispatchLens((0, _state_management.switchVisualization)({
    suggestion: {
      newVisualizationId: suggestion.visualizationId,
      visualizationState: suggestion.visualizationState,
      datasourceState: suggestion.datasourceState,
      datasourceId: suggestion.datasourceId
    },
    clearStagedPreview: options === null || options === void 0 ? void 0 : options.clearStagedPreview
  }));
  if (options !== null && options !== void 0 && options.applyImmediately) {
    dispatchLens((0, _state_management.applyChanges)());
  }
}
function getTopSuggestionForField(datasourceLayers, visualization, datasourceStates, visualizationMap, datasource, field, dataViews) {
  var _activeVisualization$2;
  const hasData = Object.values(datasourceLayers).some(datasourceLayer => datasourceLayer && datasourceLayer.getTableSpec().length > 0);
  const activeVisualization = visualization.activeId ? visualizationMap[visualization.activeId] : undefined;
  const mainPalette = activeVisualization === null || activeVisualization === void 0 ? void 0 : (_activeVisualization$2 = activeVisualization.getMainPalette) === null || _activeVisualization$2 === void 0 ? void 0 : _activeVisualization$2.call(activeVisualization, visualization.state);
  const suggestions = getSuggestions({
    datasourceMap: {
      [datasource.id]: datasource
    },
    datasourceStates,
    visualizationMap: hasData && visualization.activeId ? {
      [visualization.activeId]: activeVisualization
    } : visualizationMap,
    activeVisualization,
    visualizationState: visualization.state,
    field,
    mainPalette,
    dataViews
  });
  return suggestions.find(s => s.visualizationId === visualization.activeId) || suggestions.filter(suggestion => !suggestion.hide)[0] || suggestions[0];
}