"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDocumentationSections = getDocumentationSections;
exports.getFunctionSignatureLabel = getFunctionSignatureLabel;
exports.getHelpTextContent = getHelpTextContent;
var _react = _interopRequireDefault(require("react"));
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/kibana-react-plugin/public");
var _lodash = require("lodash");
var _util = require("../util");
var _math_completion = require("./math_completion");
var _validation = require("../validation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getDocumentationSections({
  indexPattern,
  operationDefinitionMap
}) {
  const helpGroups = [];
  helpGroups.push({
    label: _i18n.i18n.translate('xpack.lens.formulaDocumentationHeading', {
      defaultMessage: 'How it works'
    }),
    items: []
  });
  helpGroups.push({
    label: _i18n.i18n.translate('xpack.lens.formulaFrequentlyUsedHeading', {
      defaultMessage: 'Common formulas'
    }),
    description: _i18n.i18n.translate('xpack.lens.formulaCommonFormulaDocumentation', {
      defaultMessage: `The most common formulas are dividing two values to produce a percent. To display accurately, set "value format" to "percent".`
    }),
    items: [{
      label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.filterRatio', {
        defaultMessage: 'Filter ratio'
      }),
      description: /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: _i18n.i18n.translate('xpack.lens.formulaDocumentation.filterRatioDescription.markdown', {
          defaultMessage: `### Filter ratio:

Use \`kql=''\` to filter one set of documents and compare it to other documents within the same grouping.
For example, to see how the error rate changes over time:

\`\`\`
count(kql='response.status_code > 400') / count()
\`\`\`
        `,
          description: 'Text is in markdown. Do not translate function names, special characters, or field names like sum(bytes)'
        })
      })
    }, {
      label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.weekOverWeek', {
        defaultMessage: 'Week over week'
      }),
      description: /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: _i18n.i18n.translate('xpack.lens.formulaDocumentation.weekOverWeekDescription.markdown', {
          defaultMessage: `### Week over week:

Use \`shift='1w'\` to get the value of each grouping from
the previous week. Time shift should not be used with the *Top values* function.

\`\`\`
percentile(system.network.in.bytes, percentile=99) /
percentile(system.network.in.bytes, percentile=99, shift='1w')
\`\`\`
        `,
          description: 'Text is in markdown. Do not translate function names, special characters, or field names like sum(bytes)'
        })
      })
    }, {
      label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.percentOfTotal', {
        defaultMessage: 'Percent of total'
      }),
      description: /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: _i18n.i18n.translate('xpack.lens.formulaDocumentation.percentOfTotalDescription.markdown', {
          defaultMessage: `### Percent of total

Formulas can calculate \`overall_sum\` for all the groupings,
which lets you convert each grouping into a percent of total:

\`\`\`
sum(products.base_price) / overall_sum(sum(products.base_price))
\`\`\`
        `,
          description: 'Text is in markdown. Do not translate function names, special characters, or field names like sum(bytes)'
        })
      })
    }, {
      label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.recentChange', {
        defaultMessage: 'Recent change'
      }),
      description: /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: _i18n.i18n.translate('xpack.lens.formulaDocumentation.recentChangeDescription.markdown', {
          defaultMessage: `### Recent change

Use \`reducedTimeRange='30m'\` to add an additional filter on the time range of a metric aligned with the end of the global time range. This can be used to calculate how much a value changed recently.

\`\`\`
max(system.network.in.bytes, reducedTimeRange="30m")
 - min(system.network.in.bytes, reducedTimeRange="30m")
\`\`\`
        `,
          description: 'Text is in markdown. Do not translate function names, special characters, or field names like sum(bytes)'
        })
      })
    }]
  });
  helpGroups.push({
    label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.elasticsearchSection', {
      defaultMessage: 'Elasticsearch'
    }),
    description: _i18n.i18n.translate('xpack.lens.formulaDocumentation.elasticsearchSectionDescription', {
      defaultMessage: 'These functions will be executed on the raw documents for each row of the resulting table, aggregating all documents matching the break down dimensions into a single value.'
    }),
    items: []
  });
  const {
    elasticsearch: esFunction,
    calculation: calculationFunction,
    math: mathOperations,
    comparison: comparisonOperations
  } = (0, _lodash.groupBy)((0, _math_completion.getPossibleFunctions)(indexPattern), key => {
    if (key in operationDefinitionMap) {
      var _operationDefinitionM;
      return (_operationDefinitionM = operationDefinitionMap[key].documentation) === null || _operationDefinitionM === void 0 ? void 0 : _operationDefinitionM.section;
    }
    if (key in _util.tinymathFunctions) {
      return _util.tinymathFunctions[key].section;
    }
  });

  // Es aggs
  helpGroups[2].items.push(...esFunction.sort().map(key => {
    var _operationDefinitionM2, _operationDefinitionM3;
    return {
      label: key,
      description: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement("h3", null, key, "(", (_operationDefinitionM2 = operationDefinitionMap[key].documentation) === null || _operationDefinitionM2 === void 0 ? void 0 : _operationDefinitionM2.signature, ")"), (_operationDefinitionM3 = operationDefinitionMap[key].documentation) !== null && _operationDefinitionM3 !== void 0 && _operationDefinitionM3.description ? /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: operationDefinitionMap[key].documentation.description
      }) : null)
    };
  }));
  helpGroups.push({
    label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.columnCalculationSection', {
      defaultMessage: 'Column calculations'
    }),
    description: _i18n.i18n.translate('xpack.lens.formulaDocumentation.columnCalculationSectionDescription', {
      defaultMessage: 'These functions are executed for each row, but are provided with the whole column as context. This is also known as a window function.'
    }),
    items: []
  });

  // Calculations aggs
  helpGroups[3].items.push(...calculationFunction.sort().map(key => {
    var _operationDefinitionM4, _operationDefinitionM5;
    return {
      label: key,
      description: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement("h3", null, key, "(", (_operationDefinitionM4 = operationDefinitionMap[key].documentation) === null || _operationDefinitionM4 === void 0 ? void 0 : _operationDefinitionM4.signature, ")"), (_operationDefinitionM5 = operationDefinitionMap[key].documentation) !== null && _operationDefinitionM5 !== void 0 && _operationDefinitionM5.description ? /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: operationDefinitionMap[key].documentation.description
      }) : null)
    };
  }));
  helpGroups.push({
    label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.mathSection', {
      defaultMessage: 'Math'
    }),
    description: _i18n.i18n.translate('xpack.lens.formulaDocumentation.mathSectionDescription', {
      defaultMessage: 'These functions will be executed for reach row of the resulting table using single values from the same row calculated using other functions.'
    }),
    items: []
  });
  const mathFns = mathOperations.sort().map(key => {
    const [description, examples] = _util.tinymathFunctions[key].help.split(`\`\`\``);
    return {
      label: key,
      description: description.replace(/\n/g, '\n\n'),
      examples: examples ? `\`\`\`${examples}\`\`\`` : ''
    };
  });
  helpGroups[4].items.push(...mathFns.map(({
    label,
    description,
    examples
  }) => {
    return {
      label,
      description: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement("h3", null, getFunctionSignatureLabel(label, operationDefinitionMap)), /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: `${description}${examples}`
      }))
    };
  }));
  helpGroups.push({
    label: _i18n.i18n.translate('xpack.lens.formulaDocumentation.comparisonSection', {
      defaultMessage: 'Comparison'
    }),
    description: _i18n.i18n.translate('xpack.lens.formulaDocumentation.comparisonSectionDescription', {
      defaultMessage: 'These functions are used to perform value comparison.'
    }),
    items: []
  });
  const comparisonFns = comparisonOperations.sort().map(key => {
    const [description, examples] = _util.tinymathFunctions[key].help.split(`\`\`\``);
    return {
      label: key,
      description: description.replace(/\n/g, '\n\n'),
      examples: examples ? `\`\`\`${examples}\`\`\`` : ''
    };
  });
  helpGroups[5].items.push(...comparisonFns.map(({
    label,
    description,
    examples
  }) => {
    return {
      label,
      description: /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement("h3", null, getFunctionSignatureLabel(label, operationDefinitionMap)), /*#__PURE__*/_react.default.createElement(_public.Markdown, {
        markdown: `${description}${examples}`
      }))
    };
  }));
  const sections = {
    groups: helpGroups,
    initialSection: /*#__PURE__*/_react.default.createElement(_public.Markdown, {
      markdown: _i18n.i18n.translate('xpack.lens.formulaDocumentation.markdown', {
        defaultMessage: `## How it works

Lens formulas let you do math using a combination of Elasticsearch aggregations and
math functions. There are three main types of functions:

* Elasticsearch metrics, like \`sum(bytes)\`
* Time series functions use Elasticsearch metrics as input, like \`cumulative_sum()\`
* Math functions like \`round()\`

An example formula that uses all of these:

\`\`\`
round(100 * moving_average(
average(cpu.load.pct),
window=10,
kql='datacenter.name: east*'
))
\`\`\`

Elasticsearch functions take a field name, which can be in quotes. \`sum(bytes)\` is the same
as \`sum('bytes')\`.

Some functions take named arguments, like \`moving_average(count(), window=5)\`.

Elasticsearch metrics can be filtered using KQL or Lucene syntax. To add a filter, use the named
parameter \`kql='field: value'\` or \`lucene=''\`. Always use single quotes when writing KQL or Lucene
queries. If your search has a single quote in it, use a backslash to escape, like: \`kql='Women's'\'

Math functions can take positional arguments, like pow(count(), 3) is the same as count() * count() * count()

Use the symbols +, -, /, and * to perform basic math.
      `,
        description: 'Text is in markdown. Do not translate function names, special characters, or field names like sum(bytes)'
      })
    })
  };
  return sections;
}
function getFunctionSignatureLabel(name, operationDefinitionMap, firstParam) {
  if (_util.tinymathFunctions[name]) {
    return `${name}(${_util.tinymathFunctions[name].positionalArguments.map(({
      name: argName,
      optional,
      type
    }) => `[${argName}]${optional ? '?' : ''}: ${type}`).join(', ')})`;
  }
  if (operationDefinitionMap[name]) {
    var _def$documentation;
    const def = operationDefinitionMap[name];
    const extraArgs = [];
    if (def.filterable) {
      extraArgs.push(_i18n.i18n.translate('xpack.lens.formula.kqlExtraArguments', {
        defaultMessage: '[kql]?: string, [lucene]?: string'
      }));
    }
    if (def.shiftable) {
      extraArgs.push(_i18n.i18n.translate('xpack.lens.formula.shiftExtraArguments', {
        defaultMessage: '[shift]?: string'
      }));
    }
    if (def.canReduceTimeRange) {
      extraArgs.push(_i18n.i18n.translate('xpack.lens.formula.reducedTimeRangeExtraArguments', {
        defaultMessage: '[reducedTimeRange]?: string'
      }));
    }
    const extraComma = extraArgs.length ? ', ' : '';
    return `${name}(${(_def$documentation = def.documentation) === null || _def$documentation === void 0 ? void 0 : _def$documentation.signature}${extraComma}${extraArgs.join(', ')})`;
  }
  return '';
}
function getFunctionArgumentsStringified(params) {
  return params.map(({
    name,
    type: argType,
    defaultValue = 5
  }) => `${name}=${argType === 'string' ? `"${defaultValue}"` : defaultValue}`).join(', ');
}

/**
 * Get an array of strings containing all possible information about a specific
 * operation type: examples and infos.
 */
function getHelpTextContent(type, operationDefinitionMap) {
  var _definition$documenta, _definition$documenta2;
  const definition = operationDefinitionMap[type];
  const description = (_definition$documenta = (_definition$documenta2 = definition.documentation) === null || _definition$documenta2 === void 0 ? void 0 : _definition$documenta2.description) !== null && _definition$documenta !== void 0 ? _definition$documenta : '';

  // as for the time being just add examples text.
  // Later will enrich with more information taken from the operation definitions.
  const examples = [];
  // If the description already contain examples skip it
  if (!/Example/.test(description)) {
    if (!(0, _validation.hasFunctionFieldArgument)(type)) {
      // ideally this should have the same example automation as the operations below
      examples.push(`${type}()`);
      return {
        description,
        examples
      };
    }
    if (definition.input === 'field') {
      var _definition$operation;
      const mandatoryArgs = ((_definition$operation = definition.operationParams) === null || _definition$operation === void 0 ? void 0 : _definition$operation.filter(({
        required
      }) => required)) || [];
      if (mandatoryArgs.length === 0) {
        examples.push(`${type}(bytes)`);
      }
      if (mandatoryArgs.length) {
        const additionalArgs = getFunctionArgumentsStringified(mandatoryArgs);
        examples.push(`${type}(bytes, ${additionalArgs})`);
      }
      if (definition.operationParams && mandatoryArgs.length !== definition.operationParams.length) {
        const additionalArgs = getFunctionArgumentsStringified(definition.operationParams);
        examples.push(`${type}(bytes, ${additionalArgs})`);
      }
    }
    if (definition.input === 'fullReference') {
      var _definition$operation2;
      const mandatoryArgs = ((_definition$operation2 = definition.operationParams) === null || _definition$operation2 === void 0 ? void 0 : _definition$operation2.filter(({
        required
      }) => required)) || [];
      if (mandatoryArgs.length === 0) {
        examples.push(`${type}(sum(bytes))`);
      }
      if (mandatoryArgs.length) {
        const additionalArgs = getFunctionArgumentsStringified(mandatoryArgs);
        examples.push(`${type}(sum(bytes), ${additionalArgs})`);
      }
      if (definition.operationParams && mandatoryArgs.length !== definition.operationParams.length) {
        const additionalArgs = getFunctionArgumentsStringified(definition.operationParams);
        examples.push(`${type}(sum(bytes), ${additionalArgs})`);
      }
    }
  }
  return {
    description,
    examples
  };
}