"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.changeIndexPattern = changeIndexPattern;
exports.changeLayerIndexPattern = changeLayerIndexPattern;
exports.extractReferences = extractReferences;
exports.injectReferences = injectReferences;
exports.loadInitialState = loadInitialState;
exports.onRefreshIndexPattern = onRefreshIndexPattern;
exports.renameIndexPattern = renameIndexPattern;
exports.triggerActionOnIndexPatternChange = triggerActionOnIndexPatternChange;
var _lodash = require("lodash");
var _public = require("@kbn/unified-search-plugin/public");
var _operations = require("./operations");
var _settings_storage = require("../../settings_storage");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function onRefreshIndexPattern() {
  if (_operations.memoizedGetAvailableOperationsByMetadata.cache.clear) {
    // clear operations meta data cache because index pattern reference may change
    _operations.memoizedGetAvailableOperationsByMetadata.cache.clear();
  }
}
const getLastUsedIndexPatternId = (storage, indexPatternRefs) => {
  var _indexPatternRefs$fin;
  const indexPattern = (0, _settings_storage.readFromStorage)(storage, 'indexPatternId');
  return indexPattern && ((_indexPatternRefs$fin = indexPatternRefs.find(i => i.id === indexPattern)) === null || _indexPatternRefs$fin === void 0 ? void 0 : _indexPatternRefs$fin.id);
};
const setLastUsedIndexPatternId = (storage, value) => {
  (0, _settings_storage.writeToStorage)(storage, 'indexPatternId', value);
};
function getLayerReferenceName(layerId) {
  return `indexpattern-datasource-layer-${layerId}`;
}
function extractReferences({
  layers
}) {
  const savedObjectReferences = [];
  const persistableState = {
    layers: {}
  };
  Object.entries(layers).forEach(([layerId, {
    indexPatternId,
    ...persistableLayer
  }]) => {
    persistableState.layers[layerId] = persistableLayer;
    savedObjectReferences.push({
      type: 'index-pattern',
      id: indexPatternId,
      name: getLayerReferenceName(layerId)
    });
  });
  return {
    savedObjectReferences,
    state: persistableState
  };
}
function injectReferences(state, references) {
  const layers = {};
  Object.entries(state.layers).forEach(([layerId, persistedLayer]) => {
    var _references$find;
    const indexPatternId = (_references$find = references.find(({
      name
    }) => name === getLayerReferenceName(layerId))) === null || _references$find === void 0 ? void 0 : _references$find.id;
    if (indexPatternId) {
      layers[layerId] = {
        ...persistedLayer,
        indexPatternId
      };
    }
  });
  return {
    layers
  };
}
function createStateFromPersisted({
  persistedState,
  references
}) {
  return persistedState && references ? injectReferences(persistedState, references) : undefined;
}
function getUsedIndexPatterns({
  state,
  indexPatternRefs,
  storage,
  initialContext,
  defaultIndexPatternId
}) {
  var _indexPatternRefs$;
  const lastUsedIndexPatternId = getLastUsedIndexPatternId(storage, indexPatternRefs);
  const fallbackId = lastUsedIndexPatternId || defaultIndexPatternId || ((_indexPatternRefs$ = indexPatternRefs[0]) === null || _indexPatternRefs$ === void 0 ? void 0 : _indexPatternRefs$.id);
  const indexPatternIds = [];
  if (initialContext) {
    if ('isVisualizeAction' in initialContext) {
      indexPatternIds.push(...initialContext.indexPatternIds);
    } else {
      indexPatternIds.push(initialContext.dataViewSpec.id);
    }
  }
  const usedPatterns = (initialContext ? indexPatternIds : (0, _lodash.uniq)(state ? Object.values(state.layers).map(l => l.indexPatternId) : [fallbackId])

  // take out the undefined from the list
  ).filter(Boolean);
  return {
    usedPatterns,
    allIndexPatternIds: indexPatternIds
  };
}
function loadInitialState({
  persistedState,
  references,
  defaultIndexPatternId,
  storage,
  initialContext,
  indexPatternRefs = [],
  indexPatterns = {}
}) {
  const state = createStateFromPersisted({
    persistedState,
    references
  });
  const {
    usedPatterns,
    allIndexPatternIds: indexPatternIds
  } = getUsedIndexPatterns({
    state,
    defaultIndexPatternId,
    storage,
    initialContext,
    indexPatternRefs
  });
  const availableIndexPatterns = new Set(indexPatternRefs.map(({
    id
  }) => id));
  const notUsedPatterns = (0, _lodash.difference)([...availableIndexPatterns], usedPatterns);

  // Priority list:
  // * start with the indexPattern in context
  // * then fallback to the used ones
  // * then as last resort use a first one from not used refs
  const currentIndexPatternId = [...indexPatternIds, ...usedPatterns, ...notUsedPatterns].find(id => id != null && availableIndexPatterns.has(id) && indexPatterns[id]);
  if (currentIndexPatternId) {
    setLastUsedIndexPatternId(storage, currentIndexPatternId);
  }
  return {
    layers: {},
    ...state,
    currentIndexPatternId
  };
}
function changeIndexPattern({
  indexPatternId,
  state,
  storage,
  indexPatterns
}) {
  setLastUsedIndexPatternId(storage, indexPatternId);
  return {
    ...state,
    layers: isSingleEmptyLayer(state.layers) ? (0, _lodash.mapValues)(state.layers, layer => (0, _operations.updateLayerIndexPattern)(layer, indexPatterns[indexPatternId])) : state.layers,
    currentIndexPatternId: indexPatternId
  };
}
function renameIndexPattern({
  oldIndexPatternId,
  newIndexPatternId,
  state
}) {
  return {
    ...state,
    layers: (0, _lodash.mapValues)(state.layers, layer => layer.indexPatternId === oldIndexPatternId ? {
      ...layer,
      indexPatternId: newIndexPatternId
    } : layer),
    currentIndexPatternId: state.currentIndexPatternId === oldIndexPatternId ? newIndexPatternId : oldIndexPatternId
  };
}
function triggerActionOnIndexPatternChange({
  state,
  layerId,
  uiActions,
  indexPatternId
}) {
  var _state$layers$layerId;
  const fromDataView = (_state$layers$layerId = state.layers[layerId]) === null || _state$layers$layerId === void 0 ? void 0 : _state$layers$layerId.indexPatternId;
  if (!fromDataView) return;
  const toDataView = indexPatternId;
  const trigger = uiActions.getTrigger(_public.UPDATE_FILTER_REFERENCES_TRIGGER);
  const action = uiActions.getAction(_public.UPDATE_FILTER_REFERENCES_ACTION);
  action === null || action === void 0 ? void 0 : action.execute({
    trigger,
    fromDataView,
    toDataView,
    defaultDataView: toDataView,
    usedDataViews: Object.values(Object.values(state.layers).map(layer => layer.indexPatternId))
  });
}
function changeLayerIndexPattern({
  indexPatternId,
  indexPatterns,
  layerIds,
  state,
  replaceIfPossible,
  storage
}) {
  setLastUsedIndexPatternId(storage, indexPatternId);
  const newLayers = {
    ...state.layers
  };
  layerIds.forEach(layerId => {
    newLayers[layerId] = (0, _operations.updateLayerIndexPattern)(state.layers[layerId], indexPatterns[indexPatternId]);
  });
  return {
    ...state,
    layers: newLayers,
    currentIndexPatternId: replaceIfPossible ? indexPatternId : state.currentIndexPatternId
  };
}
function isSingleEmptyLayer(layerMap) {
  const layers = Object.values(layerMap);
  return layers.length === 1 && layers[0].columnOrder.length === 0;
}