"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDatasourceSuggestionsForField = getDatasourceSuggestionsForField;
exports.getDatasourceSuggestionsForVisualizeCharts = getDatasourceSuggestionsForVisualizeCharts;
exports.getDatasourceSuggestionsForVisualizeField = getDatasourceSuggestionsForVisualizeField;
exports.getDatasourceSuggestionsFromCurrentState = getDatasourceSuggestionsFromCurrentState;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _id_generator = require("../../id_generator");
var _form_based = require("./form_based");
var _operations = require("./operations");
var _pure_utils = require("./pure_utils");
var _document_field = require("./document_field");
var _formula = require("./operations/definitions/formula");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function buildSuggestion({
  state,
  updatedLayer,
  layerId,
  label,
  changeType
}) {
  const updatedState = updatedLayer ? {
    ...state,
    layers: {
      ...state.layers,
      [layerId]: updatedLayer
    }
  } : state;

  // It's fairly easy to accidentally introduce a mismatch between
  // columnOrder and columns, so this is a safeguard to ensure the
  // two match up.
  const layers = (0, _lodash.mapValues)(updatedState.layers, layer => ({
    ...layer,
    columns: (0, _lodash.pick)(layer.columns, layer.columnOrder)
  }));
  const columnOrder = layers[layerId].columnOrder;
  const columnMap = layers[layerId].columns;
  const isMultiRow = Object.values(columnMap).some(column => column.isBucketed);
  return {
    state: {
      ...updatedState,
      layers
    },
    table: {
      columns: columnOrder
      // Hide any referenced columns from what visualizations know about
      .filter(columnId => !(0, _operations.isReferenced)(layers[layerId], columnId)).map(columnId => ({
        columnId,
        operation: (0, _form_based.columnToOperation)(columnMap[columnId])
      })),
      isMultiRow,
      layerId,
      changeType,
      label
    },
    keptLayerIds: Object.keys(state.layers)
  };
}
function getDatasourceSuggestionsForField(state, indexPatternId, field, indexPatterns, filterLayers) {
  const layers = Object.keys(state.layers);
  let layerIds = layers.filter(id => state.layers[id].indexPatternId === indexPatternId);
  if (filterLayers) {
    layerIds = layerIds.filter(filterLayers);
  }
  if (layerIds.length === 0) {
    // The field we're suggesting on does not match any existing layer.
    // This generates a set of suggestions where we add a layer.
    // A second set of suggestions is generated for visualizations that don't work with layers
    const newId = (0, _id_generator.generateId)();
    return getEmptyLayerSuggestionsForField(state, newId, indexPatternId, field, indexPatterns).concat(getEmptyLayerSuggestionsForField({
      ...state,
      layers: {}
    }, newId, indexPatternId, field, indexPatterns));
  } else {
    // The field we're suggesting on matches an existing layer. In this case we find the layer with
    // the fewest configured columns and try to add the field to this table. If this layer does not
    // contain any layers yet, behave as if there is no layer.
    const mostEmptyLayerId = (0, _lodash.minBy)(layerIds, layerId => state.layers[layerId].columnOrder.length);
    if (state.layers[mostEmptyLayerId].columnOrder.length === 0) {
      return getEmptyLayerSuggestionsForField(state, mostEmptyLayerId, indexPatternId, field, indexPatterns);
    } else {
      return getExistingLayerSuggestionsForField(state, mostEmptyLayerId, field, indexPatterns);
    }
  }
}

// Called when the user navigates from Visualize editor to Lens
function getDatasourceSuggestionsForVisualizeCharts(state, contextLayers, indexPatterns) {
  return getEmptyLayersSuggestionsForVisualizeCharts(state, contextLayers, indexPatterns);
}
function getEmptyLayersSuggestionsForVisualizeCharts(state, contextLayers, indexPatterns) {
  const suggestions = [];
  contextLayers.forEach(layer => {
    const indexPattern = indexPatterns[layer.indexPatternId];
    if (!indexPattern) return [];
    const newLayer = createNewLayerWithMetricAggregationFromVizEditor(indexPattern, layer);
    const suggestion = buildSuggestion({
      state,
      updatedLayer: newLayer,
      layerId: layer.layerId,
      changeType: 'initial'
    });
    suggestions.push(suggestion);
  });
  return suggestions;
}
function isReferenceColumn(column) {
  return 'references' in column && column.references.length > 0;
}
function isFieldBasedColumn(column) {
  return 'sourceField' in column;
}
function isTermsColumn(column) {
  return column.operationType === 'terms';
}
function getSourceField(column, indexPattern) {
  return isFieldBasedColumn(column) ? column.sourceField === 'document' ? _document_field.documentField : indexPattern.getFieldByName(column.sourceField) : undefined;
}
function getParams(column) {
  return {
    ...column.params
  };
}
function getIncompleteParams(column) {
  return {
    filter: column.filter,
    timeShift: column.timeShift,
    timeScale: column.timeScale,
    dataType: column.dataType,
    ...(column.reducedTimeRange && {
      reducedTimeRange: column.reducedTimeRange
    })
  };
}
function getFieldWithLabel(column, indexPattern) {
  const field = getSourceField(column, indexPattern);
  if (field && column.label) {
    return {
      ...field,
      customLabel: column.label
    };
  }
  return field;
}
function createColumnChange(column, indexPattern) {
  return {
    op: column.operationType,
    columnId: column.columnId,
    field: getFieldWithLabel(column, indexPattern),
    indexPattern,
    visualizationGroups: [],
    incompleteParams: getIncompleteParams(column),
    initialParams: {
      params: getParams(column)
    },
    columnParams: getParams(column)
  };
}
function convertToColumnChange(columns, indexPattern) {
  return columns.reduce((acc, column) => {
    if (!columns.some(c => isReferenceColumn(c) && column.columnId === c.references[0])) {
      const newColumn = createColumnChange(column, indexPattern);
      if (isReferenceColumn(column)) {
        const referenceColumn = columns.find(c => c.columnId === column.references[0]);
        newColumn.references = [createColumnChange(referenceColumn, indexPattern)];
      }
      if (isTermsColumn(column) && column.params.orderAgg && newColumn.columnParams && !columns.some(c => {
        var _column$params$orderA;
        return c.columnId === ((_column$params$orderA = column.params.orderAgg) === null || _column$params$orderA === void 0 ? void 0 : _column$params$orderA.columnId);
      })) {
        var _column$params$orderA2;
        const orderColumn = column.params.orderAgg;
        const operationDefinition = _operations.operationDefinitionMap[orderColumn.operationType];
        const layer = {
          indexPatternId: indexPattern.id,
          columns: {},
          columnOrder: []
        };
        newColumn.columnParams.orderAgg = operationDefinition.buildColumn({
          previousColumn: {
            ...column.params.orderAgg,
            label: ((_column$params$orderA2 = column.params.orderAgg) === null || _column$params$orderA2 === void 0 ? void 0 : _column$params$orderA2.label) || ''
          },
          indexPattern,
          layer,
          referenceIds: [],
          field: getFieldWithLabel(column.params.orderAgg, indexPattern)
        }, column.params);
      }
      acc.push(newColumn);
    }
    return acc;
  }, []);
}
function createNewLayerWithMetricAggregationFromVizEditor(indexPattern, layer) {
  const columns = convertToColumnChange(layer.columns, indexPattern);
  let newLayer = {
    indexPatternId: indexPattern.id,
    columns: {},
    columnOrder: []
  };
  columns.forEach(column => {
    if (column.op === 'formula') {
      var _column$columnParams;
      const operationDefinition = _operations.operationDefinitionMap.formula;
      const previousColumn = layer.columns.find(c => c.columnId === column.columnId);
      const newColumn = operationDefinition.buildColumn({
        previousColumn: {
          ...previousColumn,
          label: (previousColumn === null || previousColumn === void 0 ? void 0 : previousColumn.label) || ((_column$columnParams = column.columnParams) === null || _column$columnParams === void 0 ? void 0 : _column$columnParams.formula) || ''
        },
        indexPattern,
        layer: newLayer
      }, column.columnParams);
      newLayer = (0, _formula.insertOrReplaceFormulaColumn)(column.columnId, newColumn, newLayer, {
        indexPattern
      }).layer;
    } else {
      newLayer = (0, _operations.insertNewColumn)({
        ...column,
        layer: newLayer,
        respectOrder: true
      });
    }
  });
  let updatedLayer = newLayer;
  layer.columns.forEach(({
    columnId,
    label: customLabel
  }) => {
    if (customLabel) {
      updatedLayer = (0, _operations.updateColumnLabel)({
        layer: updatedLayer,
        columnId,
        customLabel: (0, _operations.isReferenced)(updatedLayer, columnId) ? '' : customLabel
      });
    }
  });
  return updatedLayer;
}

// Called when the user navigates from Discover to Lens (Visualize button)
function getDatasourceSuggestionsForVisualizeField(state, indexPatternId, fieldName, indexPatterns) {
  const layers = Object.keys(state.layers);
  const layerIds = layers.filter(id => state.layers[id].indexPatternId === indexPatternId);
  // Identify the field by the indexPatternId and the fieldName
  const indexPattern = indexPatterns[indexPatternId];
  const field = indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.getFieldByName(fieldName);
  if (layerIds.length !== 0 || !field) return [];
  const newId = (0, _id_generator.generateId)();
  return getEmptyLayerSuggestionsForField(state, newId, indexPatternId, field, indexPatterns).concat(getEmptyLayerSuggestionsForField({
    ...state,
    layers: {}
  }, newId, indexPatternId, field, indexPatterns));
}

// TODO: Stop hard-coding the specific operation types
function getBucketOperation(field) {
  // We allow numeric bucket types in some cases, but it's generally not the right suggestion,
  // so we eliminate it here.
  if (field.type !== 'number') {
    return (0, _operations.getOperationTypesForField)(field).find(op => op === 'date_histogram' || op === 'terms');
  }
}
function getExistingLayerSuggestionsForField(state, layerId, field, indexPatterns) {
  const layer = state.layers[layerId];
  const indexPattern = indexPatterns[layer.indexPatternId];
  const operations = (0, _operations.getOperationTypesForField)(field);
  const usableAsBucketOperation = getBucketOperation(field);
  const fieldInUse = Object.values(layer.columns).some(column => (0, _pure_utils.hasField)(column) && column.sourceField === field.name);
  const suggestions = [];
  if (usableAsBucketOperation && !fieldInUse) {
    if (usableAsBucketOperation === 'date_histogram' && layer.columnOrder.some(colId => layer.columns[colId].operationType === 'date_histogram')) {
      const previousDate = layer.columnOrder.find(colId => layer.columns[colId].operationType === 'date_histogram');
      suggestions.push(buildSuggestion({
        state,
        updatedLayer: (0, _operations.replaceColumn)({
          layer,
          indexPattern,
          field,
          op: usableAsBucketOperation,
          columnId: previousDate,
          visualizationGroups: []
        }),
        layerId,
        changeType: 'initial'
      }));
    } else {
      suggestions.push(buildSuggestion({
        state,
        updatedLayer: (0, _operations.insertNewColumn)({
          layer,
          indexPattern,
          field,
          op: usableAsBucketOperation,
          columnId: (0, _id_generator.generateId)(),
          visualizationGroups: []
        }),
        layerId,
        changeType: 'extended'
      }));
    }
  }
  if (!usableAsBucketOperation && operations.length > 0 && !fieldInUse) {
    const [metricOperation] = (0, _operations.getMetricOperationTypes)(field);
    if (metricOperation) {
      const layerWithNewMetric = (0, _operations.insertNewColumn)({
        layer,
        indexPattern,
        field,
        columnId: (0, _id_generator.generateId)(),
        op: metricOperation.type,
        visualizationGroups: []
      });
      if (layerWithNewMetric) {
        suggestions.push(buildSuggestion({
          state,
          layerId,
          updatedLayer: layerWithNewMetric,
          changeType: 'extended'
        }));
      }
      const [, metrics, references] = (0, _operations.getExistingColumnGroups)(layer);
      // TODO: Write test for the case where we have exactly one metric and one reference. We shouldn't switch the inner metric.
      if (metrics.length === 1 && references.length === 0) {
        const layerWithReplacedMetric = (0, _operations.replaceColumn)({
          layer,
          indexPattern,
          field,
          columnId: metrics[0],
          op: metricOperation.type,
          visualizationGroups: []
        });
        if (layerWithReplacedMetric) {
          suggestions.push(buildSuggestion({
            state,
            layerId,
            updatedLayer: layerWithReplacedMetric,
            changeType: 'extended'
          }));
        }
      }
    }
  }
  if (!fieldInUse) {
    const metricSuggestion = createMetricSuggestion(indexPattern, layerId, state, field);
    if (metricSuggestion) {
      suggestions.push(metricSuggestion);
    }
  }
  return suggestions;
}
function getEmptyLayerSuggestionsForField(state, layerId, indexPatternId, field, indexPatterns) {
  const indexPattern = indexPatterns[indexPatternId];
  let newLayer;
  const bucketOperation = getBucketOperation(field);
  if (bucketOperation) {
    newLayer = createNewLayerWithBucketAggregation(indexPattern, field, bucketOperation);
  } else if (indexPattern.timeFieldName && (0, _operations.getOperationTypesForField)(field).length > 0) {
    newLayer = createNewLayerWithMetricAggregation(indexPattern, field);
  }

  // copy the sampling rate to the new layer
  // or just default to 1
  if (newLayer) {
    var _state$layers$layerId, _state$layers$layerId2;
    newLayer.sampling = (_state$layers$layerId = (_state$layers$layerId2 = state.layers[layerId]) === null || _state$layers$layerId2 === void 0 ? void 0 : _state$layers$layerId2.sampling) !== null && _state$layers$layerId !== void 0 ? _state$layers$layerId : 1;
  }
  const newLayerSuggestions = newLayer ? [buildSuggestion({
    state,
    updatedLayer: newLayer,
    layerId,
    changeType: 'initial'
  })] : [];
  const metricLayer = createMetricSuggestion(indexPattern, layerId, state, field);
  return metricLayer ? newLayerSuggestions.concat(metricLayer) : newLayerSuggestions;
}
function createNewLayerWithBucketAggregation(indexPattern, field, operation) {
  return (0, _operations.insertNewColumn)({
    op: operation,
    layer: (0, _operations.insertNewColumn)({
      op: 'count',
      layer: {
        indexPatternId: indexPattern.id,
        columns: {},
        columnOrder: []
      },
      columnId: (0, _id_generator.generateId)(),
      field: _document_field.documentField,
      indexPattern,
      visualizationGroups: []
    }),
    columnId: (0, _id_generator.generateId)(),
    field,
    indexPattern,
    visualizationGroups: []
  });
}
function createNewLayerWithMetricAggregation(indexPattern, field) {
  const dateField = indexPattern.getFieldByName(indexPattern.timeFieldName);
  const [metricOperation] = (0, _operations.getMetricOperationTypes)(field);
  if (!metricOperation) {
    return;
  }
  return (0, _operations.insertNewColumn)({
    op: 'date_histogram',
    layer: (0, _operations.insertNewColumn)({
      op: metricOperation.type,
      layer: {
        indexPatternId: indexPattern.id,
        columns: {},
        columnOrder: []
      },
      columnId: (0, _id_generator.generateId)(),
      field,
      indexPattern,
      visualizationGroups: []
    }),
    columnId: (0, _id_generator.generateId)(),
    field: dateField,
    indexPattern,
    visualizationGroups: []
  });
}
function getDatasourceSuggestionsFromCurrentState(state, indexPatterns, filterLayers = () => true) {
  const layers = Object.entries(state.layers || {}).filter(([layerId]) => filterLayers(layerId));
  if (layers.length > 1) {
    // Return suggestions that reduce the data to each layer individually
    return layers.map(([layerId, layer], index) => {
      const hasMatchingLayer = layers.some(([otherLayerId, otherLayer]) => otherLayerId !== layerId && otherLayer.indexPatternId === layer.indexPatternId);
      const suggestionTitle = hasMatchingLayer ? _i18n.i18n.translate('xpack.lens.indexPatternSuggestion.removeLayerPositionLabel', {
        defaultMessage: 'Show only layer {layerNumber}',
        values: {
          layerNumber: index + 1
        }
      }) : _i18n.i18n.translate('xpack.lens.indexPatternSuggestion.removeLayerLabel', {
        defaultMessage: 'Show only {indexPatternTitle}',
        values: {
          indexPatternTitle: indexPatterns[layer.indexPatternId].title
        }
      });
      return buildSuggestion({
        state: {
          ...state,
          layers: {
            [layerId]: layer
          }
        },
        layerId,
        changeType: 'layers',
        label: suggestionTitle
      });
    }).concat([buildSuggestion({
      state,
      layerId: layers[0][0],
      changeType: 'unchanged'
    })]);
  }
  return (0, _lodash.flatten)(layers.filter(([_id, layer]) => layer.columnOrder.length && layer.indexPatternId).map(([layerId, layer]) => {
    const indexPattern = indexPatterns[layer.indexPatternId];
    const [buckets, metrics, references] = (0, _operations.getExistingColumnGroups)(layer);
    const timeDimension = layer.columnOrder.find(columnId => layer.columns[columnId].isBucketed && layer.columns[columnId].dataType === 'date');
    const timeField = (indexPattern === null || indexPattern === void 0 ? void 0 : indexPattern.timeFieldName) && indexPattern.getFieldByName(indexPattern.timeFieldName);
    const hasNumericDimension = buckets.length === 1 && buckets.some(columnId => layer.columns[columnId].dataType === 'number');
    const suggestions = [];

    // Always suggest an unchanged table, including during invalid states
    suggestions.push(buildSuggestion({
      state,
      layerId,
      changeType: 'unchanged'
    }));
    if (!references.length && metrics.length && buckets.length === 0) {
      if (timeField && buckets.length < 1 && !(0, _operations.hasTermsWithManyBuckets)(layer)) {
        // suggest current metric over time if there is a default time field
        suggestions.push(createSuggestionWithDefaultDateHistogram(state, layerId, timeField, indexPatterns));
      }
      if (indexPattern) {
        suggestions.push(...createAlternativeMetricSuggestions(indexPattern, layerId, state));
      }
    } else {
      suggestions.push(...createSimplifiedTableSuggestions(state, layerId));

      // base range intervals are of number dataType.
      // Custom range/intervals have a different dataType so they still receive the Over Time suggestion
      if (!timeDimension && timeField && buckets.length < 2 && !hasNumericDimension && !(0, _operations.hasTermsWithManyBuckets)(layer)) {
        // suggest current configuration over time if there is a default time field
        // and no time dimension yet
        suggestions.push(createSuggestionWithDefaultDateHistogram(state, layerId, timeField, indexPatterns));
      }
      if (buckets.length === 2) {
        suggestions.push(createChangedNestingSuggestion(state, layerId, indexPatterns));
      }
    }
    return suggestions;
  }));
}
function createChangedNestingSuggestion(state, layerId, indexPatterns) {
  var _indexPattern$getFiel, _indexPattern$getFiel2;
  const layer = state.layers[layerId];
  const [firstBucket, secondBucket, ...rest] = layer.columnOrder;
  const updatedLayer = {
    ...layer,
    columnOrder: [secondBucket, firstBucket, ...rest]
  };
  const indexPattern = indexPatterns[state.currentIndexPatternId];
  const firstBucketColumn = layer.columns[firstBucket];
  const firstBucketLabel = (0, _pure_utils.hasField)(firstBucketColumn) && ((_indexPattern$getFiel = indexPattern.getFieldByName(firstBucketColumn.sourceField)) === null || _indexPattern$getFiel === void 0 ? void 0 : _indexPattern$getFiel.displayName) || '';
  const secondBucketColumn = layer.columns[secondBucket];
  const secondBucketLabel = (0, _pure_utils.hasField)(secondBucketColumn) && ((_indexPattern$getFiel2 = indexPattern.getFieldByName(secondBucketColumn.sourceField)) === null || _indexPattern$getFiel2 === void 0 ? void 0 : _indexPattern$getFiel2.displayName) || '';
  return buildSuggestion({
    state,
    layerId,
    updatedLayer,
    label: getNestedTitle([secondBucketLabel, firstBucketLabel]),
    changeType: 'reorder'
  });
}
function createMetricSuggestion(indexPattern, layerId, state, field) {
  const [operation] = (0, _operations.getMetricOperationTypes)(field);
  if (!operation) {
    return;
  }
  return buildSuggestion({
    layerId,
    state,
    changeType: 'initial',
    updatedLayer: (0, _operations.insertNewColumn)({
      layer: {
        indexPatternId: indexPattern.id,
        columns: {},
        columnOrder: []
      },
      columnId: (0, _id_generator.generateId)(),
      op: operation.type,
      field: operation.type === 'count' ? _document_field.documentField : field,
      indexPattern,
      visualizationGroups: []
    })
  });
}
function getNestedTitle([outerBucketLabel, innerBucketLabel]) {
  return _i18n.i18n.translate('xpack.lens.indexpattern.suggestions.nestingChangeLabel', {
    defaultMessage: '{innerOperation} for each {outerOperation}',
    values: {
      innerOperation: innerBucketLabel,
      outerOperation: outerBucketLabel
    }
  });
}

// Replaces all metrics on the table with a different field-based function
function createAlternativeMetricSuggestions(indexPattern, layerId, state) {
  const layer = state.layers[layerId];
  const suggestions = [];
  const topLevelMetricColumns = layer.columnOrder.filter(columnId => !(0, _operations.isReferenced)(layer, columnId));
  topLevelMetricColumns.forEach(columnId => {
    const column = layer.columns[columnId];
    if (!(0, _pure_utils.hasField)(column)) {
      return;
    }
    const field = indexPattern.getFieldByName(column.sourceField);
    if (!field) {
      return;
    }
    const possibleOperations = (0, _operations.getMetricOperationTypes)(field).filter(({
      type
    }) => type !== column.operationType);
    if (possibleOperations.length) {
      const layerWithNewMetric = (0, _operations.replaceColumn)({
        layer,
        indexPattern,
        field,
        columnId,
        op: possibleOperations[0].type,
        visualizationGroups: []
      });
      if (layerWithNewMetric) {
        suggestions.push(buildSuggestion({
          state,
          layerId,
          updatedLayer: layerWithNewMetric,
          changeType: 'initial'
        }));
      }
    }
  });
  return suggestions;
}
function createSuggestionWithDefaultDateHistogram(state, layerId, timeField, indexPatterns) {
  const layer = state.layers[layerId];
  const indexPattern = indexPatterns[layer.indexPatternId];
  return buildSuggestion({
    state,
    layerId,
    updatedLayer: (0, _operations.insertNewColumn)({
      layer,
      indexPattern,
      field: timeField,
      op: 'date_histogram',
      columnId: (0, _id_generator.generateId)(),
      visualizationGroups: []
    }),
    label: _i18n.i18n.translate('xpack.lens.indexpattern.suggestions.overTimeLabel', {
      defaultMessage: 'Over time'
    }),
    changeType: 'extended'
  });
}
function createSimplifiedTableSuggestions(state, layerId) {
  const layer = state.layers[layerId];
  const [availableBucketedColumns, availableMetricColumns] = (0, _lodash.partition)(layer.columnOrder, colId => layer.columns[colId].isBucketed);
  const topLevelMetricColumns = availableMetricColumns.filter(columnId => !(0, _operations.isReferenced)(layer, columnId));
  return (0, _lodash.flatten)(availableBucketedColumns.map((_col, index) => {
    // build suggestions with fewer buckets
    const bucketedColumns = availableBucketedColumns.slice(0, index + 1);
    const allMetricsSuggestion = {
      ...layer,
      columnOrder: [...bucketedColumns, ...availableMetricColumns],
      noBuckets: false
    };
    if (bucketedColumns.length > 0 && topLevelMetricColumns.length > 1) {
      return [{
        ...layer,
        columnOrder: [...bucketedColumns, topLevelMetricColumns[0], ...(0, _operations.getReferencedColumnIds)(layer, topLevelMetricColumns[0])],
        noBuckets: false
      }];
    } else if (availableBucketedColumns.length > 1) {
      return allMetricsSuggestion;
    }
    return [];
  })).concat(
  // if there is just a single top level metric, the unchanged suggestion will take care of this case - only split up if there are multiple metrics or at least one bucket
  availableBucketedColumns.length > 0 || topLevelMetricColumns.length > 1 ? topLevelMetricColumns.map(columnId => {
    return {
      ...layer,
      columnOrder: [columnId, ...(0, _operations.getReferencedColumnIds)(layer, columnId)],
      noBuckets: true
    };
  }) : []).map(({
    noBuckets,
    ...updatedLayer
  }) => {
    return buildSuggestion({
      state,
      layerId,
      updatedLayer,
      changeType: 'reduced',
      label: noBuckets ? getMetricSuggestionTitle(updatedLayer, availableMetricColumns.length === 1) : undefined
    });
  });
}
function getMetricSuggestionTitle(layer, onlySimpleMetric) {
  const {
    operationType,
    label
  } = layer.columns[layer.columnOrder[0]];
  return _i18n.i18n.translate('xpack.lens.indexpattern.suggestions.overallLabel', {
    defaultMessage: '{operation} overall',
    values: {
      operation: onlySimpleMetric ? _operations.operationDefinitionMap[operationType].displayName : label
    },
    description: 'Title of a suggested chart containing only a single numerical metric calculated over all available data'
  });
}