"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.onDrop = onDrop;
var _utils = require("../../../../utils");
var _types = require("../../../../types");
var _operations = require("../../operations");
var _state_helpers = require("../../state_helpers");
var _get_drop_props = require("./get_drop_props");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function onDrop(props) {
  const {
    target,
    source,
    dropType,
    state,
    indexPatterns
  } = props;
  if ((0, _utils.isDraggedDataViewField)(source) && isFieldDropType(dropType)) {
    return onFieldDrop({
      ...props,
      target: {
        ...target,
        dataView: indexPatterns[state.layers[target.layerId].indexPatternId]
      },
      source,
      indexPatterns
    }, dropType === 'field_combine');
  }
  if (!(0, _types.isOperation)(source)) {
    return false;
  }
  const sourceDataView = indexPatterns[state.layers[source.layerId].indexPatternId];
  const targetDataView = indexPatterns[state.layers[target.layerId].indexPatternId];
  if (sourceDataView !== targetDataView) {
    return false;
  }
  const operationProps = {
    ...props,
    target: {
      ...target,
      dataView: targetDataView
    },
    source: {
      ...source,
      dataView: sourceDataView
    },
    indexPatterns
  };
  if (dropType === 'reorder') {
    return onReorder(operationProps);
  }
  if (['duplicate_compatible', 'replace_duplicate_compatible', 'move_compatible', 'replace_compatible'].includes(dropType)) {
    return onMoveCompatible(operationProps);
  }
  if (['duplicate_incompatible', 'replace_duplicate_incompatible', 'move_incompatible', 'replace_incompatible'].includes(dropType)) {
    return onMoveIncompatible(operationProps);
  }
  if (dropType === 'swap_compatible') {
    return onSwapCompatible(operationProps);
  }
  if (dropType === 'swap_incompatible') {
    return onSwapIncompatible(operationProps);
  }
  if (['combine_incompatible', 'combine_compatible'].includes(dropType)) {
    return onCombine(operationProps);
  }
}
const isFieldDropType = dropType => ['field_add', 'field_replace', 'field_combine'].includes(dropType);
function onFieldDrop(props, shouldAddField) {
  var _targetLayerDimension, _getOperationHelperFo;
  const {
    setState,
    state,
    source,
    target,
    targetLayerDimensionGroups,
    indexPatterns
  } = props;
  const prioritizedOperation = (_targetLayerDimension = targetLayerDimensionGroups.find(g => g.groupId === target.groupId)) === null || _targetLayerDimension === void 0 ? void 0 : _targetLayerDimension.prioritizedOperation;
  const layer = state.layers[target.layerId];
  const indexPattern = indexPatterns[layer.indexPatternId];
  const targetColumn = layer.columns[target.columnId];
  // discourage already used operations for a field
  const alreadyUsedOperations = new Set(Object.values(layer.columns).filter(column => 'sourceField' in column && column.sourceField === source.field.name).map(column => column.operationType));
  const newOperation = shouldAddField ? targetColumn.operationType : (0, _get_drop_props.getNewOperation)(source.field, target.filterOperations, targetColumn, prioritizedOperation, alreadyUsedOperations);
  if (!(0, _utils.isDraggedDataViewField)(source) || !newOperation || shouldAddField && !(0, _operations.hasOperationSupportForMultipleFields)(indexPattern, targetColumn, undefined, source.field)) {
    return false;
  }
  const field = shouldAddField ? (0, _get_drop_props.getField)(targetColumn, indexPattern) : source.field;
  const initialParams = shouldAddField ? {
    params: ((_getOperationHelperFo = (0, _operations.getOperationHelperForMultipleFields)(targetColumn.operationType)) === null || _getOperationHelperFo === void 0 ? void 0 : _getOperationHelperFo({
      targetColumn,
      field: source.field,
      indexPattern
    })) || {}
  } : undefined;
  const newLayer = (0, _operations.insertOrReplaceColumn)({
    layer,
    columnId: target.columnId,
    indexPattern,
    op: newOperation,
    field,
    visualizationGroups: targetLayerDimensionGroups,
    targetGroup: target.groupId,
    shouldCombineField: shouldAddField,
    initialParams
  });
  setState((0, _state_helpers.mergeLayer)({
    state,
    layerId: target.layerId,
    newLayer
  }));
  return true;
}
function onMoveCompatible({
  setState,
  state,
  source,
  target,
  targetLayerDimensionGroups
}) {
  let modifiedLayers = (0, _operations.copyColumn)({
    layers: state.layers,
    target,
    source
  });
  const updatedColumnOrder = (0, _operations.reorderByGroups)(targetLayerDimensionGroups, (0, _operations.getColumnOrder)(modifiedLayers[target.layerId]), target.groupId, target.columnId);
  modifiedLayers = {
    ...modifiedLayers,
    [target.layerId]: {
      ...modifiedLayers[target.layerId],
      columnOrder: updatedColumnOrder,
      columns: modifiedLayers[target.layerId].columns
    }
  };
  setState((0, _state_helpers.mergeLayers)({
    state,
    newLayers: modifiedLayers
  }));
  return true;
}
function onReorder({
  setState,
  state,
  source,
  target
}) {
  function reorderElements(items, targetId, sourceId) {
    const result = items.filter(c => c !== sourceId);
    const targetIndex = items.findIndex(c => c === sourceId);
    const sourceIndex = items.findIndex(c => c === targetId);
    const targetPosition = result.indexOf(targetId);
    result.splice(targetIndex < sourceIndex ? targetPosition + 1 : targetPosition, 0, sourceId);
    return result;
  }
  setState((0, _state_helpers.mergeLayer)({
    state,
    layerId: target.layerId,
    newLayer: {
      columnOrder: reorderElements(state.layers[target.layerId].columnOrder, target.columnId, source.columnId)
    }
  }));
  return true;
}
function onMoveIncompatible({
  setState,
  state,
  source,
  targetLayerDimensionGroups,
  target,
  indexPatterns
}) {
  const targetLayer = state.layers[target.layerId];
  const targetColumn = targetLayer.columns[target.columnId] || null;
  const sourceLayer = state.layers[source.layerId];
  const indexPattern = indexPatterns[sourceLayer.indexPatternId];
  const sourceColumn = sourceLayer.columns[source.columnId];
  const sourceField = (0, _get_drop_props.getField)(sourceColumn, indexPattern);
  const newOperation = (0, _get_drop_props.getNewOperation)(sourceField, target.filterOperations, targetColumn);
  if (!newOperation) {
    return false;
  }
  if (target.layerId === source.layerId) {
    const newLayer = (0, _operations.insertOrReplaceColumn)({
      layer: sourceLayer,
      columnId: target.columnId,
      indexPattern,
      op: newOperation,
      field: sourceField,
      visualizationGroups: targetLayerDimensionGroups,
      targetGroup: target.groupId,
      shouldResetLabel: true
    });
    setState((0, _state_helpers.mergeLayer)({
      state,
      layerId: target.layerId,
      newLayer
    }));
    return true;
  } else {
    const outputTargetLayer = (0, _operations.insertOrReplaceColumn)({
      layer: targetLayer,
      columnId: target.columnId,
      indexPattern,
      op: newOperation,
      field: sourceField,
      visualizationGroups: targetLayerDimensionGroups,
      targetGroup: target.groupId,
      shouldResetLabel: true
    });
    setState((0, _state_helpers.mergeLayers)({
      state,
      newLayers: {
        [source.layerId]: sourceLayer,
        [target.layerId]: outputTargetLayer
      }
    }));
    return true;
  }
}
function onSwapIncompatible({
  setState,
  state,
  source,
  targetLayerDimensionGroups,
  target,
  indexPatterns
}) {
  const targetLayer = state.layers[target.layerId];
  const sourceLayer = state.layers[source.layerId];
  const indexPattern = indexPatterns[targetLayer.indexPatternId];
  const sourceColumn = sourceLayer.columns[source.columnId];
  const targetColumn = targetLayer.columns[target.columnId];
  const sourceField = (0, _get_drop_props.getField)(sourceColumn, indexPattern);
  const targetField = (0, _get_drop_props.getField)(targetColumn, indexPattern);
  const newOperationForSource = (0, _get_drop_props.getNewOperation)(sourceField, target.filterOperations, targetColumn);
  const newOperationForTarget = (0, _get_drop_props.getNewOperation)(targetField, source.filterOperations, sourceColumn);
  if (!newOperationForSource || !newOperationForTarget) {
    return false;
  }
  const outputTargetLayer = (0, _operations.insertOrReplaceColumn)({
    layer: targetLayer,
    columnId: target.columnId,
    targetGroup: target.groupId,
    indexPattern,
    op: newOperationForSource,
    field: sourceField,
    visualizationGroups: targetLayerDimensionGroups,
    shouldResetLabel: true
  });
  if (source.layerId === target.layerId) {
    const newLayer = (0, _operations.insertOrReplaceColumn)({
      layer: outputTargetLayer,
      columnId: source.columnId,
      indexPattern,
      op: newOperationForTarget,
      field: targetField,
      visualizationGroups: targetLayerDimensionGroups,
      targetGroup: source.groupId,
      shouldResetLabel: true
    });
    setState((0, _state_helpers.mergeLayer)({
      state,
      layerId: target.layerId,
      newLayer
    }));
    return true;
  } else {
    const outputSourceLayer = (0, _operations.insertOrReplaceColumn)({
      layer: sourceLayer,
      columnId: source.columnId,
      indexPattern,
      op: newOperationForTarget,
      field: targetField,
      visualizationGroups: targetLayerDimensionGroups,
      targetGroup: source.groupId,
      shouldResetLabel: true
    });
    setState((0, _state_helpers.mergeLayers)({
      state,
      newLayers: {
        [source.layerId]: outputSourceLayer,
        [target.layerId]: outputTargetLayer
      }
    }));
    return true;
  }
}
const swapColumnOrder = (columnOrder, sourceId, targetId) => {
  const sourceIndex = columnOrder.findIndex(c => c === sourceId);
  const targetIndex = columnOrder.findIndex(c => c === targetId);
  const newColumnOrder = [...columnOrder];
  newColumnOrder[sourceIndex] = targetId;
  newColumnOrder[targetIndex] = sourceId;
  return newColumnOrder;
};
function onSwapCompatible({
  setState,
  state,
  source,
  targetLayerDimensionGroups,
  target
}) {
  if (target.layerId === source.layerId) {
    const layer = state.layers[target.layerId];
    const newColumns = {
      ...layer.columns,
      [target.columnId]: {
        ...layer.columns[source.columnId]
      },
      [source.columnId]: {
        ...layer.columns[target.columnId]
      }
    };
    let updatedColumnOrder = swapColumnOrder(layer.columnOrder, source.columnId, target.columnId);
    updatedColumnOrder = (0, _operations.reorderByGroups)(targetLayerDimensionGroups, updatedColumnOrder, target.groupId, target.columnId);
    setState((0, _state_helpers.mergeLayer)({
      state,
      layerId: target.layerId,
      newLayer: {
        columnOrder: updatedColumnOrder,
        columns: newColumns
      }
    }));
    return true;
  } else {
    // TODO why not reorderByGroups for both columns? Are they already in that order?
    const newTargetLayer = (0, _operations.copyColumn)({
      layers: state.layers,
      target,
      source,
      shouldDeleteSource: true
    })[target.layerId];
    const newSourceLayer = (0, _operations.copyColumn)({
      layers: state.layers,
      target: source,
      source: target,
      shouldDeleteSource: true
    })[source.layerId];
    setState((0, _state_helpers.mergeLayers)({
      state,
      newLayers: {
        [source.layerId]: newSourceLayer,
        [target.layerId]: newTargetLayer
      }
    }));
    return true;
  }
}
function onCombine({
  state,
  setState,
  source,
  target,
  targetLayerDimensionGroups,
  indexPatterns
}) {
  var _getOperationHelperFo2, _getOperationHelperFo3;
  const targetLayer = state.layers[target.layerId];
  const targetColumn = targetLayer.columns[target.columnId];
  const targetField = (0, _get_drop_props.getField)(targetColumn, target.dataView);
  const indexPattern = indexPatterns[targetLayer.indexPatternId];
  const sourceLayer = state.layers[source.layerId];
  const sourceColumn = sourceLayer.columns[source.columnId];
  const sourceField = (0, _get_drop_props.getField)(sourceColumn, indexPattern);
  // extract the field from the source column
  if (!sourceField || !targetField) {
    return false;
  }
  // pass it to the target column and delete the source column
  const initialParams = {
    params: (_getOperationHelperFo2 = (_getOperationHelperFo3 = (0, _operations.getOperationHelperForMultipleFields)(targetColumn.operationType)) === null || _getOperationHelperFo3 === void 0 ? void 0 : _getOperationHelperFo3({
      targetColumn,
      sourceColumn,
      indexPattern
    })) !== null && _getOperationHelperFo2 !== void 0 ? _getOperationHelperFo2 : {}
  };
  const outputTargetLayer = (0, _operations.replaceColumn)({
    layer: targetLayer,
    columnId: target.columnId,
    indexPattern,
    op: targetColumn.operationType,
    field: targetField,
    visualizationGroups: targetLayerDimensionGroups,
    targetGroup: target.groupId,
    initialParams,
    shouldCombineField: true
  });
  setState((0, _state_helpers.mergeLayers)({
    state,
    newLayers: {
      ...state.layers,
      [target.layerId]: outputTargetLayer
    }
  }));
  return true;
}