"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createKbnUrlStateStorage = void 0;
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _lodash = require("lodash");
var _url = require("../../state_management/url");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Creates {@link IKbnUrlStateStorage} state storage
 * @returns - {@link IKbnUrlStateStorage}
 * @public
 */
const createKbnUrlStateStorage = ({
  useHash = false,
  useHashQuery = true,
  history,
  onGetError,
  onSetError
} = {
  useHash: false,
  useHashQuery: true
}) => {
  const url = (0, _url.createKbnUrlControls)(history);
  const onGetErrorThrottled = onGetError ? (0, _lodash.throttle)(e => onGetError(e), 100) : undefined;
  return {
    set: (key, state, {
      replace = false
    } = {
      replace: false
    }) => {
      // syncState() utils doesn't wait for this promise
      return url.updateAsync(currentUrl => {
        try {
          return (0, _url.setStateToKbnUrl)(key, state, {
            useHash,
            storeInHashQuery: useHashQuery
          }, currentUrl);
        } catch (error) {
          if (onSetError) onSetError(error);
        }
      }, replace);
    },
    get: key => {
      // if there is a pending url update, then state will be extracted from that pending url,
      // otherwise current url will be used to retrieve state from
      try {
        return (0, _url.getStateFromKbnUrl)(key, url.getPendingUrl(), {
          getFromHashQuery: useHashQuery
        });
      } catch (e) {
        if (onGetErrorThrottled) onGetErrorThrottled(e);
        return null;
      }
    },
    change$: key => new _rxjs.Observable(observer => {
      const unlisten = url.listen(() => {
        observer.next();
      });
      return () => {
        unlisten();
      };
    }).pipe((0, _operators.map)(() => (0, _url.getStateFromKbnUrl)(key, undefined, {
      getFromHashQuery: useHashQuery
    })), (0, _operators.catchError)(error => {
      if (onGetErrorThrottled) onGetErrorThrottled(error);
      return (0, _rxjs.of)(null);
    }), (0, _operators.share)()),
    cancel() {
      url.cancel();
    },
    kbnUrlControls: url
  };
};
exports.createKbnUrlStateStorage = createKbnUrlStateStorage;