"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useModuleStatus = void 0;
var _react = require("react");
var _infra_ml = require("../../../common/infra_ml");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createInitialState = ({
  jobTypes
}) => ({
  jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
    ...accumulatedJobStatus,
    [jobType]: 'unknown'
  }), {}),
  jobSummaries: [],
  lastSetupErrorMessages: [],
  setupStatus: {
    type: 'initializing'
  }
});
const createStatusReducer = jobTypes => (state, action) => {
  switch (action.type) {
    case 'startedSetup':
      {
        return {
          ...state,
          jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
            ...accumulatedJobStatus,
            [jobType]: 'initializing'
          }), {}),
          setupStatus: {
            type: 'pending'
          }
        };
      }
    case 'finishedSetup':
      {
        const {
          datafeedSetupResults,
          jobSetupResults,
          jobSummaries,
          spaceId,
          sourceId
        } = action;
        const nextJobStatus = jobTypes.reduce((accumulatedJobStatus, jobType) => ({
          ...accumulatedJobStatus,
          [jobType]: hasSuccessfullyCreatedJob((0, _infra_ml.getJobId)(spaceId, sourceId, jobType))(jobSetupResults) && hasSuccessfullyStartedDatafeed((0, _infra_ml.getDatafeedId)(spaceId, sourceId, jobType))(datafeedSetupResults) ? 'started' : 'failed'
        }), {});
        const nextSetupStatus = Object.values(nextJobStatus).every(jobState => jobState === 'started') ? {
          type: 'succeeded'
        } : {
          type: 'failed',
          reasons: [...Object.values(datafeedSetupResults).filter(hasError).map(datafeed => datafeed.error.msg), ...Object.values(jobSetupResults).filter(hasError).map(job => job.error.msg)]
        };
        return {
          ...state,
          jobStatus: nextJobStatus,
          jobSummaries,
          setupStatus: nextSetupStatus
        };
      }
    case 'failedSetup':
      {
        return {
          ...state,
          jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
            ...accumulatedJobStatus,
            [jobType]: 'failed'
          }), {}),
          setupStatus: {
            type: 'failed',
            reasons: ['unknown']
          }
        };
      }
    case 'fetchingJobStatuses':
      {
        return {
          ...state,
          setupStatus: state.setupStatus.type === 'unknown' ? {
            type: 'initializing'
          } : state.setupStatus
        };
      }
    case 'fetchedJobStatuses':
      {
        const {
          payload: jobSummaries,
          spaceId,
          sourceId
        } = action;
        const {
          setupStatus
        } = state;
        const nextJobStatus = jobTypes.reduce((accumulatedJobStatus, jobType) => ({
          ...accumulatedJobStatus,
          [jobType]: getJobStatus((0, _infra_ml.getJobId)(spaceId, sourceId, jobType))(jobSummaries)
        }), {});
        const nextSetupStatus = getSetupStatus(nextJobStatus)(setupStatus);
        return {
          ...state,
          jobSummaries,
          jobStatus: nextJobStatus,
          setupStatus: nextSetupStatus
        };
      }
    case 'failedFetchingJobStatuses':
      {
        return {
          ...state,
          setupStatus: {
            type: 'unknown'
          },
          jobStatus: jobTypes.reduce((accumulatedJobStatus, jobType) => ({
            ...accumulatedJobStatus,
            [jobType]: 'unknown'
          }), {})
        };
      }
    case 'viewedResults':
      {
        return {
          ...state,
          setupStatus: {
            type: 'skipped',
            newlyCreated: true
          }
        };
      }
    default:
      {
        return state;
      }
  }
};
const hasSuccessfullyCreatedJob = jobId => jobSetupResponses => jobSetupResponses.filter(jobSetupResponse => jobSetupResponse.id === jobId && jobSetupResponse.success && !jobSetupResponse.error).length > 0;
const hasSuccessfullyStartedDatafeed = datafeedId => datafeedSetupResponses => datafeedSetupResponses.filter(datafeedSetupResponse => datafeedSetupResponse.id === datafeedId && datafeedSetupResponse.success && datafeedSetupResponse.started && !datafeedSetupResponse.error).length > 0;
const getJobStatus = jobId => jobSummaries => {
  return jobSummaries.filter(jobSummary => jobSummary.id === jobId).map(jobSummary => {
    if (jobSummary.jobState === 'failed' || jobSummary.datafeedState === '') {
      return 'failed';
    } else if (jobSummary.jobState === 'closed' && jobSummary.datafeedState === 'stopped' && jobSummary.fullJob && jobSummary.fullJob.finished_time != null) {
      return 'finished';
    } else if (jobSummary.jobState === 'closed' || jobSummary.jobState === 'closing' || jobSummary.datafeedState === 'stopped') {
      return 'stopped';
    } else if (jobSummary.jobState === 'opening' && jobSummary.awaitingNodeAssignment === false || jobSummary.jobState === 'resetting' || jobSummary.jobState === 'reverting') {
      return 'initializing';
    } else if (jobSummary.jobState === 'opened' && jobSummary.datafeedState === 'started' || jobSummary.jobState === 'opening' && jobSummary.datafeedState === 'starting' && jobSummary.awaitingNodeAssignment === true) {
      return 'started';
    }
    return 'unknown';
  })[0] || 'missing';
};
const getSetupStatus = everyJobStatus => previousSetupStatus => {
  return Object.entries(everyJobStatus).reduce((setupStatus, [, jobStatus]) => {
    if (jobStatus === 'missing') {
      return {
        type: 'required'
      };
    } else if (setupStatus.type === 'required' || setupStatus.type === 'succeeded') {
      return setupStatus;
    } else if (setupStatus.type === 'skipped' || (0, _infra_ml.isJobStatusWithResults)(jobStatus)) {
      return {
        type: 'skipped',
        // preserve newlyCreated status
        newlyCreated: setupStatus.type === 'skipped' && setupStatus.newlyCreated
      };
    }
    return setupStatus;
  }, previousSetupStatus);
};
const hasError = value => value.error != null;
const useModuleStatus = jobTypes => {
  return (0, _react.useReducer)(createStatusReducer(jobTypes), {
    jobTypes
  }, createInitialState);
};
exports.useModuleStatus = useModuleStatus;