"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useInfrastructureNodeMetrics = void 0;
var _datemath = require("@kbn/datemath");
var _react = require("react");
var _public = require("@kbn/kibana-react-plugin/public");
var _metrics_source = require("../../../../containers/metrics_source");
var _use_tracked_promise = require("../../../../utils/use_tracked_promise");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const NODE_COUNT_LIMIT = 10000;
const TOTAL_NODES_LIMIT = 100;
const TABLE_PAGE_SIZE = 10;
const nullData = {
  series: [],
  pageInfo: {
    afterKey: null,
    total: -1
  }
};
const useInfrastructureNodeMetrics = options => {
  const {
    metricsExplorerOptions,
    timerange,
    transform,
    sortState,
    currentPageIndex
  } = options;
  const [transformedNodes, setTransformedNodes] = (0, _react.useState)([]);
  const fetch = useKibanaHttpFetch();
  const {
    source,
    isLoadingSource,
    loadSourceRequest,
    metricIndicesExist
  } = (0, _metrics_source.useSourceContext)();
  const timerangeWithInterval = useTimerangeWithInterval(timerange);
  const [fetchNodesRequest, fetchNodes] = (0, _use_tracked_promise.useTrackedPromise)({
    createPromise: () => {
      if (!source) {
        return Promise.resolve(nullData);
      }
      const request = {
        metrics: metricsExplorerOptions.metrics,
        groupBy: metricsExplorerOptions.groupBy,
        limit: NODE_COUNT_LIMIT,
        indexPattern: source.configuration.metricAlias,
        filterQuery: metricsExplorerOptions.filterQuery,
        timerange: timerangeWithInterval
      };
      return fetch('/api/infra/metrics_explorer', {
        method: 'POST',
        body: JSON.stringify(request)
      });
    },
    onResolve: response => {
      setTransformedNodes(response.series.map(transform));
    },
    cancelPreviousOn: 'creation'
  }, [source, metricsExplorerOptions, timerangeWithInterval]);
  const isLoadingNodes = fetchNodesRequest.state === 'pending' || fetchNodesRequest.state === 'uninitialized';
  const isLoading = isLoadingSource || isLoadingNodes;
  const errors = (0, _react.useMemo)(() => [...(loadSourceRequest.state === 'rejected' ? [wrapAsError(loadSourceRequest.value)] : []), ...(fetchNodesRequest.state === 'rejected' ? [wrapAsError(fetchNodesRequest.value)] : [])], [fetchNodesRequest, loadSourceRequest]);
  (0, _react.useEffect)(() => {
    fetchNodes();
  }, [fetchNodes]);
  const sortedNodes = (0, _react.useMemo)(() => {
    return [...transformedNodes].sort(makeSortNodes(sortState));
  }, [transformedNodes, sortState]);
  const top100Nodes = (0, _react.useMemo)(() => {
    return sortedNodes.slice(0, TOTAL_NODES_LIMIT);
  }, [sortedNodes]);
  const nodes = (0, _react.useMemo)(() => {
    const pageStartIndex = currentPageIndex * TABLE_PAGE_SIZE;
    const pageEndIndex = pageStartIndex + TABLE_PAGE_SIZE;
    return top100Nodes.slice(pageStartIndex, pageEndIndex);
  }, [top100Nodes, currentPageIndex]);
  const pageCount = (0, _react.useMemo)(() => Math.ceil(top100Nodes.length / TABLE_PAGE_SIZE), [top100Nodes]);
  const data = (0, _react.useMemo)(() => errors.length > 0 ? {
    state: 'error',
    errors
  } : metricIndicesExist == null ? {
    state: 'unknown'
  } : !metricIndicesExist ? {
    state: 'no-indices'
  } : nodes.length <= 0 ? {
    state: 'empty-indices'
  } : {
    state: 'data',
    currentPageIndex,
    pageCount,
    rows: nodes
  }, [currentPageIndex, errors, metricIndicesExist, nodes, pageCount]);
  return {
    isLoading,
    data
  };
};
exports.useInfrastructureNodeMetrics = useInfrastructureNodeMetrics;
function useKibanaHttpFetch() {
  var _kibana$services$http;
  const kibana = (0, _public.useKibana)();
  const fetch = (_kibana$services$http = kibana.services.http) === null || _kibana$services$http === void 0 ? void 0 : _kibana$services$http.fetch;
  if (!fetch) {
    throw new Error('Could not find Kibana HTTP fetch');
  }
  return fetch;
}
function useTimerangeWithInterval(timerange) {
  return (0, _react.useMemo)(() => {
    const from = (0, _datemath.parse)(timerange.from);
    const to = (0, _datemath.parse)(timerange.to);
    if (!from || !to) {
      throw new Error('Could not parse timerange');
    }
    return {
      from: from.valueOf(),
      to: to.valueOf(),
      interval: 'modules'
    };
  }, [timerange]);
}
function makeSortNodes(sortState) {
  return (nodeA, nodeB) => {
    const nodeAValue = nodeA[sortState.field];
    const nodeBValue = nodeB[sortState.field];
    if (sortState.direction === 'asc') {
      return sortAscending(nodeAValue, nodeBValue);
    }
    return sortDescending(nodeAValue, nodeBValue);
  };
}
function sortAscending(nodeAValue, nodeBValue) {
  if (nodeAValue === null) {
    return -1;
  } else if (nodeBValue === null) {
    return 1;
  }
  if (typeof nodeAValue === 'string' && typeof nodeBValue === 'string') {
    return nodeAValue.localeCompare(nodeBValue);
  }
  if (typeof nodeAValue === 'number' && typeof nodeBValue === 'number') {
    return nodeAValue - nodeBValue;
  }
  return 0;
}
function sortDescending(nodeAValue, nodeBValue) {
  if (nodeAValue === null) {
    return 1;
  } else if (nodeBValue === null) {
    return -1;
  }
  if (typeof nodeAValue === 'string' && typeof nodeBValue === 'string') {
    return nodeBValue.localeCompare(nodeAValue);
  }
  if (typeof nodeAValue === 'number' && typeof nodeBValue === 'number') {
    return nodeBValue - nodeAValue;
  }
  return 0;
}
const wrapAsError = value => value instanceof Error ? value : new Error(`${value}`);