"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TemplateForm = void 0;
var _react = _interopRequireWildcard(require("react"));
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _eui = require("@elastic/eui");
var _shared_imports = require("../../../shared_imports");
var _shared = require("../shared");
var _documentation = require("../../services/documentation");
var _section_error = require("../section_error");
var _index_templates = require("../index_templates");
var _steps = require("./steps");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const {
  stripEmptyFields
} = _shared_imports.serializers;
const {
  FormWizard,
  FormWizardStep
} = _shared_imports.Forms;
const {
  useGlobalFlyout
} = _shared_imports.GlobalFlyout;
const wizardSections = {
  logistics: {
    id: 'logistics',
    label: _i18n.i18n.translate('xpack.idxMgmt.templateForm.steps.logisticsStepName', {
      defaultMessage: 'Logistics'
    })
  },
  components: {
    id: 'components',
    label: _i18n.i18n.translate('xpack.idxMgmt.templateForm.steps.componentsStepName', {
      defaultMessage: 'Component templates'
    })
  },
  settings: {
    id: 'settings',
    label: _i18n.i18n.translate('xpack.idxMgmt.templateForm.steps.settingsStepName', {
      defaultMessage: 'Index settings'
    })
  },
  mappings: {
    id: 'mappings',
    label: _i18n.i18n.translate('xpack.idxMgmt.templateForm.steps.mappingsStepName', {
      defaultMessage: 'Mappings'
    })
  },
  aliases: {
    id: 'aliases',
    label: _i18n.i18n.translate('xpack.idxMgmt.templateForm.steps.aliasesStepName', {
      defaultMessage: 'Aliases'
    })
  },
  review: {
    id: 'review',
    label: _i18n.i18n.translate('xpack.idxMgmt.templateForm.steps.summaryStepName', {
      defaultMessage: 'Review template'
    })
  }
};
const TemplateForm = ({
  title,
  defaultValue,
  isEditing,
  isSaving,
  isLegacy = false,
  saveError,
  clearSaveError,
  onSave,
  history
}) => {
  const [wizardContent, setWizardContent] = (0, _react.useState)(null);
  const {
    addContent: addContentToGlobalFlyout,
    closeFlyout
  } = useGlobalFlyout();
  const simulateTemplateFilters = (0, _react.useRef)({
    mappings: true,
    settings: true,
    aliases: true
  });

  // eslint-disable-next-line react-hooks/exhaustive-deps
  const indexTemplate = defaultValue !== null && defaultValue !== void 0 ? defaultValue : {
    name: '',
    indexPatterns: [],
    template: {},
    _kbnMeta: {
      type: 'default',
      hasDatastream: false,
      isLegacy
    }
  };
  const {
    template: {
      settings,
      mappings,
      aliases
    } = {},
    composedOf,
    _kbnMeta,
    ...logistics
  } = indexTemplate;
  const wizardDefaultValue = {
    logistics,
    settings,
    mappings,
    aliases,
    components: indexTemplate.composedOf
  };
  const i18nTexts = {
    save: isEditing ? /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.idxMgmt.templateForm.saveButtonLabel",
      defaultMessage: "Save template"
    }) : /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.idxMgmt.templateForm.createButtonLabel",
      defaultMessage: "Create template"
    })
  };
  const apiError = saveError ? /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_section_error.SectionError, {
    title: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.idxMgmt.templateForm.saveTemplateError",
      defaultMessage: "Unable to create template"
    }),
    error: saveError,
    "data-test-subj": "saveTemplateError"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "m"
  })) : null;

  /**
   * If no mappings, settings or aliases are defined, it is better to not send empty
   * object for those values.
   * This method takes care of that and other cleanup of empty fields.
   * @param template The template object to clean up
   */
  const cleanupTemplateObject = template => {
    const outputTemplate = {
      ...template
    };
    if (outputTemplate.template) {
      if (outputTemplate.template.settings === undefined) {
        delete outputTemplate.template.settings;
      }
      if (outputTemplate.template.mappings === undefined) {
        delete outputTemplate.template.mappings;
      }
      if (outputTemplate.template.aliases === undefined) {
        delete outputTemplate.template.aliases;
      }
      if (Object.keys(outputTemplate.template).length === 0) {
        delete outputTemplate.template;
      }
    }
    return outputTemplate;
  };
  const buildTemplateObject = (0, _react.useCallback)(initialTemplate => wizardData => {
    const outputTemplate = {
      ...wizardData.logistics,
      _kbnMeta: initialTemplate._kbnMeta,
      composedOf: wizardData.components,
      template: {
        settings: wizardData.settings,
        mappings: wizardData.mappings,
        aliases: wizardData.aliases
      }
    };
    return cleanupTemplateObject(outputTemplate);
  }, []);
  const onWizardContentChange = (0, _react.useCallback)(content => {
    setWizardContent(content);
  }, []);
  const onSaveTemplate = (0, _react.useCallback)(async wizardData => {
    const template = buildTemplateObject(indexTemplate)(wizardData);

    // We need to strip empty string, otherwise if the "order" or "version"
    // are not set, they will be empty string and ES expect a number for those parameters.
    onSave(stripEmptyFields(template, {
      types: ['string']
    }));
    clearSaveError();
  }, [indexTemplate, buildTemplateObject, onSave, clearSaveError]);
  const getSimulateTemplate = (0, _react.useCallback)(async () => {
    if (!wizardContent) {
      return;
    }
    const isValid = await wizardContent.validate();
    if (!isValid) {
      return;
    }
    const wizardData = wizardContent.getData();
    const template = buildTemplateObject(indexTemplate)(wizardData);
    return template;
  }, [buildTemplateObject, indexTemplate, wizardContent]);
  const onSimulateTemplateFiltersChange = (0, _react.useCallback)(filters => {
    simulateTemplateFilters.current = filters;
  }, []);
  const showPreviewFlyout = () => {
    addContentToGlobalFlyout({
      id: 'simulateTemplate',
      Component: _index_templates.SimulateTemplateFlyoutContent,
      props: {
        getTemplate: getSimulateTemplate,
        onClose: closeFlyout,
        filters: simulateTemplateFilters.current,
        onFiltersChange: onSimulateTemplateFiltersChange
      },
      flyoutProps: _index_templates.simulateTemplateFlyoutProps
    });
  };
  const getRightContentWizardNav = stepId => {
    if (isLegacy) {
      return null;
    }

    // Don't show "Preview template" button on logistics and review steps
    if (stepId === 'logistics' || stepId === 'review') {
      return null;
    }
    return /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
      size: "s",
      onClick: showPreviewFlyout,
      "data-test-subj": "previewIndexTemplate"
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.idxMgmt.templateForm.previewIndexTemplateButtonLabel",
      defaultMessage: "Preview index template"
    }));
  };
  const isLegacyIndexTemplate = indexTemplate._kbnMeta.isLegacy === true;
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiPageHeader, {
    pageTitle: /*#__PURE__*/_react.default.createElement("span", {
      "data-test-subj": "pageTitle"
    }, title),
    bottomBorder: true
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "m"
  }), isLegacyIndexTemplate && /*#__PURE__*/_react.default.createElement(_index_templates.LegacyIndexTemplatesDeprecation, {
    history: history,
    showCta: true
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s"
  }), /*#__PURE__*/_react.default.createElement(FormWizard, {
    defaultValue: wizardDefaultValue,
    onSave: onSaveTemplate,
    isEditing: isEditing,
    isSaving: isSaving,
    apiError: apiError,
    texts: i18nTexts,
    onChange: onWizardContentChange,
    rightContentNav: getRightContentWizardNav
  }, /*#__PURE__*/_react.default.createElement(FormWizardStep, {
    id: wizardSections.logistics.id,
    label: wizardSections.logistics.label,
    isRequired: true
  }, /*#__PURE__*/_react.default.createElement(_steps.StepLogisticsContainer, {
    isEditing: isEditing,
    isLegacy: indexTemplate._kbnMeta.isLegacy
  })), !isLegacyIndexTemplate && /*#__PURE__*/_react.default.createElement(FormWizardStep, {
    id: wizardSections.components.id,
    label: wizardSections.components.label
  }, /*#__PURE__*/_react.default.createElement(_steps.StepComponentContainer, null)), /*#__PURE__*/_react.default.createElement(FormWizardStep, {
    id: wizardSections.settings.id,
    label: wizardSections.settings.label
  }, /*#__PURE__*/_react.default.createElement(_shared.StepSettingsContainer, {
    esDocsBase: _documentation.documentationService.getEsDocsBase()
  })), /*#__PURE__*/_react.default.createElement(FormWizardStep, {
    id: wizardSections.mappings.id,
    label: wizardSections.mappings.label
  }, /*#__PURE__*/_react.default.createElement(_shared.StepMappingsContainer, {
    esDocsBase: _documentation.documentationService.getEsDocsBase()
  })), /*#__PURE__*/_react.default.createElement(FormWizardStep, {
    id: wizardSections.aliases.id,
    label: wizardSections.aliases.label
  }, /*#__PURE__*/_react.default.createElement(_shared.StepAliasesContainer, {
    esDocsBase: _documentation.documentationService.getEsDocsBase()
  })), /*#__PURE__*/_react.default.createElement(FormWizardStep, {
    id: wizardSections.review.id,
    label: wizardSections.review.label
  }, /*#__PURE__*/_react.default.createElement(_steps.StepReviewContainer, {
    getTemplateData: buildTemplateObject(indexTemplate)
  }))));
};
exports.TemplateForm = TemplateForm;