"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.savingSaga = exports.saveWorkspace = exports.loadingSaga = exports.loadSavedWorkspace = exports.fillWorkspace = void 0;
var _typescriptFsa = _interopRequireDefault(require("typescript-fsa"));
var _i18n = require("@kbn/i18n");
var _effects = require("redux-saga/effects");
var _ = require(".");
var _datasource = require("./datasource");
var _fields = require("./fields");
var _advanced_settings = require("./advanced_settings");
var _url_templates = require("./url_templates");
var _persistence = require("../services/persistence");
var _meta_data = require("./meta_data");
var _save_modal = require("../services/save_modal");
var _url = require("../services/url");
var _saved_workspace_utils = require("../helpers/saved_workspace_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const actionCreator = (0, _typescriptFsa.default)('x-pack/graph');
const loadSavedWorkspace = actionCreator('LOAD_WORKSPACE');
exports.loadSavedWorkspace = loadSavedWorkspace;
const saveWorkspace = actionCreator('SAVE_WORKSPACE');
exports.saveWorkspace = saveWorkspace;
const fillWorkspace = actionCreator('FILL_WORKSPACE');

/**
 * Saga handling loading of a saved workspace.
 *
 * It will load the index pattern associated with the saved object and deserialize all properties
 * into the store. Existing state will be overwritten.
 */
exports.fillWorkspace = fillWorkspace;
const loadingSaga = ({
  createWorkspace,
  notifications,
  indexPatternProvider
}) => {
  function* deserializeWorkspace(action) {
    const {
      dataViews,
      savedWorkspace,
      urlQuery
    } = action.payload;
    const migrationStatus = (0, _persistence.migrateLegacyIndexPatternRef)(savedWorkspace, dataViews);
    if (!migrationStatus.success) {
      notifications.toasts.addDanger(_i18n.i18n.translate('xpack.graph.loadWorkspace.missingDataViewErrorMessage', {
        defaultMessage: 'Data view "{name}" not found',
        values: {
          name: migrationStatus.missingIndexPattern
        }
      }));
      return;
    }
    const selectedIndexPatternId = (0, _persistence.lookupIndexPatternId)(savedWorkspace);
    let indexPattern;
    try {
      indexPattern = yield (0, _effects.call)(indexPatternProvider.get, selectedIndexPatternId);
    } catch (e) {
      notifications.toasts.addDanger(_i18n.i18n.translate('xpack.graph.loadWorkspace.missingDataViewErrorMessage', {
        defaultMessage: 'Data view "{name}" not found',
        values: {
          name: selectedIndexPatternId
        }
      }));
      return;
    }
    const initialSettings = (0, _advanced_settings.settingsSelector)(yield (0, _effects.select)());
    const createdWorkspace = createWorkspace(indexPattern.title, initialSettings);
    const {
      urlTemplates,
      advancedSettings,
      allFields
    } = (0, _persistence.savedWorkspaceToAppState)(savedWorkspace, indexPattern, createdWorkspace);

    // put everything in the store
    yield (0, _effects.put)((0, _meta_data.updateMetaData)({
      title: savedWorkspace.title,
      description: savedWorkspace.description,
      savedObjectId: savedWorkspace.id
    }));
    yield (0, _effects.put)((0, _datasource.setDatasource)({
      type: 'indexpattern',
      id: indexPattern.id,
      title: indexPattern.title
    }));
    yield (0, _effects.put)((0, _fields.loadFields)(allFields));
    yield (0, _effects.put)((0, _advanced_settings.updateSettings)(advancedSettings));
    yield (0, _effects.put)((0, _url_templates.loadTemplates)(urlTemplates));
    if (urlQuery) {
      yield (0, _effects.put)((0, _.submitSearch)(urlQuery));
    }
    createdWorkspace.runLayout();
  }
  return function* () {
    yield (0, _effects.takeLatest)(loadSavedWorkspace.match, deserializeWorkspace);
  };
};

/**
 * Saga handling saving of current state.
 *
 * It will serialize everything and save it using the saved objects client
 */
exports.loadingSaga = loadingSaga;
const savingSaga = deps => {
  function* persistWorkspace(action) {
    const savedWorkspace = action.payload;
    const state = yield (0, _effects.select)();
    const workspace = deps.getWorkspace();
    const selectedDatasource = (0, _datasource.datasourceSelector)(state).current;
    if (!workspace || selectedDatasource.type === 'none') {
      return;
    }
    const savedObjectId = yield (0, _effects.cps)(showModal, {
      deps,
      workspace,
      savedWorkspace,
      state,
      selectedDatasource
    });
    if (savedObjectId) {
      yield (0, _effects.put)((0, _meta_data.updateMetaData)({
        savedObjectId
      }));
    }
  }
  return function* () {
    yield (0, _effects.takeLatest)(saveWorkspace.match, persistWorkspace);
  };
};
exports.savingSaga = savingSaga;
function showModal({
  deps,
  workspace,
  savedWorkspace,
  state,
  selectedDatasource
}, savingCallback) {
  const saveWorkspaceHandler = async (saveOptions, userHasConfirmedSaveWorkspaceData, services) => {
    const canSaveData = deps.savePolicy === 'configAndData' || deps.savePolicy === 'configAndDataWithConsent' && userHasConfirmedSaveWorkspaceData;
    (0, _persistence.appStateToSavedWorkspace)(savedWorkspace, {
      workspace,
      urlTemplates: (0, _url_templates.templatesSelector)(state),
      advancedSettings: (0, _advanced_settings.settingsSelector)(state),
      selectedIndex: selectedDatasource,
      selectedFields: (0, _fields.selectedFieldsSelector)(state)
    }, canSaveData);
    try {
      const id = await (0, _saved_workspace_utils.saveSavedWorkspace)(savedWorkspace, saveOptions, services);
      if (id) {
        const title = _i18n.i18n.translate('xpack.graph.saveWorkspace.successNotificationTitle', {
          defaultMessage: 'Saved "{workspaceTitle}"',
          values: {
            workspaceTitle: savedWorkspace.title
          }
        });
        let text;
        if (!canSaveData && workspace.nodes.length > 0) {
          text = _i18n.i18n.translate('xpack.graph.saveWorkspace.successNotification.noDataSavedText', {
            defaultMessage: 'The configuration was saved, but the data was not saved'
          });
        }
        deps.notifications.toasts.addSuccess({
          title,
          text,
          'data-test-subj': 'saveGraphSuccess'
        });
        if (savedWorkspace.id !== (0, _meta_data.metaDataSelector)(state).savedObjectId) {
          deps.changeUrl((0, _url.getEditPath)(savedWorkspace));
        }
      }
      savingCallback(null, id);
      return {
        id
      };
    } catch (error) {
      deps.notifications.toasts.addDanger(_i18n.i18n.translate('xpack.graph.saveWorkspace.savingErrorMessage', {
        defaultMessage: 'Failed to save workspace: {message}',
        values: {
          message: error
        }
      }));
      return {
        error
      };
    }
  };
  (0, _save_modal.openSaveModal)({
    savePolicy: deps.savePolicy,
    hasData: workspace.nodes.length > 0 || workspace.blocklistedNodes.length > 0,
    workspace: savedWorkspace,
    saveWorkspace: saveWorkspaceHandler,
    services: {
      savedObjectsClient: deps.savedObjectsClient,
      overlays: deps.overlays
    }
  });
}