"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWorkspaceLoader = void 0;
var _react = require("react");
var _reactRouterDom = require("react-router-dom");
var _i18n = require("@kbn/i18n");
var _saved_workspace_utils = require("./saved_workspace_utils");
var _url = require("../services/url");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useWorkspaceLoader = ({
  coreStart,
  spaces,
  workspaceRef,
  store,
  savedObjectsClient,
  data
}) => {
  const [state, setState] = (0, _react.useState)();
  const {
    replace: historyReplace
  } = (0, _reactRouterDom.useHistory)();
  const {
    search
  } = (0, _reactRouterDom.useLocation)();
  const {
    id
  } = (0, _reactRouterDom.useParams)();

  /**
   * The following effect initializes workspace initially and reacts
   * on changes in id parameter and URL query only.
   */
  (0, _react.useEffect)(() => {
    const urlQuery = new URLSearchParams(search).get('query');
    function loadWorkspace(fetchedSavedWorkspace, dataViews) {
      store.dispatch({
        type: 'x-pack/graph/LOAD_WORKSPACE',
        payload: {
          savedWorkspace: fetchedSavedWorkspace,
          dataViews,
          urlQuery
        }
      });
    }
    function clearStore() {
      store.dispatch({
        type: 'x-pack/graph/RESET'
      });
    }
    async function fetchSavedWorkspace() {
      return id ? await (0, _saved_workspace_utils.getSavedWorkspace)(savedObjectsClient, id).catch(function (e) {
        coreStart.notifications.toasts.addError(e, {
          title: _i18n.i18n.translate('xpack.graph.missingWorkspaceErrorMessage', {
            defaultMessage: "Couldn't load graph with ID"
          })
        });
        historyReplace('/home');
        // return promise that never returns to prevent the controller from loading
        return new Promise(() => {});
      }) : (0, _saved_workspace_utils.getEmptyWorkspace)();
    }
    async function initializeWorkspace() {
      const {
        savedObject: fetchedSavedWorkspace,
        sharingSavedObjectProps: fetchedSharingSavedObjectProps
      } = await fetchSavedWorkspace();
      if (spaces && (fetchedSharingSavedObjectProps === null || fetchedSharingSavedObjectProps === void 0 ? void 0 : fetchedSharingSavedObjectProps.outcome) === 'aliasMatch') {
        // We found this object by a legacy URL alias from its old ID; redirect the user to the page with its new ID, preserving any URL hash
        const newObjectId = fetchedSharingSavedObjectProps.aliasTargetId; // This is always defined if outcome === 'aliasMatch'
        const newPath = (0, _url.getEditUrl)(coreStart.http.basePath.prepend, {
          id: newObjectId
        }) + search;
        spaces.ui.redirectLegacyUrl({
          path: newPath,
          aliasPurpose: fetchedSharingSavedObjectProps.aliasPurpose,
          objectNoun: _i18n.i18n.translate('xpack.graph.legacyUrlConflict.objectNoun', {
            defaultMessage: 'Graph'
          })
        });
        return null;
      }
      const dataViews = await data.dataViews.getIdsWithTitle();

      /**
       * Deal with situation of request to open saved workspace. Otherwise clean up store,
       * when navigating to a new workspace from existing one.
       */
      if (fetchedSavedWorkspace.id) {
        loadWorkspace(fetchedSavedWorkspace, dataViews);
      } else if (workspaceRef.current) {
        clearStore();
      }
      setState({
        savedWorkspace: fetchedSavedWorkspace,
        sharingSavedObjectProps: fetchedSharingSavedObjectProps
      });
    }
    initializeWorkspace();
  }, [id, search, store, historyReplace, savedObjectsClient, setState, coreStart, workspaceRef, spaces, data.dataViews]);
  return state;
};
exports.useWorkspaceLoader = useWorkspaceLoader;