"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGraphLoader = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/inspector-plugin/public");
var _format_http_error = require("./format_http_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useGraphLoader = ({
  toastNotifications,
  coreStart
}) => {
  const [loading, setLoading] = (0, _react.useState)(false);
  const requestAdapter = (0, _react.useMemo)(() => new _public.RequestAdapter(), []);
  const handleHttpError = (0, _react.useCallback)(error => {
    toastNotifications.addDanger((0, _format_http_error.formatHttpError)(error));
  }, [toastNotifications]);
  const handleSearchQueryError = (0, _react.useCallback)(err => {
    const toastTitle = _i18n.i18n.translate('xpack.graph.errorToastTitle', {
      defaultMessage: 'Graph Error',
      description: '"Graph" is a product name and should not be translated.'
    });
    if (err instanceof Error) {
      toastNotifications.addError(err, {
        title: toastTitle
      });
    } else {
      toastNotifications.addDanger({
        title: toastTitle,
        text: String(err)
      });
    }
  }, [toastNotifications]);
  const getRequestInspector = (0, _react.useCallback)(indexName => {
    const inspectRequest = requestAdapter.start(_i18n.i18n.translate('xpack.graph.inspectAdapter.graphExploreRequest.name', {
      defaultMessage: 'Data'
    }), {
      description: _i18n.i18n.translate('xpack.graph.inspectAdapter.graphExploreRequest.description', {
        defaultMessage: 'This request queries Elasticsearch to fetch the data for the Graph.'
      })
    });
    inspectRequest.stats({
      indexPattern: {
        label: _i18n.i18n.translate('xpack.graph.inspectAdapter.graphExploreRequest.dataView.description.label', {
          defaultMessage: 'Data view'
        }),
        value: indexName,
        description: _i18n.i18n.translate('xpack.graph.inspectAdapter.graphExploreRequest.dataView.description', {
          defaultMessage: 'The data view that connected to the Elasticsearch indices.'
        })
      }
    });
    return inspectRequest;
  }, [requestAdapter]);

  // Replacement function for graphClientWorkspace's comms so
  // that it works with Kibana.
  const callNodeProxy = (0, _react.useCallback)((indexName, query, responseHandler) => {
    const dsl = {
      index: indexName,
      query
    };
    const request = {
      body: JSON.stringify(dsl)
    };
    setLoading(true);
    requestAdapter.reset();
    const inspectRequest = getRequestInspector(indexName);
    inspectRequest.json(dsl);
    return coreStart.http.post('../api/graph/graphExplore', request).then(function (data) {
      const response = data.resp;
      if (response !== null && response !== void 0 && response.timed_out) {
        toastNotifications.addWarning(_i18n.i18n.translate('xpack.graph.exploreGraph.timedOutWarningText', {
          defaultMessage: 'Exploration timed out'
        }));
      }
      inspectRequest.stats({}).ok({
        json: response
      });
      responseHandler(response);
    }).catch(e => {
      inspectRequest.error({
        json: e
      });
      handleHttpError(e);
    }).finally(() => setLoading(false));
  }, [coreStart.http, getRequestInspector, handleHttpError, requestAdapter, toastNotifications]);

  // Helper function for the graphClientWorkspace to perform a query
  const callSearchNodeProxy = (0, _react.useCallback)((indexName, query, responseHandler) => {
    const dsl = {
      index: indexName,
      body: query
    };
    const request = {
      body: JSON.stringify(dsl)
    };
    setLoading(true);
    requestAdapter.reset();
    const inspectRequest = getRequestInspector(indexName);
    inspectRequest.json(dsl);
    coreStart.http.post('../api/graph/searchProxy', request).then(function (data) {
      const response = data.resp;
      inspectRequest.stats({}).ok({
        json: response
      });
      responseHandler(response);
    }).catch(e => {
      inspectRequest.error({
        json: e
      });
      handleHttpError(e);
    }).finally(() => setLoading(false));
  }, [coreStart.http, getRequestInspector, handleHttpError, requestAdapter]);
  return {
    loading,
    requestAdapter,
    callNodeProxy,
    callSearchNodeProxy,
    handleSearchQueryError
  };
};
exports.useGraphLoader = useGraphLoader;