"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useComboInput = useComboInput;
exports.useInput = useInput;
exports.useNumberInput = useNumberInput;
exports.useSelectInput = useSelectInput;
exports.useSwitchInput = useSwitchInput;
exports.validateInputs = validateInputs;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateInputs(inputs) {
  return Object.values(inputs).reduce((acc, input) => {
    const res = input.validate();
    return acc === false ? acc : res;
  }, true);
}
function useInput(defaultValue = '', validate, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [errors, setErrors] = (0, _react.useState)();
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  const onChange = (0, _react.useCallback)(e => {
    const newValue = e.target.value;
    setValue(newValue);
    if (errors && validate && validate(newValue) === undefined) {
      setErrors(undefined);
    }
  }, [errors, validate]);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== defaultValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const isInvalid = errors !== undefined;
  return {
    value,
    errors,
    props: {
      onChange,
      value,
      isInvalid,
      disabled
    },
    formRowProps: {
      error: errors,
      isInvalid
    },
    clear: () => {
      setValue('');
    },
    validate: () => {
      if (validate) {
        const newErrors = validate(value);
        setErrors(newErrors);
        return newErrors === undefined;
      }
      return true;
    },
    setValue,
    hasChanged
  };
}
function useSwitchInput(defaultValue = false, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== defaultValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const onChange = e => {
    const newValue = e.target.checked;
    setValue(newValue);
  };
  const validate = (0, _react.useCallback)(() => true, []);
  return {
    value,
    props: {
      onChange,
      checked: value,
      disabled
    },
    validate,
    formRowProps: {},
    setValue,
    hasChanged
  };
}
function useComboInput(id, defaultValue = [], validate, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [errors, setErrors] = (0, _react.useState)();
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value.length !== defaultValue.length || value.some((val, idx) => val !== defaultValue[idx])) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const isInvalid = errors !== undefined;
  const validateCallback = (0, _react.useCallback)(() => {
    if (validate) {
      const newErrors = validate(value);
      setErrors(newErrors);
      return newErrors === undefined;
    }
    return true;
  }, [validate, value]);
  const onChange = (0, _react.useCallback)(newValues => {
    setValue(newValues);
    if (errors && validate) {
      setErrors(validate(newValues));
    }
  }, [validate, errors]);
  return {
    props: {
      id,
      value,
      onChange,
      errors,
      isInvalid,
      disabled
    },
    formRowProps: {
      error: errors,
      isInvalid
    },
    value,
    clear: () => {
      setValue([]);
    },
    setValue,
    validate: validateCallback,
    hasChanged
  };
}
function useNumberInput(defaultValue, validate, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [errors, setErrors] = (0, _react.useState)();
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  const onChange = (0, _react.useCallback)(e => {
    const newValue = e.target.value ? Number(e.target.value) : undefined;
    setValue(newValue);
    if (newValue && errors && validate && validate(newValue) === undefined) {
      setErrors(undefined);
    }
  }, [errors, validate]);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== defaultValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const isInvalid = errors !== undefined;
  return {
    value,
    errors,
    props: {
      onChange,
      value,
      isInvalid,
      disabled
    },
    formRowProps: {
      error: errors,
      isInvalid
    },
    clear: () => {
      setValue(undefined);
    },
    validate: () => {
      if (validate && value) {
        const newErrors = validate(value);
        setErrors(newErrors);
        return newErrors === undefined;
      }
      return true;
    },
    setValue,
    hasChanged
  };
}
function useSelectInput(options, defaultValue = '', disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const onChange = (0, _react.useCallback)(e => {
    setValue(e.target.value);
  }, []);
  return {
    props: {
      options,
      value,
      onChange,
      disabled
    },
    value,
    clear: () => {
      setValue('');
    },
    setValue
  };
}