"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentBulkActions = void 0;
var _react = _interopRequireWildcard(require("react"));
var _eui = require("@elastic/eui");
var _i18nReact = require("@kbn/i18n-react");
var _components = require("../../components");
var _hooks = require("../../../../hooks");
var _constants = require("../../../../../../../common/constants");
var _services = require("../../../../services");
var _utils = require("../utils");
var _agent_request_diagnostics_modal = require("../../components/agent_request_diagnostics_modal");
var _tags_add_remove = require("./tags_add_remove");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AgentBulkActions = ({
  totalAgents,
  totalInactiveAgents,
  selectionMode,
  currentQuery,
  selectedAgents,
  visibleAgents,
  refreshAgents,
  allTags,
  agentPolicies
}) => {
  const licenseService = (0, _hooks.useLicense)();
  const isLicenceAllowingScheduleUpgrade = licenseService.hasAtLeast(_constants.LICENSE_FOR_SCHEDULE_UPGRADE);

  // Bulk actions menu states
  const [isMenuOpen, setIsMenuOpen] = (0, _react.useState)(false);
  const closeMenu = () => setIsMenuOpen(false);
  const onClickMenu = () => setIsMenuOpen(!isMenuOpen);

  // Actions states
  const [isReassignFlyoutOpen, setIsReassignFlyoutOpen] = (0, _react.useState)(false);
  const [isUnenrollModalOpen, setIsUnenrollModalOpen] = (0, _react.useState)(false);
  const [updateModalState, setUpgradeModalState] = (0, _react.useState)({
    isOpen: false,
    isScheduled: false
  });
  const [isTagAddVisible, setIsTagAddVisible] = (0, _react.useState)(false);
  const [isRequestDiagnosticsModalOpen, setIsRequestDiagnosticsModalOpen] = (0, _react.useState)(false);
  const [managedAgents, setManagedAgents] = (0, _react.useState)([]);

  // get all the managed policies
  const fetchManagedAgents = (0, _react.useCallback)(async () => {
    if (selectionMode === 'query') {
      var _agentPoliciesRespons, _agentPoliciesRespons2, _response$data$items, _response$data;
      const managedPoliciesKuery = `${_constants.AGENT_POLICY_SAVED_OBJECT_TYPE}.is_managed:true`;
      const agentPoliciesResponse = await (0, _hooks.sendGetAgentPolicies)({
        kuery: managedPoliciesKuery,
        perPage: _constants.SO_SEARCH_LIMIT,
        full: false
      });
      if (agentPoliciesResponse.error) {
        throw new Error(agentPoliciesResponse.error.message);
      }
      const managedPolicies = (_agentPoliciesRespons = (_agentPoliciesRespons2 = agentPoliciesResponse.data) === null || _agentPoliciesRespons2 === void 0 ? void 0 : _agentPoliciesRespons2.items) !== null && _agentPoliciesRespons !== void 0 ? _agentPoliciesRespons : [];

      // find all the agents that have those policies and are not unenrolled
      const policiesKuery = managedPolicies.map(policy => `policy_id:"${policy.id}"`).join(' or ');
      const kuery = `NOT (status:unenrolled) and ${policiesKuery}`;
      const response = await (0, _hooks.sendGetAgents)({
        kuery,
        perPage: _constants.SO_SEARCH_LIMIT,
        showInactive: true
      });
      if (response.error) {
        throw new Error(response.error.message);
      }
      return (_response$data$items = (_response$data = response.data) === null || _response$data === void 0 ? void 0 : _response$data.items) !== null && _response$data$items !== void 0 ? _response$data$items : [];
    }
    return [];
  }, [selectionMode]);
  (0, _react.useEffect)(() => {
    async function fetchDataAsync() {
      const allManagedAgents = await fetchManagedAgents();
      setManagedAgents(allManagedAgents === null || allManagedAgents === void 0 ? void 0 : allManagedAgents.map(agent => agent.id));
    }
    fetchDataAsync();
  }, [fetchManagedAgents]);

  // update the query removing the "managed" agents
  const selectionQuery = (0, _react.useMemo)(() => {
    if (managedAgents.length) {
      const excludedKuery = `${_constants.AGENTS_PREFIX}.agent.id : (${managedAgents.map(id => `"${id}"`).join(' or ')})`;
      return `${currentQuery} AND NOT (${excludedKuery})`;
    } else {
      return currentQuery;
    }
  }, [currentQuery, managedAgents]);

  // Check if user is working with only inactive agents
  const atLeastOneActiveAgentSelected = selectionMode === 'manual' ? !!selectedAgents.find(agent => agent.active) : totalAgents > totalInactiveAgents;
  const totalActiveAgents = totalAgents - totalInactiveAgents;
  const agentCount = selectionMode === 'manual' ? selectedAgents.length : totalActiveAgents - (managedAgents === null || managedAgents === void 0 ? void 0 : managedAgents.length);
  const agents = selectionMode === 'manual' ? selectedAgents : selectionQuery;
  const [tagsPopoverButton, setTagsPopoverButton] = (0, _react.useState)();
  const {
    diagnosticFileUploadEnabled
  } = _services.ExperimentalFeaturesService.get();
  const menuItems = [{
    name: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentBulkActions.addRemoveTags",
      "data-test-subj": "agentBulkActionsAddRemoveTags",
      defaultMessage: "Add / remove tags"
    }),
    icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
      type: "tag",
      size: "m"
    }),
    disabled: !atLeastOneActiveAgentSelected,
    onClick: event => {
      setTagsPopoverButton(event.target.closest('button'));
      setIsTagAddVisible(!isTagAddVisible);
    }
  }, {
    name: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentBulkActions.reassignPolicy",
      "data-test-subj": "agentBulkActionsReassign",
      defaultMessage: "Assign to new policy"
    }),
    icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
      type: "pencil",
      size: "m"
    }),
    disabled: !atLeastOneActiveAgentSelected,
    onClick: () => {
      closeMenu();
      setIsReassignFlyoutOpen(true);
    }
  }, {
    name: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentBulkActions.unenrollAgents",
      "data-test-subj": "agentBulkActionsUnenroll",
      defaultMessage: "Unenroll {agentCount, plural, one {# agent} other {# agents}}",
      values: {
        agentCount
      }
    }),
    icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
      type: "trash",
      size: "m"
    }),
    disabled: !atLeastOneActiveAgentSelected,
    onClick: () => {
      closeMenu();
      setIsUnenrollModalOpen(true);
    }
  }, {
    name: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentBulkActions.upgradeAgents",
      "data-test-subj": "agentBulkActionsUpgrade",
      defaultMessage: "Upgrade {agentCount, plural, one {# agent} other {# agents}}",
      values: {
        agentCount
      }
    }),
    icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
      type: "refresh",
      size: "m"
    }),
    disabled: !atLeastOneActiveAgentSelected,
    onClick: () => {
      closeMenu();
      setUpgradeModalState({
        isOpen: true,
        isScheduled: false
      });
    }
  }, {
    name: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentBulkActions.scheduleUpgradeAgents",
      "data-test-subj": "agentBulkActionsScheduleUpgrade",
      defaultMessage: "Schedule upgrade for {agentCount, plural, one {# agent} other {# agents}}",
      values: {
        agentCount
      }
    }),
    icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
      type: "timeRefresh",
      size: "m"
    }),
    disabled: !atLeastOneActiveAgentSelected || !isLicenceAllowingScheduleUpgrade,
    onClick: () => {
      closeMenu();
      setUpgradeModalState({
        isOpen: true,
        isScheduled: true
      });
    }
  }];
  if (diagnosticFileUploadEnabled) {
    menuItems.push({
      name: /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
        id: "xpack.fleet.agentBulkActions.requestDiagnostics",
        "data-test-subj": "agentBulkActionsRequestDiagnostics",
        defaultMessage: "Request diagnostics for {agentCount, plural, one {# agent} other {# agents}}",
        values: {
          agentCount
        }
      }),
      icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
        type: "download",
        size: "m"
      }),
      disabled: !atLeastOneActiveAgentSelected,
      onClick: () => {
        closeMenu();
        setIsRequestDiagnosticsModalOpen(true);
      }
    });
  }
  const panels = [{
    id: 0,
    items: menuItems
  }];
  const getSelectedTagsFromAgents = (0, _react.useMemo)(() => (0, _utils.getCommonTags)(agents, visibleAgents !== null && visibleAgents !== void 0 ? visibleAgents : [], agentPolicies), [agents, visibleAgents, agentPolicies]);
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, isReassignFlyoutOpen && /*#__PURE__*/_react.default.createElement(_eui.EuiPortal, null, /*#__PURE__*/_react.default.createElement(_components.AgentReassignAgentPolicyModal, {
    agents: agents,
    onClose: () => {
      setIsReassignFlyoutOpen(false);
      refreshAgents();
    }
  })), isUnenrollModalOpen && /*#__PURE__*/_react.default.createElement(_eui.EuiPortal, null, /*#__PURE__*/_react.default.createElement(_components.AgentUnenrollAgentModal, {
    agents: agents,
    agentCount: agentCount,
    onClose: () => {
      setIsUnenrollModalOpen(false);
      refreshAgents({
        refreshTags: true
      });
    }
  })), updateModalState.isOpen && /*#__PURE__*/_react.default.createElement(_eui.EuiPortal, null, /*#__PURE__*/_react.default.createElement(_components.AgentUpgradeAgentModal, {
    agents: agents,
    agentCount: agentCount,
    isScheduled: updateModalState.isScheduled,
    onClose: () => {
      setUpgradeModalState({
        isOpen: false,
        isScheduled: false
      });
      refreshAgents();
    }
  })), isTagAddVisible && /*#__PURE__*/_react.default.createElement(_tags_add_remove.TagsAddRemove, {
    agents: Array.isArray(agents) ? agents.map(agent => agent.id) : agents,
    allTags: allTags !== null && allTags !== void 0 ? allTags : [],
    selectedTags: getSelectedTagsFromAgents,
    button: tagsPopoverButton,
    onTagsUpdated: () => {
      refreshAgents({
        refreshTags: true
      });
    },
    onClosePopover: () => {
      setIsTagAddVisible(false);
      closeMenu();
    }
  }), isRequestDiagnosticsModalOpen && /*#__PURE__*/_react.default.createElement(_eui.EuiPortal, null, /*#__PURE__*/_react.default.createElement(_agent_request_diagnostics_modal.AgentRequestDiagnosticsModal, {
    agents: agents,
    agentCount: agentCount,
    onClose: () => {
      setIsRequestDiagnosticsModalOpen(false);
    }
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    gutterSize: "m",
    alignItems: "center"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiPopover, {
    id: "agentBulkActionsMenu",
    button: /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
      fill: true,
      iconType: "arrowDown",
      iconSide: "right",
      onClick: onClickMenu,
      "data-test-subj": "agentBulkActionsButton"
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.fleet.agentBulkActions.actions",
      defaultMessage: "Actions"
    })),
    isOpen: isMenuOpen,
    closePopover: closeMenu,
    panelPaddingSize: "none",
    anchorPosition: "downLeft"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiContextMenu, {
    initialPanelId: 0,
    panels: panels
  })))));
};
exports.AgentBulkActions = AgentBulkActions;