"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateAuthz = void 0;
exports.calculatePackagePrivilegesFromCapabilities = calculatePackagePrivilegesFromCapabilities;
exports.calculatePackagePrivilegesFromKibanaPrivileges = calculatePackagePrivilegesFromKibanaPrivileges;
var _plugin = require("./constants/plugin");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const calculateAuthz = ({
  fleet,
  integrations,
  isSuperuser
}) => ({
  fleet: {
    all: fleet.all && (integrations.all || integrations.read),
    // These are currently used by Fleet Server setup
    setup: fleet.all || fleet.setup,
    readEnrollmentTokens: fleet.all || fleet.setup,
    readAgentPolicies: fleet.all || fleet.setup
  },
  integrations: {
    readPackageInfo: fleet.all || fleet.setup || integrations.all || integrations.read,
    readInstalledPackages: integrations.all || integrations.read,
    installPackages: fleet.all && integrations.all,
    upgradePackages: fleet.all && integrations.all,
    removePackages: fleet.all && integrations.all,
    uploadPackages: isSuperuser,
    readPackageSettings: fleet.all && integrations.all,
    writePackageSettings: fleet.all && integrations.all,
    readIntegrationPolicies: fleet.all && (integrations.all || integrations.read),
    writeIntegrationPolicies: fleet.all && integrations.all
  }
});
exports.calculateAuthz = calculateAuthz;
function calculatePackagePrivilegesFromCapabilities(capabilities) {
  if (!capabilities) {
    return {};
  }
  const endpointActions = Object.entries(_constants.ENDPOINT_PRIVILEGES).reduce((acc, [privilege, {
    privilegeName
  }]) => {
    return {
      ...acc,
      [privilege]: {
        executePackageAction: capabilities.siem[privilegeName] || false
      }
    };
  }, {});
  const transformActions = Object.keys(capabilities.transform).reduce((acc, privilegeName) => {
    return {
      ...acc,
      [privilegeName]: {
        executePackageAction: capabilities.transform[privilegeName] || false
      }
    };
  }, {});
  return {
    endpoint: {
      actions: endpointActions
    },
    transform: {
      actions: transformActions
    }
  };
}
function getAuthorizationFromPrivileges(kibanaPrivileges, prefix, searchPrivilege) {
  const privilege = kibanaPrivileges.find(p => p.privilege.endsWith(`${prefix}${searchPrivilege}`));
  return (privilege === null || privilege === void 0 ? void 0 : privilege.authorized) || false;
}
function calculatePackagePrivilegesFromKibanaPrivileges(kibanaPrivileges) {
  if (!kibanaPrivileges || !kibanaPrivileges.length) {
    return {};
  }
  const endpointActions = Object.entries(_constants.ENDPOINT_PRIVILEGES).reduce((acc, [privilege, {
    appId,
    privilegeSplit,
    privilegeName
  }]) => {
    const kibanaPrivilege = getAuthorizationFromPrivileges(kibanaPrivileges, `${appId}${privilegeSplit}`, privilegeName);
    return {
      ...acc,
      [privilege]: {
        executePackageAction: kibanaPrivilege
      }
    };
  }, {});
  const hasTransformAdmin = getAuthorizationFromPrivileges(kibanaPrivileges, `${_plugin.TRANSFORM_PLUGIN_ID}-`, `admin`);
  const transformActions = {
    canCreateTransform: {
      executePackageAction: hasTransformAdmin
    },
    canDeleteTransform: {
      executePackageAction: hasTransformAdmin
    },
    canStartStopTransform: {
      executePackageAction: hasTransformAdmin
    },
    canGetTransform: {
      executePackageAction: getAuthorizationFromPrivileges(kibanaPrivileges, `${_plugin.TRANSFORM_PLUGIN_ID}-`, `read`)
    }
  };
  return {
    endpoint: {
      actions: endpointActions
    },
    transform: {
      actions: transformActions
    }
  };
}