"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPartitionTheme = void 0;
var _types = require("../../common/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_SIZE = 1000;
const getPieDonutWaffleCommonTheme = (visParams, dimensions, rescaleFactor = 1) => {
  const isSplitChart = Boolean(visParams.dimensions.splitColumn || visParams.dimensions.splitRow);
  const preventLinksFromShowing = (visParams.labels.position === _types.LabelPositions.INSIDE || isSplitChart) && visParams.labels.show;
  const usingOuterSizeRatio = dimensions && !isSplitChart ? {
    outerSizeRatio:
    // Cap the ratio to 1 and then rescale
    rescaleFactor * Math.min(MAX_SIZE / Math.min(dimensions === null || dimensions === void 0 ? void 0 : dimensions.width, dimensions === null || dimensions === void 0 ? void 0 : dimensions.height), 1)
  } : {
    outerSizeRatio: undefined
  };
  const theme = {};
  theme.partition = {
    ...(usingOuterSizeRatio !== null && usingOuterSizeRatio !== void 0 ? usingOuterSizeRatio : {})
  };
  if (visParams.labels.show && visParams.labels.position === _types.LabelPositions.DEFAULT && visParams.labels.last_level) {
    var _visParams$labels$tru;
    theme.partition.linkLabel = {
      maxCount: Number.POSITIVE_INFINITY,
      maximumSection: Number.POSITIVE_INFINITY,
      maxTextLength: (_visParams$labels$tru = visParams.labels.truncate) !== null && _visParams$labels$tru !== void 0 ? _visParams$labels$tru : undefined
    };
  }
  if (preventLinksFromShowing || !visParams.labels.show) {
    // Prevent links from showing
    theme.partition.linkLabel = {
      maxCount: 0,
      ...(!visParams.labels.show ? {
        maximumSection: Number.POSITIVE_INFINITY
      } : {})
    };
  }
  if (!preventLinksFromShowing && dimensions && !isSplitChart) {
    // shrink up to 20% to give some room for the linked values
    theme.partition.outerSizeRatio = rescaleFactor;
  }
  return theme;
};
const getDonutSpecificTheme = (visParams, ...args) => {
  const {
    partition,
    ...restTheme
  } = getPieDonutWaffleCommonTheme(visParams, ...args);
  return {
    ...restTheme,
    partition: {
      ...partition,
      emptySizeRatio: visParams.emptySizeRatio
    }
  };
};
const getTreemapMosaicCommonTheme = visParams => {
  if (!visParams.labels.show) {
    return {
      partition: {
        fillLabel: {
          textColor: 'rgba(0,0,0,0)'
        }
      }
    };
  }
  return {};
};
const getSpecificTheme = (chartType, visParams, dimensions, rescaleFactor) => ({
  [_types.ChartTypes.PIE]: () => getPieDonutWaffleCommonTheme(visParams, dimensions, rescaleFactor),
  [_types.ChartTypes.DONUT]: () => getDonutSpecificTheme(visParams, dimensions, rescaleFactor),
  [_types.ChartTypes.TREEMAP]: () => getTreemapMosaicCommonTheme(visParams),
  [_types.ChartTypes.MOSAIC]: () => getTreemapMosaicCommonTheme(visParams),
  [_types.ChartTypes.WAFFLE]: () => getPieDonutWaffleCommonTheme(visParams, dimensions, rescaleFactor)
})[chartType]();
const getPartitionTheme = (chartType, visParams, chartTheme, dimensions, rescaleFactor = 1) => {
  var _chartTheme$barSeries, _chartTheme$barSeries2, _chartTheme$lineSerie, _chartTheme$lineSerie2, _chartTheme$axes, _chartTheme$axes$axis, _visParams$labels$tru2;
  // On small multiples we want the labels to only appear inside
  const isSplitChart = Boolean(visParams.dimensions.splitColumn || visParams.dimensions.splitRow);
  const paddingProps = dimensions && !isSplitChart ? {
    chartPaddings: {
      top: (1 - Math.min(1, MAX_SIZE / (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height))) / 2 * (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height),
      bottom: (1 - Math.min(1, MAX_SIZE / (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height))) / 2 * (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height),
      left: (1 - Math.min(1, MAX_SIZE / (dimensions === null || dimensions === void 0 ? void 0 : dimensions.width))) / 2 * (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height),
      right: (1 - Math.min(1, MAX_SIZE / (dimensions === null || dimensions === void 0 ? void 0 : dimensions.width))) / 2 * (dimensions === null || dimensions === void 0 ? void 0 : dimensions.height)
    }
  } : null;
  const partition = {
    fontFamily: (_chartTheme$barSeries = chartTheme.barSeriesStyle) === null || _chartTheme$barSeries === void 0 ? void 0 : (_chartTheme$barSeries2 = _chartTheme$barSeries.displayValue) === null || _chartTheme$barSeries2 === void 0 ? void 0 : _chartTheme$barSeries2.fontFamily,
    outerSizeRatio: 1,
    minFontSize: 10,
    maxFontSize: 16,
    emptySizeRatio: 0,
    sectorLineStroke: (_chartTheme$lineSerie = chartTheme.lineSeriesStyle) === null || _chartTheme$lineSerie === void 0 ? void 0 : (_chartTheme$lineSerie2 = _chartTheme$lineSerie.point) === null || _chartTheme$lineSerie2 === void 0 ? void 0 : _chartTheme$lineSerie2.fill,
    sectorLineWidth: 1.5,
    circlePadding: 4,
    linkLabel: {
      maxCount: 5,
      fontSize: 11,
      textColor: (_chartTheme$axes = chartTheme.axes) === null || _chartTheme$axes === void 0 ? void 0 : (_chartTheme$axes$axis = _chartTheme$axes.axisTitle) === null || _chartTheme$axes$axis === void 0 ? void 0 : _chartTheme$axes$axis.fill,
      maxTextLength: (_visParams$labels$tru2 = visParams.labels.truncate) !== null && _visParams$labels$tru2 !== void 0 ? _visParams$labels$tru2 : undefined
    }
  };
  const {
    partition: specificPartition = {},
    ...restSpecificTheme
  } = getSpecificTheme(chartType, visParams, dimensions, rescaleFactor);
  return {
    partition: {
      ...partition,
      ...specificPartition
    },
    chartMargins: {
      top: 0,
      bottom: 0,
      left: 0,
      right: 0
    },
    ...(paddingProps !== null && paddingProps !== void 0 ? paddingProps : {}),
    ...restSpecificTheme
  };
};
exports.getPartitionTheme = getPartitionTheme;