"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAddToCase = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/cases-plugin/public");
var _public2 = require("@kbn/observability-shared-plugin/public");
var _kibana_react = require("../../../../utils/kibana_react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function addToCase(http, theCase, attributes, timeRange, owner) {
  const apiPath = `/api/cases/${theCase === null || theCase === void 0 ? void 0 : theCase.id}/comments`;
  const vizPayload = {
    attributes,
    timeRange
  };
  const payload = {
    comment: `!{lens${JSON.stringify(vizPayload)}}`,
    type: 'user',
    owner: owner !== null && owner !== void 0 ? owner : _public2.observabilityFeatureId
  };
  return http.post(apiPath, {
    body: JSON.stringify(payload)
  });
}
const useAddToCase = ({
  lensAttributes,
  getToastText,
  timeRange,
  appId,
  owner = _public2.observabilityFeatureId
}) => {
  const [isSaving, setIsSaving] = (0, _react.useState)(false);
  const [isCasesOpen, setIsCasesOpen] = (0, _react.useState)(false);
  const {
    http,
    application: {
      navigateToApp
    },
    notifications: {
      toasts
    },
    storage
  } = (0, _kibana_react.useKibana)().services;
  const onCaseClicked = (0, _react.useCallback)(theCase => {
    if (theCase && lensAttributes) {
      setIsCasesOpen(false);
      setIsSaving(true);
      addToCase(http, theCase, lensAttributes, timeRange, owner).then(() => {
        setIsSaving(false);
        toasts.addSuccess({
          title: _i18n.i18n.translate('xpack.exploratoryView.expView.heading.addToCase.notification', {
            defaultMessage: 'Successfully added visualization to the case: {caseTitle}',
            values: {
              caseTitle: theCase.title
            }
          }),
          text: getToastText(theCase)
        }, {
          toastLifeTimeMs: 10000
        });
      }, error => {
        toasts.addError(error, {
          title: _i18n.i18n.translate('xpack.exploratoryView.expView.heading.addToCase.notification.error', {
            defaultMessage: 'Failed to add visualization to the selected case.'
          })
        });
      });
    } else {
      /* save lens attributes and timerange to local storage,
       ** so the description field will be automatically filled on case creation page.
       */
      storage === null || storage === void 0 ? void 0 : storage.set(_public.DRAFT_COMMENT_STORAGE_ID, {
        commentId: 'description',
        comment: `!{lens${JSON.stringify({
          timeRange,
          attributes: lensAttributes
        })}}`,
        position: '',
        caseTitle: '',
        caseTags: ''
      });
      navigateToApp(appId !== null && appId !== void 0 ? appId : _public2.observabilityFeatureId, {
        deepLinkId: _public.CasesDeepLinkId.casesCreate,
        openInNewTab: true
      });
    }
  }, [appId, getToastText, http, lensAttributes, navigateToApp, owner, storage, timeRange, toasts]);
  return {
    onCaseClicked,
    isSaving,
    isCasesOpen,
    setIsCasesOpen
  };
};
exports.useAddToCase = useAddToCase;