"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEventAnnotationService = getEventAnnotationService;
exports.hasIcon = hasIcon;
var _lodash = require("lodash");
var _common = require("@kbn/data-plugin/common");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function hasIcon(icon) {
  return icon != null && icon !== 'empty';
}
function getEventAnnotationService() {
  const annotationsToExpression = annotations => {
    const [queryBasedAnnotations, manualBasedAnnotations] = (0, _lodash.partition)(annotations, _helpers.isQueryAnnotationConfig);
    const expressions = [];
    for (const annotation of manualBasedAnnotations) {
      if ((0, _helpers.isRangeAnnotationConfig)(annotation)) {
        const {
          label,
          color,
          key,
          outside,
          id
        } = annotation;
        const {
          timestamp: time,
          endTimestamp: endTime
        } = key;
        expressions.push({
          type: 'expression',
          chain: [{
            type: 'function',
            function: 'manual_range_event_annotation',
            arguments: {
              id: [id],
              time: [time],
              endTime: [endTime],
              label: [label || _helpers.defaultAnnotationLabel],
              color: [color || _helpers.defaultAnnotationRangeColor],
              outside: [Boolean(outside)],
              isHidden: [Boolean(annotation.isHidden)]
            }
          }]
        });
      } else {
        const {
          label,
          color,
          lineStyle,
          lineWidth,
          icon,
          key,
          textVisibility,
          id
        } = annotation;
        expressions.push({
          type: 'expression',
          chain: [{
            type: 'function',
            function: 'manual_point_event_annotation',
            arguments: {
              id: [id],
              time: [key.timestamp],
              label: [label || _helpers.defaultAnnotationLabel],
              color: [color || _helpers.defaultAnnotationColor],
              lineWidth: [lineWidth || 1],
              lineStyle: [lineStyle || 'solid'],
              icon: hasIcon(icon) ? [icon] : ['triangle'],
              textVisibility: [textVisibility || false],
              isHidden: [Boolean(annotation.isHidden)]
            }
          }]
        });
      }
    }
    for (const annotation of queryBasedAnnotations) {
      const {
        id,
        label,
        color,
        lineStyle,
        lineWidth,
        icon,
        timeField,
        textVisibility,
        textField,
        filter,
        extraFields
      } = annotation;
      expressions.push({
        type: 'expression',
        chain: [{
          type: 'function',
          function: 'query_point_event_annotation',
          arguments: {
            id: [id],
            timeField: timeField ? [timeField] : [],
            label: [label || _helpers.defaultAnnotationLabel],
            color: [color || _helpers.defaultAnnotationColor],
            lineWidth: [lineWidth || 1],
            lineStyle: [lineStyle || 'solid'],
            icon: hasIcon(icon) ? [icon] : ['triangle'],
            textVisibility: [textVisibility || false],
            textField: textVisibility && textField ? [textField] : [],
            filter: filter ? [(0, _common.queryToAst)(filter)] : [],
            extraFields: extraFields || [],
            isHidden: [Boolean(annotation.isHidden)]
          }
        }]
      });
    }
    return expressions;
  };
  return {
    toExpression: annotationsToExpression,
    toFetchExpression: ({
      interval,
      groups
    }) => {
      if (groups.length === 0) {
        return [];
      }
      const groupsExpressions = groups.filter(g => g.annotations.some(a => !a.isHidden)).map(({
        annotations,
        indexPatternId,
        ignoreGlobalFilters
      }) => {
        const indexPatternExpression = {
          type: 'expression',
          chain: [{
            type: 'function',
            function: 'indexPatternLoad',
            arguments: {
              id: [indexPatternId]
            }
          }]
        };
        const annotationExpressions = annotationsToExpression(annotations);
        return {
          type: 'expression',
          chain: [{
            type: 'function',
            function: 'event_annotation_group',
            arguments: {
              dataView: [indexPatternExpression],
              annotations: [...annotationExpressions],
              ignoreGlobalFilters: [Boolean(ignoreGlobalFilters)]
            }
          }]
        };
      });
      const fetchExpression = {
        type: 'expression',
        chain: [{
          type: 'function',
          function: 'kibana',
          arguments: {}
        }, {
          type: 'function',
          function: 'fetch_event_annotations',
          arguments: {
            interval: [interval],
            groups: [...groupsExpressions]
          }
        }]
      };
      return [fetchExpression];
    }
  };
}