"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerEnginesRoutes = registerEnginesRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _error_codes = require("../../../common/types/error_codes");
var _create_api_key = require("../../lib/engines/create_api_key");
var _fetch_indices_stats = require("../../lib/engines/fetch_indices_stats");
var _field_capabilities = require("../../lib/engines/field_capabilities");
var _create_error = require("../../utils/create_error");
var _elasticsearch_error_handler = require("../../utils/elasticsearch_error_handler");
var _identify_exceptions = require("../../utils/identify_exceptions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerEnginesRoutes({
  log,
  router
}) {
  router.get({
    path: '/internal/enterprise_search/engines',
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        q: _configSchema.schema.maybe(_configSchema.schema.string()),
        size: _configSchema.schema.number({
          defaultValue: 10,
          min: 1
        })
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engines = await client.asCurrentUser.transport.request({
      method: 'GET',
      path: `/_application/search_application`,
      querystring: request.query
    });
    return response.ok({
      body: engines
    });
  }));
  router.get({
    path: '/internal/enterprise_search/engines/{engine_name}',
    validate: {
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engine = await client.asCurrentUser.transport.request({
      method: 'GET',
      path: `/_application/search_application/${request.params.engine_name}`
    });
    const indicesStats = await (0, _fetch_indices_stats.fetchIndicesStats)(client, engine.indices);
    return response.ok({
      body: {
        ...engine,
        indices: indicesStats
      }
    });
  }));
  router.put({
    path: '/internal/enterprise_search/engines/{engine_name}',
    validate: {
      body: _configSchema.schema.object({
        indices: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        name: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        create: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      const engine = await client.asCurrentUser.transport.request({
        body: {
          indices: request.body.indices
        },
        method: 'PUT',
        path: `/_application/search_application/${request.params.engine_name}`,
        querystring: request.query
      });
      return response.ok({
        body: engine
      });
    } catch (error) {
      if ((0, _identify_exceptions.isVersionConflictEngineException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.SEARCH_APPLICATION_ALREADY_EXISTS,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.createSearchApplication.searchApplciationExistsError', {
            defaultMessage: 'Search application name already taken. Choose another name.'
          }),
          response,
          statusCode: 409
        });
      }
      throw error;
    }
  }));
  router.delete({
    path: '/internal/enterprise_search/engines/{engine_name}',
    validate: {
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engine = await client.asCurrentUser.transport.request({
      method: 'DELETE',
      path: `_application/search_application/${request.params.engine_name}`
    });
    return response.ok({
      body: engine
    });
  }));
  router.post({
    path: '/internal/enterprise_search/engines/{engine_name}/search',
    validate: {
      body: _configSchema.schema.object({}, {
        unknowns: 'allow'
      }),
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string(),
        from: _configSchema.schema.maybe(_configSchema.schema.number()),
        size: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const engines = await client.asCurrentUser.search({
      index: request.params.engine_name,
      ...request.body
    });
    return response.ok({
      body: engines
    });
  }));
  router.post({
    path: '/internal/enterprise_search/engines/{engine_name}/api_key',
    validate: {
      body: _configSchema.schema.object({
        keyName: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const engineName = decodeURIComponent(request.params.engine_name);
    const {
      keyName
    } = request.body;
    const {
      client
    } = (await context.core).elasticsearch;
    const apiKey = await (0, _create_api_key.createApiKey)(client, engineName, keyName);
    return response.ok({
      body: {
        apiKey
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.get({
    path: '/internal/enterprise_search/engines/{engine_name}/field_capabilities',
    validate: {
      params: _configSchema.schema.object({
        engine_name: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    try {
      const engineName = decodeURIComponent(request.params.engine_name);
      const {
        client
      } = (await context.core).elasticsearch;
      const engine = await client.asCurrentUser.transport.request({
        method: 'GET',
        path: `/_application/search_application/${engineName}`
      });
      const data = await (0, _field_capabilities.fetchEngineFieldCapabilities)(client, engine);
      return response.ok({
        body: data,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (e) {
      if ((0, _identify_exceptions.isNotFoundException)(e)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.ENGINE_NOT_FOUND,
          message: 'Could not find engine',
          response,
          statusCode: 404
        });
      }
      throw e;
    }
  }));
}