"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseFieldsCapabilities = exports.fetchEngineFieldCapabilities = void 0;
var _available_indices = require("./available_indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchEngineFieldCapabilities = async (client, engine) => {
  const {
    name,
    updated_at_millis,
    indices
  } = engine;
  const availableIndicesList = await (0, _available_indices.availableIndices)(client, indices);
  if (!availableIndicesList.length) {
    return {
      fields: [],
      name,
      updated_at_millis
    };
  }
  const fieldCapabilities = await client.asCurrentUser.fieldCaps({
    fields: '*',
    filters: '-metadata',
    include_unmapped: true,
    index: availableIndicesList
  });
  const fields = parseFieldsCapabilities(fieldCapabilities);
  return {
    fields,
    name,
    updated_at_millis
  };
};
exports.fetchEngineFieldCapabilities = fetchEngineFieldCapabilities;
const ensureIndices = indices => {
  if (!indices) return [];
  return Array.isArray(indices) ? indices : [indices];
};
const parseFieldsCapabilities = fieldCapsResponse => {
  const {
    fields,
    indices: indexOrIndices
  } = fieldCapsResponse;
  const indices = ensureIndices(indexOrIndices);
  return Object.entries(fields).map(([fieldName, typesObject]) => {
    const typeValues = Object.values(typesObject);
    const type = calculateType(Object.keys(typesObject));
    const indicesToType = typeValues.reduce((acc, {
      type: indexType,
      indices: typeIndexOrIndices
    }) => {
      const typeIndices = ensureIndices(typeIndexOrIndices);
      typeIndices.forEach(index => {
        acc[index] = indexType;
      });
      return acc;
    }, {});
    const fieldIndices = Object.keys(indicesToType).length > 0 ? indices.map(index => {
      const indexType = indicesToType[index] || 'unmapped';
      return {
        name: index,
        type: indexType
      };
    }) : indices.map(index => ({
      name: index,
      type
    }));
    const searchable = Object.values(typesObject).some(t => t.searchable);
    const aggregatable = Object.values(typesObject).some(t => t.aggregatable);
    const metadataField = Object.values(typesObject).every(t => t.metadata_field);
    return {
      aggregatable,
      indices: fieldIndices,
      name: fieldName,
      searchable,
      type,
      ...(metadataField === undefined ? {} : {
        metadata_field: metadataField
      })
    };
  }).sort((a, b) => a.name.localeCompare(b.name));
};
exports.parseFieldsCapabilities = parseFieldsCapabilities;
const calculateType = types => {
  // If there is only one type, return it
  if (types.length === 1) return types[0];

  // Unmapped types are ignored for the purposes of determining the type
  // If all of the mapped types are the same, return that type
  const mapped = types.filter(t => t !== 'unmapped');
  if (new Set(mapped).size === 1) return mapped[0];

  // Otherwise there is a conflict
  return 'conflict';
};