"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useWorkplaceSearchBreadcrumbs = exports.useSearchExperiencesBreadcrumbs = exports.useGenerateBreadcrumbs = exports.useEuiBreadcrumbs = exports.useEnterpriseSearchEnginesBreadcrumbs = exports.useEnterpriseSearchContentBreadcrumbs = exports.useEnterpriseSearchBreadcrumbs = exports.useElasticsearchBreadcrumbs = exports.useAppSearchBreadcrumbs = exports.useAnalyticsBreadcrumbs = void 0;
var _kea = require("kea");
var _constants = require("../../../../common/constants");
var _strip_slashes = require("../../../../common/strip_slashes");
var _http = require("../http");
var _kibana = require("../kibana");
var _react_router_helpers = require("../react_router_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// A trail of breadcrumb text

/**
 * Generate an array of breadcrumbs based on:
 * 1. A passed array of breadcrumb text (the trail prop)
 * 2. The current React Router path
 *
 * To correctly generate working breadcrumbs, ensure the trail array passed to
 * SetPageChrome matches up with the routed path. For example, a page with a trail of:
 *    `['Groups', 'Example Group Name', 'Source Prioritization']`
 * should have a router pathname of:
 *   `'/groups/{example-group-id}/source_prioritization'`
 *
 * Which should then generate the following breadcrumb output:
 * Groups (linked to `/groups`)
 * > Example Group Name (linked to `/groups/{example-group-id}`)
 * > Source Prioritization (linked to `/groups/{example-group-id}/source_prioritization`)
 */

const useGenerateBreadcrumbs = trail => {
  const {
    history
  } = (0, _kea.useValues)(_kibana.KibanaLogic);
  const pathArray = (0, _strip_slashes.stripLeadingSlash)(history.location.pathname).split('/');
  return trail.map((text, i) => {
    const path = pathArray[i] ? '/' + pathArray.slice(0, i + 1).join('/') : undefined;
    return {
      text,
      path
    };
  });
};

/**
 * Convert IBreadcrumb objects to React-Router-friendly EUI breadcrumb objects
 * https://elastic.github.io/eui/#/navigation/breadcrumbs
 *
 * NOTE: Per EUI best practices, we remove the link behavior and
 * generate an inactive breadcrumb for the last breadcrumb in the list.
 */
exports.useGenerateBreadcrumbs = useGenerateBreadcrumbs;
const useEuiBreadcrumbs = breadcrumbs => {
  const {
    navigateToUrl,
    history
  } = (0, _kea.useValues)(_kibana.KibanaLogic);
  const {
    http
  } = (0, _kea.useValues)(_http.HttpLogic);
  return breadcrumbs.map(({
    text,
    path,
    shouldNotCreateHref
  }, i) => {
    const breadcrumb = {
      text
    };
    const isLastBreadcrumb = i === breadcrumbs.length - 1;
    if (path && !isLastBreadcrumb) {
      breadcrumb.href = (0, _react_router_helpers.createHref)(path, {
        history,
        http
      }, {
        shouldNotCreateHref
      });
      breadcrumb.onClick = event => {
        if ((0, _react_router_helpers.letBrowserHandleEvent)(event)) return;
        event.preventDefault();
        navigateToUrl(path, {
          shouldNotCreateHref
        });
      };
    }
    return breadcrumb;
  });
};

/**
 * Product-specific breadcrumb helpers
 */
exports.useEuiBreadcrumbs = useEuiBreadcrumbs;
const useEnterpriseSearchBreadcrumbs = (breadcrumbs = []) => useEuiBreadcrumbs([{
  text: _constants.ENTERPRISE_SEARCH_OVERVIEW_PLUGIN.NAME,
  path: _constants.ENTERPRISE_SEARCH_OVERVIEW_PLUGIN.URL,
  shouldNotCreateHref: true
}, ...breadcrumbs]);
exports.useEnterpriseSearchBreadcrumbs = useEnterpriseSearchBreadcrumbs;
const useAnalyticsBreadcrumbs = (breadcrumbs = []) => useEnterpriseSearchBreadcrumbs([{
  text: _constants.ANALYTICS_PLUGIN.NAME,
  path: '/'
}, ...breadcrumbs]);
exports.useAnalyticsBreadcrumbs = useAnalyticsBreadcrumbs;
const useElasticsearchBreadcrumbs = (breadcrumbs = []) => useEnterpriseSearchBreadcrumbs([{
  text: 'Getting started with Elasticsearch',
  path: '/'
}, ...breadcrumbs]);
exports.useElasticsearchBreadcrumbs = useElasticsearchBreadcrumbs;
const useAppSearchBreadcrumbs = (breadcrumbs = []) => useEnterpriseSearchBreadcrumbs([{
  text: _constants.APP_SEARCH_PLUGIN.NAME,
  path: '/'
}, ...breadcrumbs]);
exports.useAppSearchBreadcrumbs = useAppSearchBreadcrumbs;
const useWorkplaceSearchBreadcrumbs = (breadcrumbs = []) => useEnterpriseSearchBreadcrumbs([{
  text: _constants.WORKPLACE_SEARCH_PLUGIN.NAME,
  path: '/'
}, ...breadcrumbs]);
exports.useWorkplaceSearchBreadcrumbs = useWorkplaceSearchBreadcrumbs;
const useEnterpriseSearchContentBreadcrumbs = (breadcrumbs = []) => useEnterpriseSearchBreadcrumbs([{
  text: _constants.ENTERPRISE_SEARCH_CONTENT_PLUGIN.NAV_TITLE,
  path: '/'
}, ...breadcrumbs]);
exports.useEnterpriseSearchContentBreadcrumbs = useEnterpriseSearchContentBreadcrumbs;
const useSearchExperiencesBreadcrumbs = (breadcrumbs = []) => useEnterpriseSearchBreadcrumbs([{
  text: _constants.SEARCH_EXPERIENCES_PLUGIN.NAV_TITLE,
  path: '/'
}, ...breadcrumbs]);
exports.useSearchExperiencesBreadcrumbs = useSearchExperiencesBreadcrumbs;
const useEnterpriseSearchEnginesBreadcrumbs = (breadcrumbs = []) => useEnterpriseSearchBreadcrumbs(breadcrumbs);
exports.useEnterpriseSearchEnginesBreadcrumbs = useEnterpriseSearchEnginesBreadcrumbs;