"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTextExpansionError = exports.TextExpansionCalloutLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _api = require("../../../../../../../common/types/api");
var _ml = require("../../../../../../../common/types/ml");
var _handle_api_errors = require("../../../../../shared/flash_messages/handle_api_errors");
var _create_text_expansion_model_api_logic = require("../../../../api/ml_models/text_expansion/create_text_expansion_model_api_logic");
var _fetch_text_expansion_model_api_logic = require("../../../../api/ml_models/text_expansion/fetch_text_expansion_model_api_logic");
var _start_text_expansion_model_api_logic = require("../../../../api/ml_models/text_expansion/start_text_expansion_model_api_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FETCH_TEXT_EXPANSION_MODEL_POLLING_DURATION = 5000; // 5 seconds
const FETCH_TEXT_EXPANSION_MODEL_POLLING_DURATION_ON_FAILURE = 30000; // 30 seconds

/**
 * Extracts the topmost error in precedence order (create > start > fetch).
 * @param createError
 * @param fetchError
 * @param startError
 * @returns the extracted error or null if there is no error
 */
const getTextExpansionError = (createError, fetchError, startError) => {
  return createError !== undefined ? {
    title: _i18n.i18n.translate('xpack.enterpriseSearch.content.indices.pipelines.textExpansionCreateError.title', {
      defaultMessage: 'Error with ELSER deployment'
    }),
    message: (0, _handle_api_errors.getErrorsFromHttpResponse)(createError)[0]
  } : startError !== undefined ? {
    title: _i18n.i18n.translate('xpack.enterpriseSearch.content.indices.pipelines.textExpansionStartError.title', {
      defaultMessage: 'Error starting ELSER deployment'
    }),
    message: (0, _handle_api_errors.getErrorsFromHttpResponse)(startError)[0]
  } : fetchError !== undefined ? {
    title: _i18n.i18n.translate('xpack.enterpriseSearch.content.indices.pipelines.textExpansionFetchError.title', {
      defaultMessage: 'Error fetching ELSER model'
    }),
    message: (0, _handle_api_errors.getErrorsFromHttpResponse)(fetchError)[0]
  } : null;
};
exports.getTextExpansionError = getTextExpansionError;
const TextExpansionCalloutLogic = (0, _kea.kea)({
  actions: {
    clearTextExpansionModelPollingId: true,
    createTextExpansionModelPollingTimeout: duration => ({
      duration
    }),
    setTextExpansionModelPollingId: pollTimeoutId => ({
      pollTimeoutId
    }),
    startPollingTextExpansionModel: true,
    stopPollingTextExpansionModel: true
  },
  connect: {
    actions: [_create_text_expansion_model_api_logic.CreateTextExpansionModelApiLogic, ['makeRequest as createTextExpansionModel', 'apiSuccess as createTextExpansionModelSuccess', 'apiError as createTextExpansionModelError'], _fetch_text_expansion_model_api_logic.FetchTextExpansionModelApiLogic, ['makeRequest as fetchTextExpansionModel', 'apiSuccess as fetchTextExpansionModelSuccess', 'apiError as fetchTextExpansionModelError'], _start_text_expansion_model_api_logic.StartTextExpansionModelApiLogic, ['makeRequest as startTextExpansionModel', 'apiSuccess as startTextExpansionModelSuccess', 'apiError as startTextExpansionModelError']],
    values: [_create_text_expansion_model_api_logic.CreateTextExpansionModelApiLogic, ['data as createdTextExpansionModel', 'status as createTextExpansionModelStatus', 'error as createTextExpansionModelError'], _fetch_text_expansion_model_api_logic.FetchTextExpansionModelApiLogic, ['data as textExpansionModel', 'error as fetchTextExpansionModelError'], _start_text_expansion_model_api_logic.StartTextExpansionModelApiLogic, ['status as startTextExpansionModelStatus', 'error as startTextExpansionModelError']]
  },
  events: ({
    actions,
    values
  }) => ({
    afterMount: () => {
      actions.fetchTextExpansionModel(undefined);
    },
    beforeUnmount: () => {
      if (values.textExpansionModelPollTimeoutId !== null) {
        actions.stopPollingTextExpansionModel();
      }
    }
  }),
  listeners: ({
    actions,
    values
  }) => ({
    createTextExpansionModelPollingTimeout: ({
      duration
    }) => {
      if (values.textExpansionModelPollTimeoutId !== null) {
        clearTimeout(values.textExpansionModelPollTimeoutId);
      }
      const timeoutId = setTimeout(() => {
        actions.fetchTextExpansionModel(undefined);
      }, duration);
      actions.setTextExpansionModelPollingId(timeoutId);
    },
    createTextExpansionModelSuccess: () => {
      actions.fetchTextExpansionModel(undefined);
      actions.startPollingTextExpansionModel();
    },
    fetchTextExpansionModelError: () => {
      if (values.isPollingTextExpansionModelActive) {
        actions.createTextExpansionModelPollingTimeout(FETCH_TEXT_EXPANSION_MODEL_POLLING_DURATION_ON_FAILURE);
      }
    },
    fetchTextExpansionModelSuccess: data => {
      if ((data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloading) {
        if (!values.isPollingTextExpansionModelActive) {
          actions.startPollingTextExpansionModel();
        } else {
          actions.createTextExpansionModelPollingTimeout(FETCH_TEXT_EXPANSION_MODEL_POLLING_DURATION);
        }
      } else if ((data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloaded && values.isPollingTextExpansionModelActive) {
        actions.stopPollingTextExpansionModel();
      }
    },
    startPollingTextExpansionModel: () => {
      if (values.textExpansionModelPollTimeoutId !== null) {
        clearTimeout(values.textExpansionModelPollTimeoutId);
      }
      actions.createTextExpansionModelPollingTimeout(FETCH_TEXT_EXPANSION_MODEL_POLLING_DURATION);
    },
    startTextExpansionModelSuccess: () => {
      actions.fetchTextExpansionModel(undefined);
    },
    stopPollingTextExpansionModel: () => {
      if (values.textExpansionModelPollTimeoutId !== null) {
        clearTimeout(values.textExpansionModelPollTimeoutId);
        actions.clearTextExpansionModelPollingId();
      }
    }
  }),
  path: ['enterprise_search', 'content', 'text_expansion_callout_logic'],
  reducers: {
    textExpansionModelPollTimeoutId: [null, {
      clearTextExpansionModelPollingId: () => null,
      setTextExpansionModelPollingId: (_, {
        pollTimeoutId
      }) => pollTimeoutId
    }]
  },
  selectors: ({
    selectors
  }) => ({
    isCreateButtonDisabled: [() => [selectors.createTextExpansionModelStatus], status => status !== _api.Status.IDLE && status !== _api.Status.ERROR],
    isModelDownloadInProgress: [() => [selectors.textExpansionModel], data => (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloading],
    isModelDownloaded: [() => [selectors.textExpansionModel], data => (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Downloaded],
    isModelStarted: [() => [selectors.textExpansionModel], data => (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Starting || (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.Started || (data === null || data === void 0 ? void 0 : data.deploymentState) === _ml.MlModelDeploymentState.FullyAllocated],
    isPollingTextExpansionModelActive: [() => [selectors.textExpansionModelPollTimeoutId], pollingTimeoutId => pollingTimeoutId !== null],
    isStartButtonDisabled: [() => [selectors.startTextExpansionModelStatus], status => status !== _api.Status.IDLE && status !== _api.Status.ERROR],
    isModelRunningSingleThreaded: [() => [selectors.textExpansionModel], data =>
    // Running single threaded if model has max 1 deployment on 1 node with 1 thread
    (data === null || data === void 0 ? void 0 : data.targetAllocationCount) * (data === null || data === void 0 ? void 0 : data.threadsPerAllocation) <= 1]
  })
});
exports.TextExpansionCalloutLogic = TextExpansionCalloutLogic;