"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndexViewLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _connectors = require("../../../../../common/types/connectors");
var _flash_messages = require("../../../shared/flash_messages");
var _start_sync_api_logic = require("../../api/connector/start_sync_api_logic");
var _cached_fetch_index_api_logic = require("../../api/index/cached_fetch_index_api_logic");
var _indices = require("../../utils/indices");
var _crawler_logic = require("./crawler/crawler_logic");
var _index_name_logic = require("./index_name_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const IndexViewLogic = (0, _kea.kea)({
  actions: {
    fetchIndex: true,
    recheckIndex: true,
    resetRecheckIndexLoading: true,
    startSync: true
  },
  connect: {
    actions: [_index_name_logic.IndexNameLogic, ['setIndexName'], _cached_fetch_index_api_logic.CachedFetchIndexApiLogic, ['apiError as fetchIndexApiError', 'apiReset as resetFetchIndexApi', 'apiSuccess as fetchIndexApiSuccess', 'makeRequest as makeFetchIndexRequest', 'startPolling as startFetchIndexPoll', 'stopPolling as stopFetchIndexPoll'], _start_sync_api_logic.StartSyncApiLogic, ['apiSuccess as startSyncApiSuccess', 'makeRequest as makeStartSyncRequest'], _crawler_logic.CrawlerLogic, ['fetchCrawlerData']],
    values: [_index_name_logic.IndexNameLogic, ['indexName'], _cached_fetch_index_api_logic.CachedFetchIndexApiLogic, ['fetchIndexApiData', 'status as fetchIndexApiStatus', 'indexData', 'isInitialLoading']]
  },
  events: ({
    actions
  }) => ({
    beforeUnmount: () => {
      actions.stopFetchIndexPoll();
      actions.resetFetchIndexApi();
    }
  }),
  listeners: ({
    actions,
    values
  }) => ({
    fetchIndex: () => {
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      actions.makeFetchIndexRequest({
        indexName
      });
    },
    fetchIndexApiSuccess: index => {
      if ((0, _indices.isCrawlerIndex)(index) && index.name === values.indexName) {
        actions.fetchCrawlerData();
      }
      if (values.recheckIndexLoading) {
        actions.resetRecheckIndexLoading();
        (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.content.searchIndex.index.recheckSuccess.message', {
          defaultMessage: 'Your connector has been rechecked.'
        }));
      }
    },
    recheckIndex: () => actions.fetchIndex(),
    setIndexName: ({
      indexName
    }) => {
      actions.startFetchIndexPoll(indexName);
    },
    startSync: () => {
      if ((0, _indices.isConnectorIndex)(values.fetchIndexApiData)) {
        actions.makeStartSyncRequest({
          connectorId: values.fetchIndexApiData.connector.id
        });
      }
    }
  }),
  path: ['enterprise_search', 'content', 'index_view_logic'],
  reducers: {
    localSyncNowValue: [false, {
      fetchIndexApiSuccess: (_, index) => (0, _indices.isConnectorIndex)(index) ? index.connector.sync_now : false,
      startSyncApiSuccess: () => true
    }],
    recheckIndexLoading: [false, {
      recheckIndex: () => true,
      resetRecheckIndexLoading: () => false
    }]
  },
  selectors: ({
    selectors
  }) => ({
    connector: [() => [selectors.indexData], index => index && ((0, _indices.isConnectorViewIndex)(index) || (0, _indices.isCrawlerIndex)(index)) ? index.connector : undefined],
    connectorError: [() => [selectors.connector], connector => connector === null || connector === void 0 ? void 0 : connector.error],
    connectorId: [() => [selectors.indexData], index => (0, _indices.isConnectorViewIndex)(index) ? index.connector.id : null],
    error: [() => [selectors.connector], connector => (connector === null || connector === void 0 ? void 0 : connector.error) || (connector === null || connector === void 0 ? void 0 : connector.last_sync_error) || null],
    hasAdvancedFilteringFeature: [() => [selectors.connector], connector => {
      var _connector$features$F, _connector$features$F2, _connector$features$F3;
      return connector !== null && connector !== void 0 && connector.features ? (_connector$features$F = (_connector$features$F2 = connector.features[_connectors.FeatureName.SYNC_RULES]) === null || _connector$features$F2 === void 0 ? void 0 : (_connector$features$F3 = _connector$features$F2.advanced) === null || _connector$features$F3 === void 0 ? void 0 : _connector$features$F3.enabled) !== null && _connector$features$F !== void 0 ? _connector$features$F : connector.features[_connectors.FeatureName.FILTERING_ADVANCED_CONFIG] : false;
    }],
    hasBasicFilteringFeature: [() => [selectors.connector], connector => {
      var _connector$features$F4, _connector$features$F5, _connector$features$F6;
      return connector !== null && connector !== void 0 && connector.features ? (_connector$features$F4 = (_connector$features$F5 = connector.features[_connectors.FeatureName.SYNC_RULES]) === null || _connector$features$F5 === void 0 ? void 0 : (_connector$features$F6 = _connector$features$F5.basic) === null || _connector$features$F6 === void 0 ? void 0 : _connector$features$F6.enabled) !== null && _connector$features$F4 !== void 0 ? _connector$features$F4 : connector.features[_connectors.FeatureName.FILTERING_RULES] : false;
    }],
    hasFilteringFeature: [() => [selectors.hasAdvancedFilteringFeature, selectors.hasBasicFilteringFeature], (advancedFeature, basicFeature) => advancedFeature || basicFeature],
    htmlExtraction: [() => [selectors.connector], connector => {
      var _connector$configurat, _connector$configurat2;
      return (_connector$configurat = connector === null || connector === void 0 ? void 0 : (_connector$configurat2 = connector.configuration.extract_full_html) === null || _connector$configurat2 === void 0 ? void 0 : _connector$configurat2.value) !== null && _connector$configurat !== void 0 ? _connector$configurat : undefined;
    }],
    index: [() => [selectors.indexData], data => data ? (0, _indices.indexToViewIndex)(data) : undefined],
    ingestionMethod: [() => [selectors.indexData], data => (0, _indices.getIngestionMethod)(data)],
    ingestionStatus: [() => [selectors.indexData], data => (0, _indices.getIngestionStatus)(data)],
    isCanceling: [() => [selectors.syncStatus], syncStatus => syncStatus === _connectors.SyncStatus.CANCELING],
    isConnectorIndex: [() => [selectors.indexData], data => (0, _indices.isConnectorIndex)(data)],
    isHiddenIndex: [() => [selectors.indexData], data => {
      var _data$name;
      return (data === null || data === void 0 ? void 0 : data.hidden) || ((_data$name = data === null || data === void 0 ? void 0 : data.name) !== null && _data$name !== void 0 ? _data$name : '').startsWith('.');
    }],
    isSyncing: [() => [selectors.indexData, selectors.syncStatus], (indexData, syncStatus) => (indexData === null || indexData === void 0 ? void 0 : indexData.has_in_progress_syncs) || syncStatus === _connectors.SyncStatus.IN_PROGRESS],
    isWaitingForSync: [() => [selectors.fetchIndexApiData, selectors.localSyncNowValue], (data, localSyncNowValue) => {
      var _data$connector;
      return (data === null || data === void 0 ? void 0 : (_data$connector = data.connector) === null || _data$connector === void 0 ? void 0 : _data$connector.sync_now) || localSyncNowValue;
    }],
    lastUpdated: [() => [selectors.fetchIndexApiData], data => (0, _indices.getLastUpdated)(data)],
    pipelineData: [() => [selectors.connector], connector => {
      var _connector$pipeline;
      return (_connector$pipeline = connector === null || connector === void 0 ? void 0 : connector.pipeline) !== null && _connector$pipeline !== void 0 ? _connector$pipeline : undefined;
    }],
    syncStatus: [() => [selectors.fetchIndexApiData], data => {
      var _data$connector$last_, _data$connector2;
      return (_data$connector$last_ = data === null || data === void 0 ? void 0 : (_data$connector2 = data.connector) === null || _data$connector2 === void 0 ? void 0 : _data$connector2.last_sync_status) !== null && _data$connector$last_ !== void 0 ? _data$connector$last_ : null;
    }]
  })
});
exports.IndexViewLogic = IndexViewLogic;