"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrawlerDomainDetailLogic = void 0;
var _kea = require("kea");
var _api = require("../../../../../../../common/types/api");
var _encode_path_params = require("../../../../../shared/encode_path_params");
var _flash_messages = require("../../../../../shared/flash_messages");
var _http = require("../../../../../shared/http");
var _kibana = require("../../../../../shared/kibana");
var _delete_crawler_domain_api_logic = require("../../../../api/crawler/delete_crawler_domain_api_logic");
var _utils = require("../../../../api/crawler/utils");
var _routes = require("../../../../routes");
var _index_name_logic = require("../../index_name_logic");
var _search_index = require("../../search_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CrawlerDomainDetailLogic = (0, _kea.kea)({
  actions: {
    deleteDomain: () => true,
    deleteDomainComplete: () => true,
    fetchDomainData: domainId => ({
      domainId
    }),
    receiveDomainData: domain => ({
      domain
    }),
    submitAuthUpdate: auth => ({
      auth
    }),
    submitDeduplicationUpdate: ({
      fields,
      enabled
    }) => ({
      enabled,
      fields
    }),
    updateCrawlRules: crawlRules => ({
      crawlRules
    }),
    updateEntryPoints: entryPoints => ({
      entryPoints
    }),
    updateExtractionRules: extractionRules => ({
      extractionRules
    }),
    updateSitemaps: sitemaps => ({
      sitemaps
    })
  },
  connect: {
    actions: [_delete_crawler_domain_api_logic.DeleteCrawlerDomainApiLogic, ['apiSuccess as deleteApiSuccess', 'makeRequest as deleteMakeRequest']],
    values: [_delete_crawler_domain_api_logic.DeleteCrawlerDomainApiLogic, ['status as deleteStatus']]
  },
  listeners: ({
    actions,
    values
  }) => ({
    deleteApiSuccess: () => {
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      _kibana.KibanaLogic.values.navigateToUrl((0, _encode_path_params.generateEncodedPath)(_routes.SEARCH_INDEX_TAB_PATH, {
        indexName,
        tabId: _search_index.SearchIndexTabId.DOMAIN_MANAGEMENT
      }));
    },
    deleteDomain: async () => {
      const {
        domain
      } = values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      if (domain) {
        actions.deleteMakeRequest({
          domain,
          indexName
        });
      }
    },
    fetchDomainData: async ({
      domainId
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      try {
        const response = await http.get(`/internal/enterprise_search/indices/${indexName}/crawler/domains/${domainId}`);
        const domainData = (0, _utils.crawlerDomainServerToClient)(response);
        actions.receiveDomainData(domainData);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    submitAuthUpdate: async ({
      auth
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      const {
        domainId
      } = values;
      const payload = {
        auth
      };
      try {
        const response = await http.put(`/internal/enterprise_search/indices/${indexName}/crawler/domains/${domainId}`, {
          body: JSON.stringify(payload)
        });
        const domainData = (0, _utils.crawlerDomainServerToClient)(response);
        actions.receiveDomainData(domainData);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    submitDeduplicationUpdate: async ({
      fields,
      enabled
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      const {
        domainId
      } = values;
      const payload = {
        deduplication_enabled: enabled,
        deduplication_fields: fields
      };
      try {
        const response = await http.put(`/internal/enterprise_search/indices/${indexName}/crawler/domains/${domainId}`, {
          body: JSON.stringify(payload)
        });
        const domainData = (0, _utils.crawlerDomainServerToClient)(response);
        actions.receiveDomainData(domainData);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    }
  }),
  path: ['enterprise_search', 'crawler', 'crawler_domain_detail_logic'],
  reducers: ({
    props
  }) => ({
    domain: [null, {
      receiveDomainData: (_, {
        domain
      }) => domain,
      updateCrawlRules: (currentDomain, {
        crawlRules
      }) => currentDomain ? {
        ...currentDomain,
        crawlRules
      } : currentDomain,
      updateEntryPoints: (currentDomain, {
        entryPoints
      }) => currentDomain ? {
        ...currentDomain,
        entryPoints
      } : currentDomain,
      updateSitemaps: (currentDomain, {
        sitemaps
      }) => currentDomain ? {
        ...currentDomain,
        sitemaps
      } : currentDomain
    }],
    domainId: [props.domainId, {
      fetchDomainData: (_, {
        domainId
      }) => domainId
    }],
    getLoading: [true, {
      receiveDomainData: () => false
    }]
  }),
  selectors: ({
    selectors
  }) => ({
    deleteLoading: [() => [selectors.deleteStatus], deleteStatus => deleteStatus === _api.Status.LOADING],
    extractionRules: [() => [selectors.domain], domain => {
      var _domain$extractionRul;
      return (_domain$extractionRul = domain === null || domain === void 0 ? void 0 : domain.extractionRules) !== null && _domain$extractionRul !== void 0 ? _domain$extractionRul : [];
    }]
  })
});
exports.CrawlerDomainDetailLogic = CrawlerDomainDetailLogic;