"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrawlCustomSettingsFlyoutLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../../../shared/flash_messages");
var _http = require("../../../../../shared/http");
var _engine = require("../../../engine");
var _crawler_logic = require("../../crawler_logic");
var _utils = require("../../utils");
var _utils2 = require("../add_domain/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const filterSeedUrlsByDomainUrls = (seedUrls, domainUrls) => {
  const domainUrlMap = domainUrls.reduce((acc, domainUrl) => ({
    ...acc,
    [domainUrl]: true
  }), {});
  return seedUrls.filter(seedUrl => {
    const {
      domain
    } = (0, _utils2.extractDomainAndEntryPointFromUrl)(seedUrl);
    return !!domainUrlMap[domain];
  });
};
const CrawlCustomSettingsFlyoutLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'crawler', 'crawl_custom_settings_flyout'],
  actions: () => ({
    fetchDomainConfigData: true,
    hideFlyout: true,
    onRecieveDomainConfigData: domainConfigs => ({
      domainConfigs
    }),
    onSelectCustomEntryPointUrls: entryPointUrls => ({
      entryPointUrls
    }),
    onSelectCustomSitemapUrls: sitemapUrls => ({
      sitemapUrls
    }),
    onSelectDomainUrls: domainUrls => ({
      domainUrls
    }),
    onSelectEntryPointUrls: entryPointUrls => ({
      entryPointUrls
    }),
    onSelectMaxCrawlDepth: maxCrawlDepth => ({
      maxCrawlDepth
    }),
    onSelectSitemapUrls: sitemapUrls => ({
      sitemapUrls
    }),
    startCustomCrawl: true,
    toggleIncludeSitemapsInRobotsTxt: true,
    showFlyout: true
  }),
  reducers: () => ({
    customEntryPointUrls: [[], {
      showFlyout: () => [],
      onSelectCustomEntryPointUrls: (_, {
        entryPointUrls
      }) => entryPointUrls
    }],
    customSitemapUrls: [[], {
      showFlyout: () => [],
      onSelectCustomSitemapUrls: (_, {
        sitemapUrls
      }) => sitemapUrls
    }],
    domainConfigs: [[], {
      onRecieveDomainConfigData: (_, {
        domainConfigs
      }) => domainConfigs
    }],
    includeSitemapsInRobotsTxt: [true, {
      showFlyout: () => true,
      toggleIncludeSitemapsInRobotsTxt: includeSitemapsInRobotsTxt => !includeSitemapsInRobotsTxt
    }],
    isDataLoading: [true, {
      showFlyout: () => true,
      onRecieveDomainConfigData: () => false
    }],
    isFormSubmitting: [false, {
      [_crawler_logic.CrawlerLogic.actionTypes.startCrawl]: () => true,
      [_crawler_logic.CrawlerLogic.actionTypes.onStartCrawlRequestComplete]: () => false
    }],
    isFlyoutVisible: [false, {
      showFlyout: () => true,
      hideFlyout: () => false,
      [_crawler_logic.CrawlerLogic.actionTypes.onStartCrawlRequestComplete]: () => false
    }],
    maxCrawlDepth: [2, {
      showFlyout: () => 2,
      onSelectMaxCrawlDepth: (_, {
        maxCrawlDepth
      }) => maxCrawlDepth
    }],
    selectedDomainUrls: [[], {
      showFlyout: () => [],
      onSelectDomainUrls: (_, {
        domainUrls
      }) => domainUrls
    }],
    selectedEntryPointUrls: [[], {
      showFlyout: () => [],
      onSelectEntryPointUrls: (_, {
        entryPointUrls
      }) => entryPointUrls,
      onSelectDomainUrls: (entryPointUrls, {
        domainUrls
      }) => filterSeedUrlsByDomainUrls(entryPointUrls, domainUrls)
    }],
    selectedSitemapUrls: [[], {
      showFlyout: () => [],
      onSelectSitemapUrls: (_, {
        sitemapUrls
      }) => sitemapUrls,
      onSelectDomainUrls: (selectedSitemapUrls, {
        domainUrls
      }) => filterSeedUrlsByDomainUrls(selectedSitemapUrls, domainUrls)
    }]
  }),
  selectors: () => ({
    domainUrls: [selectors => [selectors.domainConfigs], domainConfigs => domainConfigs.map(domainConfig => domainConfig.name)],
    domainConfigMap: [selectors => [selectors.domainConfigs], domainConfigs => domainConfigs.reduce((acc, domainConfig) => ({
      ...acc,
      [domainConfig.name]: domainConfig
    }), {})],
    entryPointUrls: [selectors => [selectors.domainConfigMap, selectors.selectedDomainUrls], (domainConfigMap, selectedDomainUrls) => selectedDomainUrls.flatMap(selectedDomainUrl => domainConfigMap[selectedDomainUrl].seedUrls)],
    sitemapUrls: [selectors => [selectors.domainConfigMap, selectors.selectedDomainUrls], (domainConfigMap, selectedDomainUrls) => selectedDomainUrls.flatMap(selectedDomainUrl => domainConfigMap[selectedDomainUrl].sitemapUrls)]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    fetchDomainConfigData: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      let domainConfigs = [];
      let nextPage = 1;
      let totalPages = 1;
      let pageSize = 100;
      try {
        while (nextPage <= totalPages) {
          const {
            results,
            meta: {
              page
            }
          } = await http.get(`/internal/app_search/engines/${engineName}/crawler/domain_configs`, {
            query: {
              'page[current]': nextPage,
              'page[size]': pageSize
            }
          });
          domainConfigs = [...domainConfigs, ...results.map(_utils.domainConfigServerToClient)];
          nextPage = page.current + 1;
          totalPages = page.total_pages;
          pageSize = page.size;
        }
        actions.onRecieveDomainConfigData(domainConfigs);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    showFlyout: () => {
      actions.fetchDomainConfigData();
    },
    startCustomCrawl: () => {
      const overrides = {
        sitemap_discovery_disabled: !values.includeSitemapsInRobotsTxt,
        max_crawl_depth: values.maxCrawlDepth,
        domain_allowlist: values.selectedDomainUrls
      };
      const seedUrls = [...values.selectedEntryPointUrls, ...values.customEntryPointUrls];
      if (seedUrls.length > 0) {
        overrides.seed_urls = seedUrls;
      }
      const sitemapUrls = [...values.selectedSitemapUrls, ...values.customSitemapUrls];
      if (sitemapUrls.length > 0) {
        overrides.sitemap_urls = sitemapUrls;
      }
      _crawler_logic.CrawlerLogic.actions.startCrawl(overrides);
    }
  })
});
exports.CrawlCustomSettingsFlyoutLogic = CrawlCustomSettingsFlyoutLogic;