"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildSearchBody = buildSearchBody;
exports.useEsDocSearch = useEsDocSearch;
var _react = require("react");
var _rxjs = require("rxjs");
var _build_data_record = require("../utils/build_data_record");
var _types = require("../application/doc/types");
var _common = require("../../common");
var _use_discover_services = require("./use_discover_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Custom react hook for querying a single doc in ElasticSearch
 */
function useEsDocSearch({
  id,
  index,
  dataView,
  requestSource
}) {
  const [status, setStatus] = (0, _react.useState)(_types.ElasticRequestState.Loading);
  const [hit, setHit] = (0, _react.useState)(null);
  const {
    data,
    uiSettings
  } = (0, _use_discover_services.useDiscoverServices)();
  const useNewFieldsApi = (0, _react.useMemo)(() => !uiSettings.get(_common.SEARCH_FIELDS_FROM_SOURCE), [uiSettings]);
  const requestData = (0, _react.useCallback)(async () => {
    try {
      var _buildSearchBody, _hits$hits;
      const result = await (0, _rxjs.lastValueFrom)(data.search.search({
        params: {
          index: dataView.getIndexPattern(),
          body: (_buildSearchBody = buildSearchBody(id, index, dataView, useNewFieldsApi, requestSource)) === null || _buildSearchBody === void 0 ? void 0 : _buildSearchBody.body
        }
      }));
      const rawResponse = result.rawResponse;
      const hits = rawResponse.hits;
      if (hits !== null && hits !== void 0 && (_hits$hits = hits.hits) !== null && _hits$hits !== void 0 && _hits$hits[0]) {
        setStatus(_types.ElasticRequestState.Found);
        setHit((0, _build_data_record.buildDataTableRecord)(hits.hits[0], dataView));
      } else {
        setStatus(_types.ElasticRequestState.NotFound);
      }
    } catch (err) {
      if (err.savedObjectId) {
        setStatus(_types.ElasticRequestState.NotFoundDataView);
      } else if (err.status === 404) {
        setStatus(_types.ElasticRequestState.NotFound);
      } else {
        setStatus(_types.ElasticRequestState.Error);
      }
    }
  }, [id, index, dataView, data.search, useNewFieldsApi, requestSource]);
  (0, _react.useEffect)(() => {
    requestData();
  }, [requestData]);
  return [status, hit, requestData];
}

/**
 * helper function to build a query body for Elasticsearch
 * https://www.elastic.co/guide/en/elasticsearch/reference/current//query-dsl-ids-query.html
 */
function buildSearchBody(id, index, dataView, useNewFieldsApi, requestAllFields) {
  var _request$body;
  const computedFields = dataView.getComputedFields();
  const runtimeFields = computedFields.runtimeFields;
  const request = {
    body: {
      query: {
        bool: {
          filter: [{
            ids: {
              values: [id]
            }
          }, {
            term: {
              _index: index
            }
          }]
        }
      },
      stored_fields: computedFields.storedFields,
      script_fields: computedFields.scriptFields,
      version: true
    }
  };
  if (!request.body) {
    return undefined;
  }
  if (useNewFieldsApi) {
    // @ts-expect-error
    request.body.fields = [{
      field: '*',
      include_unmapped: 'true'
    }];
    request.body.runtime_mappings = runtimeFields ? runtimeFields : {};
    if (requestAllFields) {
      request.body._source = true;
    }
  } else {
    request.body._source = true;
  }
  request.body.fields = [...(((_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.fields) || []), ...(computedFields.docvalueFields || [])];
  return request;
}