"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useConfirmPersistencePrompt = void 0;
var _react = require("react");
var _uuid = require("uuid");
var _i18n = require("@kbn/i18n");
var _use_discover_services = require("./use_discover_services");
var _show_confirm_panel = require("./show_confirm_panel");
var _persist_saved_search = require("../application/main/utils/persist_saved_search");
var _update_filter_references = require("../application/main/utils/update_filter_references");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const useConfirmPersistencePrompt = stateContainer => {
  const services = (0, _use_discover_services.useDiscoverServices)();
  const persistDataView = (0, _react.useCallback)(async adHocDataView => {
    try {
      const persistedDataView = await services.dataViews.createAndSave({
        ...adHocDataView.toSpec(),
        id: (0, _uuid.v4)()
      });
      services.dataViews.clearInstanceCache(adHocDataView.id);
      (0, _update_filter_references.updateFiltersReferences)(adHocDataView, persistedDataView);
      stateContainer.actions.removeAdHocDataViewById(adHocDataView.id);
      await stateContainer.appState.update({
        index: persistedDataView.id
      }, true);
      const message = _i18n.i18n.translate('discover.dataViewPersist.message', {
        defaultMessage: "Saved '{dataViewName}'",
        values: {
          dataViewName: persistedDataView.getName()
        }
      });
      services.toastNotifications.addSuccess(message);
      return persistedDataView;
    } catch (error) {
      services.toastNotifications.addDanger({
        title: _i18n.i18n.translate('discover.dataViewPersistError.title', {
          defaultMessage: 'Unable to create data view'
        }),
        text: error.message
      });
      throw new Error(error);
    }
  }, [services.dataViews, services.toastNotifications, stateContainer]);
  const openConfirmSavePrompt = (0, _react.useCallback)(async dataView => {
    return new Promise(resolve => (0, _show_confirm_panel.showConfirmPanel)({
      onConfirm: () => persistDataView(dataView).then(createdDataView => resolve(createdDataView)).catch(() => resolve(undefined)),
      onCancel: () => resolve(undefined)
    }));
  }, [persistDataView]);
  const onUpdateSuccess = (0, _react.useCallback)(savedSearch => {
    services.toastNotifications.addSuccess({
      title: _i18n.i18n.translate('discover.notifications.updateSavedSearchTitle', {
        defaultMessage: `Search '{savedSearchTitle}' updated with saved data view`,
        values: {
          savedSearchTitle: savedSearch.title
        }
      }),
      'data-test-subj': 'updateSearchSuccess'
    });
  }, [services.toastNotifications]);
  const onUpdateError = (0, _react.useCallback)((error, savedSearch) => {
    services.toastNotifications.addDanger({
      title: _i18n.i18n.translate('discover.notifications.notUpdatedSavedSearchTitle', {
        defaultMessage: `Search '{savedSearchTitle}' was not updated with savedDataView.`,
        values: {
          savedSearchTitle: savedSearch.title
        }
      }),
      text: error.message
    });
  }, [services.toastNotifications]);
  const updateSavedSearch = (0, _react.useCallback)(({
    savedSearch,
    dataView,
    state
  }) => {
    return (0, _persist_saved_search.persistSavedSearch)(savedSearch, {
      dataView,
      onSuccess: () => onUpdateSuccess(savedSearch),
      onError: error => onUpdateError(error, savedSearch),
      state,
      saveOptions: {},
      services
    });
  }, [onUpdateError, onUpdateSuccess, services]);
  return {
    openConfirmSavePrompt,
    updateSavedSearch
  };
};
exports.useConfirmPersistencePrompt = useConfirmPersistencePrompt;