"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchSessionRestorationDataProvider = createSearchSessionRestorationDataProvider;
exports.getDiscoverStateContainer = getDiscoverStateContainer;
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/kibana-utils-plugin/public");
var _resolve_data_view = require("../utils/resolve_data_view");
var _discover_data_state_container = require("./discover_data_state_container");
var _discover_search_session = require("./discover_search_session");
var _common = require("../../../../common");
var _discover_app_state_container = require("./discover_app_state_container");
var _discover_internal_state_container = require("./discover_internal_state_container");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Builds and returns appState and globalState containers and helper functions
 * Used to sync URL with UI state
 */
function getDiscoverStateContainer({
  history,
  savedSearch,
  services
}) {
  const storeInSessionStorage = services.uiSettings.get('state:storeInSessionStorage');
  const toasts = services.core.notifications.toasts;
  const stateStorage = (0, _public.createKbnUrlStateStorage)({
    useHash: storeInSessionStorage,
    history,
    ...(toasts && (0, _public.withNotifyOnErrors)(toasts))
  });

  /**
   * Search session logic
   */
  const searchSessionManager = new _discover_search_session.DiscoverSearchSessionManager({
    history,
    session: services.data.search.session
  });
  /**
   * App State Container, synced with URL
   */
  const appStateContainer = (0, _discover_app_state_container.getDiscoverAppStateContainer)({
    stateStorage,
    savedSearch,
    services
  });
  const internalStateContainer = (0, _discover_internal_state_container.getInternalStateContainer)();
  const pauseAutoRefreshInterval = async () => {
    const state = stateStorage.get(_discover_app_state_container.GLOBAL_STATE_URL_KEY);
    if (state !== null && state !== void 0 && state.refreshInterval && !state.refreshInterval.pause) {
      await stateStorage.set(_discover_app_state_container.GLOBAL_STATE_URL_KEY, {
        ...state,
        refreshInterval: {
          ...(state === null || state === void 0 ? void 0 : state.refreshInterval),
          pause: true
        }
      }, {
        replace: true
      });
    }
  };
  const dataStateContainer = (0, _discover_data_state_container.getDataStateContainer)({
    services,
    searchSessionManager,
    getAppState: appStateContainer.getState,
    getSavedSearch: () => {
      // Simulating the behavior of the removed hook to always create a clean searchSource child that
      // we then use to add query, filters, etc., will be removed soon.
      return {
        ...savedSearch,
        searchSource: savedSearch.searchSource.createChild()
      };
    },
    appStateContainer
  });
  const setDataView = dataView => {
    internalStateContainer.transitions.setDataView(dataView);
  };
  const setAdHocDataViews = dataViews => internalStateContainer.transitions.setAdHocDataViews(dataViews);
  const appendAdHocDataViews = dataViews => internalStateContainer.transitions.appendAdHocDataViews(dataViews);
  const replaceAdHocDataViewWithId = (id, dataView) => internalStateContainer.transitions.replaceAdHocDataViewWithId(id, dataView);
  const removeAdHocDataViewById = id => internalStateContainer.transitions.removeAdHocDataViewById(id);
  const loadDataViewList = async () => {
    const dataViewList = await services.dataViews.getIdsWithTitle(true);
    internalStateContainer.transitions.setSavedDataViews(dataViewList);
  };
  const loadAndResolveDataView = async (id, actualSavedSearch) => {
    const nextDataViewData = await (0, _resolve_data_view.loadDataView)(services.dataViews, services.uiSettings, id);
    const nextDataView = (0, _resolve_data_view.resolveDataView)(nextDataViewData, actualSavedSearch.searchSource, services.toastNotifications);
    return {
      fallback: !nextDataViewData.stateValFound,
      dataView: nextDataView
    };
  };
  const initializeAndSync = () => appStateContainer.initAndSync(savedSearch);
  return {
    kbnUrlStateStorage: stateStorage,
    appState: appStateContainer,
    internalState: internalStateContainer,
    dataState: dataStateContainer,
    searchSessionManager,
    actions: {
      pauseAutoRefreshInterval,
      setDataView,
      loadAndResolveDataView,
      loadDataViewList,
      setAdHocDataViews,
      appendAdHocDataViews,
      replaceAdHocDataViewWithId,
      removeAdHocDataViewById,
      initializeAndSync
    }
  };
}
function createSearchSessionRestorationDataProvider(deps) {
  const getSavedSearchId = () => deps.getSavedSearch().id;
  return {
    getName: async () => {
      const savedSearch = deps.getSavedSearch();
      return savedSearch.id && savedSearch.title || _i18n.i18n.translate('discover.discoverDefaultSearchSessionName', {
        defaultMessage: 'Discover'
      });
    },
    getLocatorData: async () => {
      return {
        id: _common.DISCOVER_APP_LOCATOR,
        initialState: createUrlGeneratorState({
          ...deps,
          getSavedSearchId,
          shouldRestoreSearchSession: false
        }),
        restoreState: createUrlGeneratorState({
          ...deps,
          getSavedSearchId,
          shouldRestoreSearchSession: true
        })
      };
    }
  };
}
function createUrlGeneratorState({
  appStateContainer,
  data,
  getSavedSearchId,
  shouldRestoreSearchSession
}) {
  const appState = appStateContainer.get();
  return {
    filters: data.query.filterManager.getFilters(),
    dataViewId: appState.index,
    query: appState.query,
    savedSearchId: getSavedSearchId(),
    timeRange: shouldRestoreSearchSession ? data.query.timefilter.timefilter.getAbsoluteTime() : data.query.timefilter.timefilter.getTime(),
    searchSessionId: shouldRestoreSearchSession ? data.search.session.getSessionId() : undefined,
    columns: appState.columns,
    sort: appState.sort,
    savedQuery: appState.savedQuery,
    interval: appState.interval,
    refreshInterval: shouldRestoreSearchSession ? {
      pause: true,
      // force pause refresh interval when restoring a session
      value: 0
    } : undefined,
    useHash: false,
    viewMode: appState.viewMode,
    hideAggregatedPreview: appState.hideAggregatedPreview,
    breakdownField: appState.breakdownField
  };
}