"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDiscoverAppStateContainer = exports.GLOBAL_STATE_URL_KEY = exports.DiscoverAppStateProvider = exports.APP_STATE_URL_KEY = void 0;
exports.isEqualFilters = isEqualFilters;
exports.isEqualState = isEqualState;
exports.setState = setState;
exports.useAppStateSelector = void 0;
var _common = require("@kbn/kibana-utils-plugin/common");
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/kibana-utils-plugin/public");
var _lodash = require("lodash");
var _public2 = require("@kbn/data-plugin/public");
var _add_log = require("../../../utils/add_log");
var _get_valid_filters = require("../../../utils/get_valid_filters");
var _cleanup_url_state = require("../utils/cleanup_url_state");
var _get_state_defaults = require("../utils/get_state_defaults");
var _state_helpers = require("../../../utils/state_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const APP_STATE_URL_KEY = '_a';
exports.APP_STATE_URL_KEY = APP_STATE_URL_KEY;
const {
  Provider: DiscoverAppStateProvider,
  useSelector: useAppStateSelector
} = (0, _common.createStateContainerReactHelpers)();

/**
 * This is the app state container for Discover main, it's responsible for syncing state with the URL
 * @param stateStorage
 * @param savedSearch
 * @param services
 */
exports.useAppStateSelector = useAppStateSelector;
exports.DiscoverAppStateProvider = DiscoverAppStateProvider;
const getDiscoverAppStateContainer = ({
  stateStorage,
  savedSearch,
  services
}) => {
  let previousState = {};
  let initialState = getInitialState(stateStorage, savedSearch, services);
  const appStateContainer = (0, _common.createStateContainer)(initialState);
  const enhancedAppContainer = {
    ...appStateContainer,
    set: value => {
      if (value) {
        previousState = appStateContainer.getState();
        appStateContainer.set(value);
      }
    }
  };
  const hasChanged = () => {
    return !isEqualState(initialState, appStateContainer.getState());
  };
  const resetInitialState = () => {
    (0, _add_log.addLog)('[appState] reset initial state to the current state');
    initialState = appStateContainer.getState();
  };
  const replaceUrlState = async (newPartial = {}, merge = true) => {
    (0, _add_log.addLog)('[appState] replaceUrlState', {
      newPartial,
      merge
    });
    const state = merge ? {
      ...appStateContainer.getState(),
      ...newPartial
    } : newPartial;
    await stateStorage.set(APP_STATE_URL_KEY, state, {
      replace: true
    });
  };
  const startAppStateUrlSync = () => {
    (0, _add_log.addLog)('[appState] startAppStateUrlSync');
    return (0, _public.syncState)({
      storageKey: APP_STATE_URL_KEY,
      stateContainer: enhancedAppContainer,
      stateStorage
    });
  };
  const initializeAndSync = currentSavedSearch => {
    (0, _add_log.addLog)('[appState] initializeAndSync', currentSavedSearch);
    const dataView = currentSavedSearch.searchSource.getField('index');
    if (appStateContainer.getState().index !== dataView.id) {
      // used data view is different from the given by url/state which is invalid
      setState(appStateContainer, {
        index: dataView.id
      });
    }
    // sync initial app filters from state to filterManager
    const filters = appStateContainer.getState().filters || [];
    if (filters) {
      services.filterManager.setAppFilters((0, _lodash.cloneDeep)(filters));
    }
    const query = appStateContainer.getState().query;
    if (query) {
      services.data.query.queryString.setQuery(query);
    }
    const stopSyncingQueryAppStateWithStateContainer = (0, _public2.connectToQueryState)(services.data.query, appStateContainer, {
      filters: _esQuery.FilterStateStore.APP_STATE,
      query: true
    });

    // syncs `_g` portion of url with query services
    const {
      stop: stopSyncingGlobalStateWithUrl
    } = (0, _public2.syncGlobalQueryStateWithUrl)(services.data.query, stateStorage);

    // some filters may not be valid for this context, so update
    // the filter manager with a modified list of valid filters
    const currentFilters = services.filterManager.getFilters();
    const validFilters = (0, _get_valid_filters.getValidFilters)(dataView, currentFilters);
    if (!(0, _lodash.isEqual)(currentFilters, validFilters)) {
      services.filterManager.setFilters(validFilters);
    }
    const {
      start,
      stop
    } = startAppStateUrlSync();
    replaceUrlState({}).then(() => {
      start();
    });
    return () => {
      stopSyncingQueryAppStateWithStateContainer();
      stopSyncingGlobalStateWithUrl();
      stop();
    };
  };
  const resetWithSavedSearch = nextSavedSearch => {
    (0, _add_log.addLog)('[appState] reset to saved search', {
      nextSavedSearch
    });
    const nextAppState = getInitialState(stateStorage, nextSavedSearch, services);
    appStateContainer.set(nextAppState);
  };
  const update = (newPartial, replace = false) => {
    (0, _add_log.addLog)('[appState] update', {
      newPartial,
      replace
    });
    if (replace) {
      return replaceUrlState(newPartial);
    } else {
      previousState = {
        ...appStateContainer.getState()
      };
      setState(appStateContainer, newPartial);
    }
  };
  const getPrevious = () => previousState;
  return {
    ...enhancedAppContainer,
    getPrevious,
    hasChanged,
    initAndSync: initializeAndSync,
    resetWithSavedSearch,
    resetInitialState,
    replaceUrlState,
    syncState: startAppStateUrlSync,
    update
  };
};
exports.getDiscoverAppStateContainer = getDiscoverAppStateContainer;
const GLOBAL_STATE_URL_KEY = '_g';
exports.GLOBAL_STATE_URL_KEY = GLOBAL_STATE_URL_KEY;
function getInitialState(stateStorage, savedSearch, services) {
  const appStateFromUrl = (0, _cleanup_url_state.cleanupUrlState)(stateStorage.get(APP_STATE_URL_KEY));
  const defaultAppState = (0, _get_state_defaults.getStateDefaults)({
    savedSearch,
    services
  });
  return (0, _state_helpers.handleSourceColumnState)({
    ...defaultAppState,
    ...appStateFromUrl
  }, services.uiSettings);
}

/**
 * Helper function to merge a given new state with the existing state and to set the given state
 * container
 */
function setState(stateContainer, newState) {
  (0, _add_log.addLog)('[appstate] setState', {
    newState
  });
  const oldState = stateContainer.getState();
  const mergedState = {
    ...oldState,
    ...newState
  };
  if (!isEqualState(oldState, mergedState)) {
    stateContainer.set(mergedState);
  }
}

/**
 * Helper function to compare 2 different filter states
 */
function isEqualFilters(filtersA, filtersB) {
  if (!filtersA && !filtersB) {
    return true;
  } else if (!filtersA || !filtersB) {
    return false;
  }
  return (0, _esQuery.compareFilters)(filtersA, filtersB, _esQuery.COMPARE_ALL_OPTIONS);
}

/**
 * Helper function to compare 2 different state, is needed since comparing filters
 * works differently
 */
function isEqualState(stateA, stateB) {
  if (!stateA && !stateB) {
    return true;
  } else if (!stateA || !stateB) {
    return false;
  }
  const {
    filters: stateAFilters = [],
    ...stateAPartial
  } = stateA;
  const {
    filters: stateBFilters = [],
    ...stateBPartial
  } = stateB;
  return (0, _lodash.isEqual)(stateAPartial, stateBPartial) && isEqualFilters(stateAFilters, stateBFilters);
}