"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useTextBasedQueryLanguage = useTextBasedQueryLanguage;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
var _react = require("react");
var _public = require("@kbn/saved-search-plugin/public");
var _get_valid_view_mode = require("../utils/get_valid_view_mode");
var _types = require("../../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const MAX_NUM_OF_COLUMNS = 50;

/**
 * Hook to take care of text based query language state transformations when a new result is returned
 * If necessary this is setting displayed columns and selected data view
 */
function useTextBasedQueryLanguage({
  documents$,
  dataViews,
  stateContainer,
  savedSearch
}) {
  const prev = (0, _react.useRef)({
    columns: [],
    query: undefined
  });
  const cleanup = (0, _react.useCallback)(() => {
    if (prev.current.query) {
      // cleanup when it's not a text based query lang
      prev.current = {
        columns: [],
        query: undefined
      };
    }
  }, []);
  (0, _react.useEffect)(() => {
    const subscription = documents$.subscribe(async next => {
      var _next$result;
      const {
        query,
        recordRawType
      } = next;
      if (!query || next.fetchStatus === _types.FetchStatus.ERROR) {
        return;
      }
      const {
        columns: stateColumns,
        index,
        viewMode
      } = stateContainer.appState.getState();
      let nextColumns = [];
      const isTextBasedQueryLang = recordRawType === 'plain' && (0, _esQuery.isOfAggregateQueryType)(query) && 'sql' in query;
      const hasResults = ((_next$result = next.result) === null || _next$result === void 0 ? void 0 : _next$result.length) && next.fetchStatus === _types.FetchStatus.COMPLETE;
      const initialFetch = !prev.current.columns.length;
      if (isTextBasedQueryLang) {
        if (hasResults) {
          // check if state needs to contain column transformation due to a different columns in the resultset
          const firstRow = next.result[0];
          const firstRowColumns = Object.keys(firstRow.raw).slice(0, MAX_NUM_OF_COLUMNS);
          if (!(0, _lodash.isEqual)(firstRowColumns, prev.current.columns) && !(0, _lodash.isEqual)(query, prev.current.query)) {
            prev.current = {
              columns: firstRowColumns,
              query
            };
            nextColumns = firstRowColumns;
          }
          if (firstRowColumns && initialFetch) {
            prev.current = {
              columns: firstRowColumns,
              query
            };
          }
        }
        const indexPatternFromQuery = (0, _esQuery.getIndexPatternFromSQLQuery)(query.sql);
        const internalState = stateContainer.internalState.getState();
        const dataViewList = [...internalState.savedDataViews, ...internalState.adHocDataViews];
        let dataViewObj = dataViewList.find(({
          title
        }) => title === indexPatternFromQuery);

        // no dataview found but the index pattern is valid
        // create an adhoc instance instead
        if (!dataViewObj) {
          var _dataViewObj$fields$g, _dataViewObj$fields$g2;
          dataViewObj = await dataViews.create({
            title: indexPatternFromQuery
          });
          stateContainer.internalState.transitions.setAdHocDataViews([dataViewObj]);
          if (((_dataViewObj$fields$g = dataViewObj.fields.getByName('@timestamp')) === null || _dataViewObj$fields$g === void 0 ? void 0 : _dataViewObj$fields$g.type) === 'date') {
            dataViewObj.timeFieldName = '@timestamp';
          } else if ((_dataViewObj$fields$g2 = dataViewObj.fields.getByType('date')) !== null && _dataViewObj$fields$g2 !== void 0 && _dataViewObj$fields$g2.length) {
            const dateFields = dataViewObj.fields.getByType('date');
            dataViewObj.timeFieldName = dateFields[0].name;
          }
        }

        // don't set the columns on initial fetch, to prevent overwriting existing state
        const addColumnsToState = Boolean(nextColumns.length && (!initialFetch || !(stateColumns !== null && stateColumns !== void 0 && stateColumns.length)));
        // no need to reset index to state if it hasn't changed
        const addDataViewToState = Boolean(dataViewObj.id !== index);
        if (!addColumnsToState && !addDataViewToState) {
          return;
        }
        const nextState = {
          ...(addDataViewToState && {
            index: dataViewObj.id
          }),
          ...(addColumnsToState && {
            columns: nextColumns
          }),
          ...(viewMode === _public.VIEW_MODE.AGGREGATED_LEVEL && {
            viewMode: (0, _get_valid_view_mode.getValidViewMode)({
              viewMode,
              isTextBasedQueryMode: true
            })
          })
        };
        stateContainer.appState.replaceUrlState(nextState);
      } else {
        // cleanup for a "regular" query
        cleanup();
      }
    });
    return () => {
      // cleanup for e.g. when savedSearch is switched
      cleanup();
      subscription.unsubscribe();
    };
  }, [documents$, dataViews, stateContainer, savedSearch, cleanup]);
}