"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDiscoverState = useDiscoverState;
var _react = require("react");
var _esQuery = require("@kbn/es-query");
var _public = require("@kbn/data-views-plugin/public");
var _public2 = require("@kbn/saved-search-plugin/public");
var _build_state_subscribe = require("./utils/build_state_subscribe");
var _change_data_view = require("./utils/change_data_view");
var _use_search_session = require("./use_search_session");
var _types = require("../../types");
var _use_text_based_query_language = require("./use_text_based_query_language");
var _use_url_tracking = require("./use_url_tracking");
var _discover_state = require("../services/discover_state");
var _get_state_defaults = require("../utils/get_state_defaults");
var _restore_from_saved_search = require("../../../services/saved_searches/restore_from_saved_search");
var _use_adhoc_data_views = require("./use_adhoc_data_views");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function useDiscoverState({
  services,
  history,
  savedSearch
}) {
  const {
    data,
    filterManager,
    dataViews,
    toastNotifications,
    trackUiMetric
  } = services;
  const dataView = savedSearch.searchSource.getField('index');
  const searchSource = (0, _react.useMemo)(() => {
    savedSearch.searchSource.setField('index', dataView);
    return savedSearch.searchSource.createChild();
  }, [savedSearch, dataView]);
  const stateContainer = (0, _react.useMemo)(() => {
    const container = (0, _discover_state.getDiscoverStateContainer)({
      history,
      savedSearch,
      services
    });
    const nextDataView = savedSearch.searchSource.getField('index');
    container.actions.setDataView(nextDataView);
    if (!nextDataView.isPersisted()) {
      container.actions.appendAdHocDataViews(nextDataView);
    }
    return container;
  }, [history, savedSearch, services]);
  const {
    setUrlTracking
  } = (0, _use_url_tracking.useUrlTracking)(savedSearch, dataView);
  const {
    appState,
    searchSessionManager
  } = stateContainer;
  const [state, setState] = (0, _react.useState)(appState.getState());

  /**
   * Search session logic
   */
  (0, _use_search_session.useSearchSession)({
    services,
    stateContainer,
    savedSearch
  });

  /**
   * Adhoc data views functionality
   */
  const isTextBasedMode = (state === null || state === void 0 ? void 0 : state.query) && (0, _esQuery.isOfAggregateQueryType)(state === null || state === void 0 ? void 0 : state.query);
  const {
    persistDataView,
    updateAdHocDataViewId
  } = (0, _use_adhoc_data_views.useAdHocDataViews)({
    dataView,
    dataViews,
    stateContainer,
    savedSearch,
    setUrlTracking,
    filterManager,
    toastNotifications,
    trackUiMetric,
    isTextBasedMode
  });

  /**
   * Updates data views selector state
   */
  const updateDataViewList = (0, _react.useCallback)(async newAdHocDataViews => {
    await stateContainer.actions.loadDataViewList();
    stateContainer.actions.setAdHocDataViews(newAdHocDataViews);
  }, [stateContainer.actions]);

  /**
   * Data fetching logic
   */
  const {
    data$,
    refetch$,
    reset,
    inspectorAdapters,
    initialFetchStatus
  } = stateContainer.dataState;
  /**
   * State changes (data view, columns), when a text base query result is returned
   */
  (0, _use_text_based_query_language.useTextBasedQueryLanguage)({
    documents$: data$.documents$,
    dataViews,
    stateContainer,
    savedSearch
  });

  /**
   * Reset to display loading spinner when savedSearch is changing
   */
  (0, _react.useEffect)(() => reset(), [savedSearch.id, reset]);

  /**
   * Sync URL state with local app state on saved search load
   * or dataView / savedSearch switch
   */
  (0, _react.useEffect)(() => {
    const stopSync = stateContainer.actions.initializeAndSync(dataView, filterManager, data);
    setState(stateContainer.appState.getState());
    return () => stopSync();
  }, [stateContainer, filterManager, data, dataView]);

  /**
   * Data store subscribing to trigger fetching
   */
  (0, _react.useEffect)(() => {
    const stopSync = stateContainer.dataState.subscribe();
    return () => stopSync();
  }, [stateContainer]);

  /**
   * Track state changes that should trigger a fetch
   */
  (0, _react.useEffect)(() => {
    const unsubscribe = appState.subscribe((0, _build_state_subscribe.buildStateSubscribe)({
      stateContainer,
      savedSearch,
      setState
    }));
    return () => unsubscribe();
  }, [appState, savedSearch, services, stateContainer]);

  /**
   * Function triggered when user changes data view in the sidebar
   */
  const onChangeDataView = (0, _react.useCallback)(async id => {
    await (0, _change_data_view.changeDataView)(id, {
      services,
      discoverState: stateContainer,
      setUrlTracking
    });
    stateContainer.internalState.transitions.setExpandedDoc(undefined);
  }, [services, setUrlTracking, stateContainer]);

  /**
   * function to revert any changes to a given saved search
   */
  const resetSavedSearch = (0, _react.useCallback)(async id => {
    const newSavedSearch = await (0, _public2.getSavedSearch)(id, {
      search: services.data.search,
      savedObjectsClient: services.core.savedObjects.client,
      spaces: services.spaces,
      savedObjectsTagging: services.savedObjectsTagging
    });
    const newDataView = newSavedSearch.searchSource.getField('index') || dataView;
    newSavedSearch.searchSource.setField('index', newDataView);
    const newAppState = (0, _get_state_defaults.getStateDefaults)({
      savedSearch: newSavedSearch,
      services
    });
    (0, _restore_from_saved_search.restoreStateFromSavedSearch)({
      savedSearch: newSavedSearch,
      timefilter: services.timefilter
    });
    await stateContainer.appState.update(newAppState, true);
    setState(newAppState);
  }, [services, dataView, stateContainer]);

  /**
   * Function triggered when the user changes the query in the search bar
   */
  const onUpdateQuery = (0, _react.useCallback)((_payload, isUpdate) => {
    if (isUpdate === false) {
      searchSessionManager.removeSearchSessionIdFromURL({
        replace: false
      });
      refetch$.next(undefined);
    }
  }, [refetch$, searchSessionManager]);

  /**
   * Trigger data fetching on dataView or savedSearch changes
   */
  (0, _react.useEffect)(() => {
    if (dataView && initialFetchStatus === _types.FetchStatus.LOADING) {
      refetch$.next(undefined);
    }
  }, [initialFetchStatus, refetch$, dataView, savedSearch.id]);

  /**
   * We need to make sure the auto refresh interval is disabled for
   * non-time series data or rollups since we don't show the date picker
   */
  (0, _react.useEffect)(() => {
    if (dataView && (!dataView.isTimeBased() || dataView.type === _public.DataViewType.ROLLUP)) {
      stateContainer.actions.pauseAutoRefreshInterval();
    }
  }, [dataView, stateContainer]);
  return {
    inspectorAdapters,
    resetSavedSearch,
    onChangeDataView,
    onUpdateQuery,
    searchSource,
    stateContainer,
    persistDataView,
    updateAdHocDataViewId,
    updateDataViewList
  };
}