"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAdHocDataViews = void 0;
var _react = require("react");
var _uuid = require("uuid");
var _analytics = require("@kbn/analytics");
var _constants = require("../../../constants");
var _use_confirm_persistence_prompt = require("../../../hooks/use_confirm_persistence_prompt");
var _use_filters_validation = require("./use_filters_validation");
var _update_filter_references = require("../utils/update_filter_references");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const useAdHocDataViews = ({
  dataView,
  savedSearch,
  stateContainer,
  setUrlTracking,
  filterManager,
  dataViews,
  toastNotifications,
  trackUiMetric,
  isTextBasedMode
}) => {
  (0, _react.useEffect)(() => {
    if (!dataView.isPersisted()) {
      trackUiMetric === null || trackUiMetric === void 0 ? void 0 : trackUiMetric(_analytics.METRIC_TYPE.COUNT, _constants.ADHOC_DATA_VIEW_RENDER_EVENT);
    }
  }, [dataView, isTextBasedMode, trackUiMetric]);

  /**
   * Takes care of checking data view id references in filters
   */
  (0, _use_filters_validation.useFiltersValidation)({
    savedSearch,
    filterManager,
    toastNotifications
  });

  /**
   * When saving a saved search with an ad hoc data view, a new id needs to be generated for the data view
   * This is to prevent duplicate ids messing with our system
   */
  const updateAdHocDataViewId = (0, _react.useCallback)(async prevDataView => {
    const newDataView = await dataViews.create({
      ...prevDataView.toSpec(),
      id: (0, _uuid.v4)()
    });
    dataViews.clearInstanceCache(prevDataView.id);
    (0, _update_filter_references.updateFiltersReferences)(prevDataView, newDataView);
    stateContainer.actions.replaceAdHocDataViewWithId(prevDataView.id, newDataView);
    await stateContainer.appState.update({
      index: newDataView.id
    }, true);
    setUrlTracking(newDataView);
    return newDataView;
  }, [dataViews, setUrlTracking, stateContainer]);
  const {
    openConfirmSavePrompt,
    updateSavedSearch
  } = (0, _use_confirm_persistence_prompt.useConfirmPersistencePrompt)(stateContainer);
  const persistDataView = (0, _react.useCallback)(async () => {
    const currentDataView = savedSearch.searchSource.getField('index');
    if (!currentDataView || currentDataView.isPersisted()) {
      return currentDataView;
    }
    const createdDataView = await openConfirmSavePrompt(currentDataView);
    if (!createdDataView) {
      return currentDataView; // persistance cancelled
    }

    if (savedSearch.id) {
      // update saved search with saved data view
      const currentState = stateContainer.appState.getState();
      await updateSavedSearch({
        savedSearch,
        dataView: createdDataView,
        state: currentState
      });
    }
    return createdDataView;
  }, [stateContainer, openConfirmSavePrompt, savedSearch, updateSavedSearch]);
  return {
    persistDataView,
    updateAdHocDataViewId
  };
};
exports.useAdHocDataViews = useAdHocDataViews;