"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fieldCalculator = void 0;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const getFieldValues = (hits, field) => (0, _lodash.map)(hits, hit => hit.flattened[field.name]);
const getFieldValueCounts = params => {
  params = (0, _lodash.defaults)(params, {
    count: 5,
    grouped: false
  });
  if (params.field.type === 'geo_point' || params.field.type === 'geo_shape' || params.field.type === 'attachment') {
    return {
      error: _i18n.i18n.translate('discover.fieldChooser.fieldCalculator.analysisIsNotAvailableForGeoFieldsErrorMessage', {
        defaultMessage: 'Analysis is not available for geo fields.'
      })
    };
  }
  const allValues = getFieldValues(params.hits, params.field);
  const missing = _countMissing(allValues);
  try {
    const groups = _groupValues(allValues, params);
    const counts = (0, _lodash.sortBy)(groups, 'count').reverse().slice(0, params.count).map(bucket => ({
      value: bucket.value,
      count: bucket.count,
      percent: Number((bucket.count / (params.hits.length - missing) * 100).toFixed(1)),
      display: params.dataView.getFormatterForField(params.field).convert(bucket.value)
    }));
    if (params.hits.length - missing === 0) {
      return {
        error: _i18n.i18n.translate('discover.fieldChooser.fieldCalculator.fieldIsNotPresentInDocumentsErrorMessage', {
          defaultMessage: 'This field is present in your Elasticsearch mapping but not in the {hitsLength} documents shown in the doc table. You may still be able to visualize or search on it.',
          values: {
            hitsLength: params.hits.length
          }
        })
      };
    }
    return {
      total: params.hits.length,
      exists: params.hits.length - missing,
      missing,
      buckets: counts
    };
  } catch (e) {
    return {
      error: e.message
    };
  }
};

// returns a count of fields in the array that are undefined or null
const _countMissing = array => array.length - (0, _lodash.without)(array, undefined, null).length;
const _groupValues = (allValues, params) => {
  const groups = {};
  let k;
  allValues.forEach(value => {
    if ((0, _lodash.isObject)(value) && !Array.isArray(value)) {
      throw new Error(_i18n.i18n.translate('discover.fieldChooser.fieldCalculator.analysisIsNotAvailableForObjectFieldsErrorMessage', {
        defaultMessage: 'Analysis is not available for object fields.'
      }));
    }
    if (Array.isArray(value) && !params.grouped) {
      k = value;
    } else {
      k = value == null ? undefined : [value];
    }
    (0, _lodash.each)(k, key => {
      if (groups.hasOwnProperty(key)) {
        groups[key].count++;
      } else {
        groups[key] = {
          value: params.grouped ? value : key,
          count: 1
        };
      }
    });
  });
  return groups;
};
const fieldCalculator = {
  _groupValues,
  _countMissing,
  getFieldValues,
  getFieldValueCounts
};
exports.fieldCalculator = fieldCalculator;