"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.rateLimitingForkJoin = rateLimitingForkJoin;
exports.useOverallStats = useOverallStats;
var _react = require("react");
var _rxjs = require("rxjs");
var _operators = require("rxjs/operators");
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _kibana_context = require("../../kibana_context");
var _overall_stats = require("../search_strategy/requests/overall_stats");
var _index_data_visualizer_view = require("../components/index_data_visualizer_view/index_data_visualizer_view");
var _field_stats = require("../../../../common/types/field_stats");
var _get_document_stats = require("../search_strategy/requests/get_document_stats");
var _progress_utils = require("../progress_utils");
var _index_data_visualizer_viewer = require("../constants/index_data_visualizer_viewer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper function to run forkJoin
 * with restrictions on how many input observables can be subscribed to concurrently
 */
function rateLimitingForkJoin(observables, maxConcurrentRequests = _index_data_visualizer_viewer.MAX_CONCURRENT_REQUESTS) {
  return (0, _rxjs.from)(observables).pipe((0, _operators.mergeMap)((observable, index) => observable.pipe((0, _operators.last)(), (0, _operators.map)(value => ({
    index,
    value
  }))), maxConcurrentRequests), (0, _operators.toArray)(), (0, _operators.map)(indexedObservables => indexedObservables.sort((l, r) => l.index - r.index).map(obs => obs.value)));
}
function displayError(toastNotifications, index, err) {
  if (err.statusCode === 500) {
    var _err$error;
    toastNotifications.addError(err, {
      title: _i18n.i18n.translate('xpack.dataVisualizer.index.dataLoader.internalServerErrorMessage', {
        defaultMessage: 'Error loading data in index {index}. {message}. ' + 'The request may have timed out. Try using a smaller sample size or narrowing the time range.',
        values: {
          index,
          message: (_err$error = err.error) !== null && _err$error !== void 0 ? _err$error : err.message
        }
      })
    });
  } else {
    var _err$error2;
    toastNotifications.addError(err, {
      title: _i18n.i18n.translate('xpack.dataVisualizer.index.errorLoadingDataMessage', {
        defaultMessage: 'Error loading data in index {index}. {message}.',
        values: {
          index,
          message: (_err$error2 = err.error) !== null && _err$error2 !== void 0 ? _err$error2 : err.message
        }
      })
    });
  }
}
function useOverallStats(searchStrategyParams, lastRefresh, probability) {
  const {
    services: {
      data,
      notifications: {
        toasts
      }
    }
  } = (0, _kibana_context.useDataVisualizerKibana)();
  const [stats, setOverallStats] = (0, _react.useState)((0, _index_data_visualizer_view.getDefaultPageState)().overallStats);
  const [fetchState, setFetchState] = (0, _react.useReducer)((0, _progress_utils.getReducer)(), (0, _progress_utils.getInitialProgress)());
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const searchSubscription$ = (0, _react.useRef)();
  const startFetch = (0, _react.useCallback)(async () => {
    try {
      var _searchSubscription$$;
      (_searchSubscription$$ = searchSubscription$.current) === null || _searchSubscription$$ === void 0 ? void 0 : _searchSubscription$$.unsubscribe();
      abortCtrl.current.abort();
      abortCtrl.current = new AbortController();
      if (!searchStrategyParams || lastRefresh === 0) return;
      setFetchState({
        ...(0, _progress_utils.getInitialProgress)(),
        isRunning: true,
        error: undefined
      });
      const {
        aggregatableFields,
        nonAggregatableFields,
        index,
        searchQuery,
        timeFieldName,
        earliest,
        latest,
        runtimeFieldMap,
        samplingOption,
        sessionId,
        embeddableExecutionContext
      } = searchStrategyParams;
      const searchOptions = {
        abortSignal: abortCtrl.current.signal,
        sessionId,
        ...(embeddableExecutionContext ? {
          executionContext: embeddableExecutionContext
        } : {})
      };
      const documentCountStats = await (0, _get_document_stats.getDocumentCountStats)(data.search, searchStrategyParams, searchOptions, samplingOption.seed, probability);
      const nonAggregatableFieldsObs = nonAggregatableFields.map(fieldName => data.search.search({
        params: (0, _overall_stats.checkNonAggregatableFieldExistsRequest)(index, searchQuery, fieldName, timeFieldName, earliest, latest, runtimeFieldMap)
      }, searchOptions).pipe((0, _operators.map)(resp => {
        return {
          ...resp,
          rawResponse: {
            ...resp.rawResponse,
            fieldName
          }
        };
      })));

      // Have to divide into smaller requests to avoid 413 payload too large
      const aggregatableFieldsChunks = (0, _lodash.chunk)(aggregatableFields, 30);
      if ((0, _field_stats.isRandomSamplingOption)(samplingOption)) {
        var _documentCountStats$p;
        samplingOption.probability = (_documentCountStats$p = documentCountStats.probability) !== null && _documentCountStats$p !== void 0 ? _documentCountStats$p : 1;
      }
      const aggregatableOverallStatsObs = aggregatableFieldsChunks.map(aggregatableFieldsChunk => data.search.search({
        params: (0, _overall_stats.checkAggregatableFieldsExistRequest)(index, searchQuery, aggregatableFieldsChunk, samplingOption, timeFieldName, earliest, latest, undefined, runtimeFieldMap)
      }, searchOptions).pipe((0, _operators.map)(resp => {
        return {
          ...resp,
          aggregatableFields: aggregatableFieldsChunk
        };
      })));
      const sub = rateLimitingForkJoin([...aggregatableOverallStatsObs, ...nonAggregatableFieldsObs], _index_data_visualizer_viewer.MAX_CONCURRENT_REQUESTS);
      searchSubscription$.current = sub.subscribe({
        next: value => {
          var _documentCountStats$t;
          const aggregatableOverallStatsResp = [];
          const nonAggregatableOverallStatsResp = [];
          value.forEach((resp, idx) => {
            if ((0, _overall_stats.isAggregatableFieldOverallStats)(resp)) {
              aggregatableOverallStatsResp.push(resp);
            }
            if ((0, _overall_stats.isNonAggregatableFieldOverallStats)(resp)) {
              nonAggregatableOverallStatsResp.push(resp);
            }
          });
          const totalCount = (_documentCountStats$t = documentCountStats === null || documentCountStats === void 0 ? void 0 : documentCountStats.totalCount) !== null && _documentCountStats$t !== void 0 ? _documentCountStats$t : 0;
          const aggregatableOverallStats = (0, _overall_stats.processAggregatableFieldsExistResponse)(aggregatableOverallStatsResp, aggregatableFields);
          const nonAggregatableOverallStats = (0, _overall_stats.processNonAggregatableFieldsExistResponse)(nonAggregatableOverallStatsResp, nonAggregatableFields);
          setOverallStats({
            documentCountStats,
            ...nonAggregatableOverallStats,
            ...aggregatableOverallStats,
            totalCount
          });
        },
        error: error => {
          displayError(toasts, searchStrategyParams.index, (0, _mlErrorUtils.extractErrorProperties)(error));
          setFetchState({
            isRunning: false,
            error
          });
        },
        complete: () => {
          setFetchState({
            loaded: 100,
            isRunning: false
          });
        }
      });
    } catch (error) {
      // An `AbortError` gets triggered when a user cancels a request by navigating away, we need to ignore these errors.
      if (error.name !== 'AbortError') {
        displayError(toasts, searchStrategyParams.index, (0, _mlErrorUtils.extractErrorProperties)(error));
      }
    }
  }, [data.search, searchStrategyParams, toasts, lastRefresh, probability]);
  const cancelFetch = (0, _react.useCallback)(() => {
    var _searchSubscription$$2;
    (_searchSubscription$$2 = searchSubscription$.current) === null || _searchSubscription$$2 === void 0 ? void 0 : _searchSubscription$$2.unsubscribe();
    searchSubscription$.current = undefined;
    abortCtrl.current.abort();
  }, []);

  // auto-update
  (0, _react.useEffect)(() => {
    startFetch();
  }, [startFetch]);
  (0, _react.useEffect)(() => {
    return cancelFetch;
  }, [cancelFetch]);
  return (0, _react.useMemo)(() => ({
    progress: fetchState,
    overallStats: stats
  }), [stats, fetchState]);
}