"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addCombinedFieldsToMappings = addCombinedFieldsToMappings;
exports.addCombinedFieldsToPipeline = addCombinedFieldsToPipeline;
exports.createGeoPointCombinedField = createGeoPointCombinedField;
exports.getDefaultCombinedFields = getDefaultCombinedFields;
exports.getFieldNames = getFieldNames;
exports.getNameCollisionMsg = getNameCollisionMsg;
exports.isWithinLatRange = isWithinLatRange;
exports.isWithinLonRange = isWithinLonRange;
exports.removeCombinedFieldsFromMappings = removeCombinedFieldsFromMappings;
exports.removeCombinedFieldsFromPipeline = removeCombinedFieldsFromPipeline;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _uuid = require("uuid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const COMMON_LAT_NAMES = ['latitude', 'lat'];
const COMMON_LON_NAMES = ['longitude', 'long', 'lon'];
function getDefaultCombinedFields(results) {
  const combinedFields = [];
  const geoPointField = getGeoPointField(results);
  if (geoPointField) {
    combinedFields.push(geoPointField);
  }
  return combinedFields;
}
function addCombinedFieldsToMappings(mappings, combinedFields) {
  const updatedMappings = {
    ...mappings
  };
  combinedFields.forEach(combinedField => {
    updatedMappings.properties[combinedField.combinedFieldName] = {
      type: combinedField.mappingType
    };
  });
  return updatedMappings;
}
function removeCombinedFieldsFromMappings(mappings, combinedFields) {
  const updatedMappings = {
    ...mappings
  };
  combinedFields.forEach(combinedField => {
    delete updatedMappings.properties[combinedField.combinedFieldName];
  });
  return updatedMappings;
}
function addCombinedFieldsToPipeline(pipeline, combinedFields) {
  const updatedPipeline = (0, _lodash.cloneDeep)(pipeline);
  combinedFields.forEach(combinedField => {
    updatedPipeline.processors.push({
      set: {
        field: combinedField.combinedFieldName,
        value: combinedField.fieldNames.map(fieldName => {
          return `{{${fieldName}}}`;
        }).join(combinedField.delimiter)
      }
    });
  });
  return updatedPipeline;
}
function removeCombinedFieldsFromPipeline(pipeline, combinedFields) {
  return {
    ...pipeline,
    processors: pipeline.processors.filter(processor => {
      return 'set' in processor ? !combinedFields.some(combinedField => {
        return processor.set.field === combinedField.combinedFieldName;
      }) : true;
    })
  };
}
function isWithinLatRange(fieldName, fieldStats) {
  return fieldName in fieldStats && 'max_value' in fieldStats[fieldName] && fieldStats[fieldName].max_value <= 90 && 'min_value' in fieldStats[fieldName] && fieldStats[fieldName].min_value >= -90;
}
function isWithinLonRange(fieldName, fieldStats) {
  return fieldName in fieldStats && 'max_value' in fieldStats[fieldName] && fieldStats[fieldName].max_value <= 180 && 'min_value' in fieldStats[fieldName] && fieldStats[fieldName].min_value >= -180;
}
function createGeoPointCombinedField(latField, lonField, geoPointField) {
  return {
    mappingType: 'geo_point',
    delimiter: ',',
    combinedFieldName: geoPointField,
    fieldNames: [latField, lonField]
  };
}
function getNameCollisionMsg(name) {
  return _i18n.i18n.translate('xpack.dataVisualizer.nameCollisionMsg', {
    defaultMessage: '"{name}" already exists, please provide a unique name',
    values: {
      name
    }
  });
}
function getFieldNames(results) {
  return results.column_names !== undefined ? results.column_names : Object.keys(results.field_stats);
}
function getGeoPointField(results) {
  const fieldNames = getFieldNames(results);
  const latField = fieldNames.find(columnName => {
    return COMMON_LAT_NAMES.includes(columnName.toLowerCase()) && isWithinLatRange(columnName, results.field_stats);
  });
  const lonField = fieldNames.find(columnName => {
    return COMMON_LON_NAMES.includes(columnName.toLowerCase()) && isWithinLonRange(columnName, results.field_stats);
  });
  if (!latField || !lonField) {
    return null;
  }
  const combinedFieldNames = ['location', 'point_location', `${latField}_${lonField}`, `location_${(0, _uuid.v4)()}`];
  // Use first combinedFieldNames that does not have a naming collision
  const geoPointField = combinedFieldNames.find(name => {
    return !fieldNames.includes(name);
  });
  return geoPointField ? createGeoPointCombinedField(latField, lonField, geoPointField) : null;
}