"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataViewField = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _fieldTypes = require("@kbn/field-types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Data view field class
 * @public
 */
class DataViewField {
  // not writable or serialized
  /**
   * Kbn field type, used mainly for formattering.
   */

  /**
   * DataView constructor
   * @constructor
   * @param spec Configuration for the field
   */
  constructor(spec) {
    (0, _defineProperty2.default)(this, "spec", void 0);
    (0, _defineProperty2.default)(this, "kbnFieldType", void 0);
    this.spec = {
      ...spec,
      type: spec.name === '_source' ? '_source' : spec.type
    };
    this.kbnFieldType = (0, _fieldTypes.getKbnFieldType)(spec.type);
  }

  // writable attrs
  /**
   * Count is used for field popularity in discover.
   */
  get count() {
    return this.spec.count || 0;
  }

  /**
   * Set count, which is used for field popularity in discover.
   * @param count count number
   */
  set count(count) {
    this.spec.count = count;
  }

  /**
   * Returns runtime field definition or undefined if field is not runtime field.
   */

  get runtimeField() {
    return this.spec.runtimeField;
  }

  /**
   * Sets runtime field definition or unsets if undefined is provided.
   * @param runtimeField runtime field definition
   */
  set runtimeField(runtimeField) {
    this.spec.runtimeField = runtimeField;
  }

  /**
   * Script field code
   */
  get script() {
    return this.spec.script;
  }

  /**
   * Sets scripted field painless code
   * @param script Painless code
   */
  set script(script) {
    this.spec.script = script;
  }

  /**
   * Script field language
   */
  get lang() {
    return this.spec.lang;
  }

  /**
   * Sets scripted field langauge.
   * @param lang Scripted field language
   */
  set lang(lang) {
    this.spec.lang = lang;
  }

  /**
   * Returns custom label if set, otherwise undefined.
   */

  get customLabel() {
    return this.spec.customLabel;
  }

  /**
   * Sets custom label for field, or unsets if passed undefined.
   * @param customLabel custom label value
   */
  set customLabel(customLabel) {
    this.spec.customLabel = customLabel;
  }

  /**
   * Description of field type conflicts across different indices in the same index pattern.
   */
  get conflictDescriptions() {
    return this.spec.conflictDescriptions;
  }

  /**
   * Sets conflict descriptions for field.
   * @param conflictDescriptions conflict descriptions
   */

  set conflictDescriptions(conflictDescriptions) {
    this.spec.conflictDescriptions = conflictDescriptions;
  }

  // read only attrs

  /**
   * Get field name
   */
  get name() {
    return this.spec.name;
  }

  /**
   * Gets display name, calcualted based on name, custom label and shortDotsEnable.
   */

  get displayName() {
    return this.spec.customLabel ? this.spec.customLabel : this.spec.shortDotsEnable ? (0, _utils.shortenDottedString)(this.spec.name) : this.spec.name;
  }

  /**
   * Gets field type
   */
  get type() {
    return this.spec.type;
  }

  /**
   * Gets ES types as string array
   */

  get esTypes() {
    return this.spec.esTypes;
  }

  /**
   * Returns true if scripted field
   */

  get scripted() {
    return !!this.spec.scripted;
  }

  /**
   * Returns true if field is searchable
   */

  get searchable() {
    return !!(this.spec.searchable || this.scripted);
  }

  /**
   * Returns true if field is aggregatable
   */

  get aggregatable() {
    return !!(this.spec.aggregatable || this.scripted);
  }

  /**
   * returns true if field is a TSDB dimension field
   */
  get timeSeriesDimension() {
    return this.spec.timeSeriesDimension || false;
  }

  /**
   * returns type of TSDB metric or undefined
   */
  get timeSeriesMetric() {
    return this.spec.timeSeriesMetric;
  }

  /**
   * returns list of alloeed fixed intervals
   */
  get fixedInterval() {
    return this.spec.fixedInterval;
  }

  /**
   * returns true if the field is of rolled up type
   */
  get isRolledUpField() {
    var _this$esTypes;
    return (_this$esTypes = this.esTypes) === null || _this$esTypes === void 0 ? void 0 : _this$esTypes.includes('aggregate_metric_double');
  }

  /**
   * return list of allowed time zones
   */
  get timeZone() {
    return this.spec.timeZone;
  }
  /**
   * Returns true if field is available via doc values
   */

  get readFromDocValues() {
    return !!(this.spec.readFromDocValues && !this.scripted);
  }

  /**
   * Returns field subtype, multi, nested, or undefined if neither
   */

  get subType() {
    return this.spec.subType;
  }

  /**
   * Is the field part of the index mapping?
   */
  get isMapped() {
    return this.spec.isMapped;
  }

  /**
   * Returns true if runtime field defined on data view
   */

  get isRuntimeField() {
    return !this.isMapped && this.runtimeField !== undefined;
  }

  // not writable, not serialized

  /**
   * Returns true if field is sortable
   */
  get sortable() {
    return this.name === '_score' || (this.spec.indexed || this.aggregatable) && this.kbnFieldType.sortable;
  }

  /**
   * Returns true if field is filterable
   */

  get filterable() {
    return this.name === '_id' || this.scripted || (this.spec.indexed || this.searchable) && this.kbnFieldType.filterable;
  }

  /**
   * Returns true if field is visualizable
   */

  get visualizable() {
    const notVisualizableFieldTypes = [_fieldTypes.KBN_FIELD_TYPES.UNKNOWN, _fieldTypes.KBN_FIELD_TYPES.CONFLICT];
    return this.aggregatable && !notVisualizableFieldTypes.includes(this.spec.type);
  }

  /**
   * Returns true if field is subtype nested
   */
  isSubtypeNested() {
    return (0, _utils.isDataViewFieldSubtypeNested)(this);
  }

  /**
   * Returns true if field is subtype multi
   */

  isSubtypeMulti() {
    return (0, _utils.isDataViewFieldSubtypeMulti)(this);
  }

  /**
   * Returns subtype nested data if exists
   */

  getSubtypeNested() {
    return (0, _utils.getDataViewFieldSubtypeNested)(this);
  }

  /**
   * Returns subtype multi data if exists
   */

  getSubtypeMulti() {
    return (0, _utils.getDataViewFieldSubtypeMulti)(this);
  }

  /**
   * Deletes count value. Popularity as used by discover
   */

  deleteCount() {
    delete this.spec.count;
  }

  /**
   * JSON version of field
   */
  toJSON() {
    return {
      count: this.count,
      script: this.script,
      lang: this.lang,
      conflictDescriptions: this.conflictDescriptions,
      name: this.name,
      type: this.type,
      esTypes: this.esTypes,
      scripted: this.scripted,
      searchable: this.searchable,
      aggregatable: this.aggregatable,
      readFromDocValues: this.readFromDocValues,
      subType: this.subType,
      customLabel: this.customLabel
    };
  }

  /**
   * Get field in serialized form - fieldspec.
   * @param config provide a method to get a field formatter
   * @returns field in serialized form - field spec
   */
  toSpec(config = {}) {
    const {
      getFormatterForField
    } = config;
    const spec = {
      count: this.count,
      script: this.script,
      lang: this.lang,
      conflictDescriptions: this.conflictDescriptions,
      name: this.name,
      type: this.type,
      esTypes: this.esTypes,
      scripted: this.scripted,
      searchable: this.searchable,
      aggregatable: this.aggregatable,
      readFromDocValues: this.readFromDocValues,
      subType: this.subType,
      format: getFormatterForField ? getFormatterForField(this).toJSON() : undefined,
      customLabel: this.customLabel,
      shortDotsEnable: this.spec.shortDotsEnable,
      runtimeField: this.runtimeField,
      isMapped: this.isMapped,
      timeSeriesDimension: this.spec.timeSeriesDimension,
      timeSeriesMetric: this.spec.timeSeriesMetric,
      timeZone: this.spec.timeZone,
      fixedInterval: this.spec.fixedInterval
    };

    // Filter undefined values from the spec
    return Object.fromEntries(Object.entries(spec).filter(([, v]) => typeof v !== 'undefined'));
  }

  /**
   * Returns true if composite runtime field
   */

  isRuntimeCompositeSubField() {
    var _this$runtimeField;
    return ((_this$runtimeField = this.runtimeField) === null || _this$runtimeField === void 0 ? void 0 : _this$runtimeField.type) === 'composite';
  }
}
exports.DataViewField = DataViewField;