"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getStatusText = exports.StatusIndicator = void 0;
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _react = _interopRequireDefault(require("react"));
var _common = require("../../../../../common");
var _date_string = require("../lib/date_string");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

// Shared helper function
const getStatusText = statusType => {
  switch (statusType) {
    case _common.SearchSessionStatus.IN_PROGRESS:
      return _i18n.i18n.translate('data.mgmt.searchSessions.status.label.inProgress', {
        defaultMessage: 'In progress'
      });
    case _common.SearchSessionStatus.EXPIRED:
      return _i18n.i18n.translate('data.mgmt.searchSessions.status.label.expired', {
        defaultMessage: 'Expired'
      });
    case _common.SearchSessionStatus.CANCELLED:
      return _i18n.i18n.translate('data.mgmt.searchSessions.status.label.cancelled', {
        defaultMessage: 'Cancelled'
      });
    case _common.SearchSessionStatus.COMPLETE:
      return _i18n.i18n.translate('data.mgmt.searchSessions.status.label.complete', {
        defaultMessage: 'Complete'
      });
    case _common.SearchSessionStatus.ERROR:
      return _i18n.i18n.translate('data.mgmt.searchSessions.status.label.error', {
        defaultMessage: 'Error'
      });
    default:
      // eslint-disable-next-line no-console
      console.error(`Unknown status ${statusType}`);
      return statusType;
  }
};
exports.getStatusText = getStatusText;
// Get the fields needed to show each status type
// can throw errors around date conversions
const getStatusAttributes = ({
  now,
  session,
  timezone
}) => {
  let expireDate;
  if (session.expires) {
    expireDate = (0, _date_string.dateString)(session.expires, timezone);
  } else {
    expireDate = _i18n.i18n.translate('data.mgmt.searchSessions.status.expireDateUnknown', {
      defaultMessage: 'unknown'
    });
  }
  switch (session.status) {
    case _common.SearchSessionStatus.IN_PROGRESS:
      try {
        return {
          textColor: 'default',
          icon: /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, null),
          label: /*#__PURE__*/_react.default.createElement(_.TableText, null, getStatusText(session.status)),
          toolTipContent: _i18n.i18n.translate('data.mgmt.searchSessions.status.message.createdOn', {
            defaultMessage: 'Expires on {expireDate}',
            values: {
              expireDate
            }
          })
        };
      } catch (err) {
        // eslint-disable-next-line no-console
        console.error(err);
        throw new Error(`Could not instantiate a createdDate object from: ${session.created}`);
      }
    case _common.SearchSessionStatus.EXPIRED:
      try {
        const toolTipContent = _i18n.i18n.translate('data.mgmt.searchSessions.status.message.expiredOn', {
          defaultMessage: 'Expired on {expireDate}',
          values: {
            expireDate
          }
        });
        return {
          icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
            color: "#9AA",
            type: "clock"
          }),
          label: /*#__PURE__*/_react.default.createElement(_.TableText, null, getStatusText(session.status)),
          toolTipContent
        };
      } catch (err) {
        // eslint-disable-next-line no-console
        console.error(err);
        throw new Error(`Could not instantiate an expiration Date object from: ${session.expires}`);
      }
    case _common.SearchSessionStatus.CANCELLED:
      return {
        icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
          color: "#9AA",
          type: "error"
        }),
        label: /*#__PURE__*/_react.default.createElement(_.TableText, null, getStatusText(session.status)),
        toolTipContent: _i18n.i18n.translate('data.mgmt.searchSessions.status.message.cancelled', {
          defaultMessage: 'Cancelled by user'
        })
      };
    case _common.SearchSessionStatus.ERROR:
      return {
        textColor: 'danger',
        icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
          color: "danger",
          type: "error"
        }),
        label: /*#__PURE__*/_react.default.createElement(_.TableText, null, getStatusText(session.status)),
        toolTipContent: session.errors && session.errors.length > 0 ? _i18n.i18n.translate('data.mgmt.searchSessions.status.message.error', {
          defaultMessage: 'One or more searches failed to complete. Use the "Inspect" action to see the underlying errors.'
        }) : _i18n.i18n.translate('data.mgmt.searchSessions.status.message.unknownError', {
          defaultMessage: 'Unknown error'
        })
      };
    case _common.SearchSessionStatus.COMPLETE:
      try {
        const toolTipContent = _i18n.i18n.translate('data.mgmt.searchSessions.status.expiresOn', {
          defaultMessage: 'Expires on {expireDate}',
          values: {
            expireDate
          }
        });
        return {
          textColor: 'success',
          icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
            color: "success",
            type: "checkInCircleFilled"
          }),
          label: /*#__PURE__*/_react.default.createElement(_.TableText, null, getStatusText(session.status)),
          toolTipContent
        };
      } catch (err) {
        // eslint-disable-next-line no-console
        console.error(err);
        throw new Error(`Could not instantiate an expiration Date object for completed session from: ${session.expires}`);
      }

      // Error was thrown
      return null;
    default:
      throw new Error(`Unknown status: ${session.status}`);
  }
};
const StatusIndicator = props => {
  try {
    const statusDef = getStatusAttributes(props);
    const {
      session
    } = props;
    if (statusDef) {
      const {
        toolTipContent
      } = statusDef;
      let icon = statusDef.icon;
      let label = statusDef.label;
      if (icon && toolTipContent) {
        icon = /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
          content: toolTipContent
        }, icon);
      }
      if (toolTipContent) {
        label = /*#__PURE__*/_react.default.createElement(_eui.EuiToolTip, {
          content: toolTipContent
        }, /*#__PURE__*/_react.default.createElement(_.TableText, {
          "data-test-subj": `sessionManagementStatusTooltip`
        }, statusDef.label));
      }
      return /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
        gutterSize: "s",
        alignItems: "center"
      }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
        grow: false
      }, icon), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
        grow: false
      }, /*#__PURE__*/_react.default.createElement(_.TableText, {
        color: statusDef.textColor,
        "data-test-subj": `sessionManagementStatusLabel`,
        "data-test-status": session.status
      }, label)));
    }
  } catch (err) {
    // eslint-disable-next-line no-console
    console.error(err);
  }

  // Exception has been caught
  return /*#__PURE__*/_react.default.createElement(_.TableText, null, props.session.status);
};
exports.StatusIndicator = StatusIndicator;