"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.waitUntilNextSessionCompletes$ = waitUntilNextSessionCompletes$;
var _operators = require("rxjs/operators");
var _search_session_state = require("./search_session_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Creates an observable that emits when next search session completes.
 * This utility is helpful to use in the application to delay some tasks until next session completes.
 *
 * @param sessionService - {@link ISessionService}
 * @param opts - {@link WaitUntilNextSessionCompletesOptions}
 */
function waitUntilNextSessionCompletes$(sessionService, {
  waitForIdle = 1000
} = {
  waitForIdle: 1000
}) {
  return sessionService.state$.pipe(
  // wait until new session starts
  (0, _operators.skipUntil)(sessionService.state$.pipe((0, _operators.first)(state => state === _search_session_state.SearchSessionState.None))),
  // wait until new session starts loading
  (0, _operators.skipUntil)(sessionService.state$.pipe((0, _operators.first)(state => state === _search_session_state.SearchSessionState.Loading))),
  // debounce to ignore quick switches from loading <-> completed.
  // that could happen between sequential search requests inside a single session
  (0, _operators.debounceTime)(waitForIdle),
  // then wait until it finishes
  (0, _operators.first)(state => state === _search_session_state.SearchSessionState.Completed || state === _search_session_state.SearchSessionState.BackgroundCompleted));
}