"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDashboardMenuItems = void 0;
var _reactRedux = require("react-redux");
var _react = require("react");
var _public = require("@kbn/embeddable-plugin/public");
var _common = require("../../../common");
var _dashboard_app = require("../dashboard_app");
var _dashboard_app_strings = require("../_dashboard_app_strings");
var _show_share_modal = require("./share/show_share_modal");
var _plugin_services = require("../../services/plugin_services");
var _dashboard_constants = require("../../dashboard_constants");
var _confirm_overlays = require("../../dashboard_listing/confirm_overlays");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const useDashboardMenuItems = ({
  redirectTo,
  isLabsShown,
  setIsLabsShown
}) => {
  const [isSaveInProgress, setIsSaveInProgress] = (0, _react.useState)(false);

  /**
   * Unpack dashboard services
   */
  const {
    share,
    settings: {
      uiSettings
    },
    dashboardCapabilities: {
      showWriteControls
    }
  } = _plugin_services.pluginServices.getServices();
  const isLabsEnabled = uiSettings.get(_common.UI_SETTINGS.ENABLE_LABS_UI);

  /**
   * Unpack dashboard state from redux
   */
  const dashboard = (0, _dashboard_app.useDashboardAPI)();
  const hasUnsavedChanges = dashboard.select(state => state.componentState.hasUnsavedChanges);
  const hasOverlays = dashboard.select(state => state.componentState.hasOverlays);
  const lastSavedId = dashboard.select(state => state.componentState.lastSavedId);
  const dashboardTitle = dashboard.select(state => state.explicitInput.title);
  const viewMode = dashboard.select(state => state.explicitInput.viewMode);

  /**
   * Show the Dashboard app's share menu
   */
  const showShare = (0, _react.useCallback)(anchorElement => {
    (0, _show_share_modal.ShowShareModal)({
      dashboardTitle,
      anchorElement,
      savedObjectId: lastSavedId,
      isDirty: Boolean(hasUnsavedChanges)
    });
  }, [dashboardTitle, hasUnsavedChanges, lastSavedId]);
  const maybeRedirect = (0, _react.useCallback)(result => {
    if (!result) return;
    const {
      redirectRequired,
      id
    } = result;
    if (redirectRequired) {
      redirectTo({
        id,
        editMode: true,
        useReplace: true,
        destination: 'dashboard'
      });
    }
  }, [redirectTo]);

  /**
   * Save the dashboard without any UI or popups.
   */
  const quickSaveDashboard = (0, _react.useCallback)(() => {
    setIsSaveInProgress(true);
    dashboard.runQuickSave().then(() => setTimeout(() => setIsSaveInProgress(false), _dashboard_constants.CHANGE_CHECK_DEBOUNCE));
  }, [dashboard]);

  /**
   * Show the dashboard's save modal
   */
  const saveDashboardAs = (0, _react.useCallback)(() => {
    dashboard.runSaveAs().then(result => maybeRedirect(result));
  }, [maybeRedirect, dashboard]);

  /**
   * Clone the dashboard
   */
  const clone = (0, _react.useCallback)(() => {
    dashboard.runClone().then(result => maybeRedirect(result));
  }, [maybeRedirect, dashboard]);

  /**
   * Show the dashboard's "Confirm reset changes" modal. If confirmed:
   * (1) reset the dashboard to the last saved state, and
   * (2) if `switchToViewMode` is `true`, set the dashboard to view mode.
   */
  const resetChanges = (0, _react.useCallback)((switchToViewMode = false) => {
    dashboard.clearOverlays();
    if (hasUnsavedChanges) {
      (0, _confirm_overlays.confirmDiscardUnsavedChanges)(() => {
        (0, _reactRedux.batch)(() => {
          dashboard.resetToLastSavedState();
          if (switchToViewMode) dashboard.dispatch.setViewMode(_public.ViewMode.VIEW);
        });
      }, viewMode);
    } else {
      if (switchToViewMode) dashboard.dispatch.setViewMode(_public.ViewMode.VIEW);
    }
  }, [dashboard, hasUnsavedChanges, viewMode]);

  /**
   * Register all of the top nav configs that can be used by dashboard.
   */
  const menuItems = (0, _react.useMemo)(() => {
    return {
      fullScreen: {
        ..._dashboard_app_strings.topNavStrings.fullScreen,
        id: 'full-screen',
        testId: 'dashboardFullScreenMode',
        run: () => dashboard.dispatch.setFullScreenMode(true)
      },
      labs: {
        ..._dashboard_app_strings.topNavStrings.labs,
        id: 'labs',
        testId: 'dashboardLabs',
        run: () => setIsLabsShown(!isLabsShown)
      },
      edit: {
        ..._dashboard_app_strings.topNavStrings.edit,
        emphasize: true,
        id: 'edit',
        iconType: 'pencil',
        testId: 'dashboardEditMode',
        className: 'eui-hideFor--s eui-hideFor--xs',
        // hide for small screens - editing doesn't work in mobile mode.
        run: () => {
          dashboard.dispatch.setViewMode(_public.ViewMode.EDIT);
          dashboard.clearOverlays();
        }
      },
      quickSave: {
        ..._dashboard_app_strings.topNavStrings.quickSave,
        id: 'quick-save',
        iconType: 'save',
        emphasize: true,
        isLoading: isSaveInProgress,
        testId: 'dashboardQuickSaveMenuItem',
        disableButton: !hasUnsavedChanges || isSaveInProgress || hasOverlays,
        run: () => quickSaveDashboard()
      },
      saveAs: {
        description: _dashboard_app_strings.topNavStrings.saveAs.description,
        disableButton: isSaveInProgress || hasOverlays,
        id: 'save',
        emphasize: !Boolean(lastSavedId),
        testId: 'dashboardSaveMenuItem',
        iconType: Boolean(lastSavedId) ? undefined : 'save',
        label: Boolean(lastSavedId) ? _dashboard_app_strings.topNavStrings.saveAs.label : _dashboard_app_strings.topNavStrings.quickSave.label,
        run: () => saveDashboardAs()
      },
      switchToViewMode: {
        ..._dashboard_app_strings.topNavStrings.switchToViewMode,
        id: 'cancel',
        disableButton: isSaveInProgress || !lastSavedId || hasOverlays,
        testId: 'dashboardViewOnlyMode',
        run: () => resetChanges(true)
      },
      share: {
        ..._dashboard_app_strings.topNavStrings.share,
        id: 'share',
        testId: 'shareTopNavButton',
        disableButton: isSaveInProgress || hasOverlays,
        run: showShare
      },
      settings: {
        ..._dashboard_app_strings.topNavStrings.settings,
        id: 'settings',
        testId: 'dashboardSettingsButton',
        disableButton: isSaveInProgress || hasOverlays,
        run: () => dashboard.showSettings()
      },
      clone: {
        ..._dashboard_app_strings.topNavStrings.clone,
        id: 'clone',
        testId: 'dashboardClone',
        disableButton: isSaveInProgress,
        run: () => clone()
      }
    };
  }, [quickSaveDashboard, hasUnsavedChanges, isSaveInProgress, saveDashboardAs, setIsLabsShown, resetChanges, hasOverlays, lastSavedId, isLabsShown, showShare, dashboard, clone]);
  const resetChangesMenuItem = (0, _react.useMemo)(() => {
    return {
      ..._dashboard_app_strings.topNavStrings.resetChanges,
      id: 'reset',
      testId: 'dashboardDiscardChangesMenuItem',
      disableButton: !hasUnsavedChanges || hasOverlays || viewMode === _public.ViewMode.EDIT && (isSaveInProgress || !lastSavedId),
      run: () => resetChanges()
    };
  }, [hasOverlays, lastSavedId, resetChanges, viewMode, isSaveInProgress, hasUnsavedChanges]);

  /**
   * Build ordered menus for view and edit mode.
   */
  const viewModeTopNavConfig = (0, _react.useMemo)(() => {
    const labsMenuItem = isLabsEnabled ? [menuItems.labs] : [];
    const shareMenuItem = share ? [menuItems.share] : [];
    const cloneMenuItem = showWriteControls ? [menuItems.clone] : [];
    const editMenuItem = showWriteControls ? [menuItems.edit] : [];
    return [...labsMenuItem, menuItems.fullScreen, ...shareMenuItem, ...cloneMenuItem, resetChangesMenuItem, ...editMenuItem];
  }, [menuItems, share, showWriteControls, resetChangesMenuItem, isLabsEnabled]);
  const editModeTopNavConfig = (0, _react.useMemo)(() => {
    const labsMenuItem = isLabsEnabled ? [menuItems.labs] : [];
    const shareMenuItem = share ? [menuItems.share] : [];
    const editModeItems = [];
    if (lastSavedId) {
      editModeItems.push(menuItems.saveAs, menuItems.switchToViewMode, resetChangesMenuItem, menuItems.quickSave);
    } else {
      editModeItems.push(menuItems.switchToViewMode, menuItems.saveAs);
    }
    return [...labsMenuItem, menuItems.settings, ...shareMenuItem, ...editModeItems];
  }, [lastSavedId, menuItems, share, resetChangesMenuItem, isLabsEnabled]);
  return {
    viewModeTopNavConfig,
    editModeTopNavConfig
  };
};
exports.useDashboardMenuItems = useDashboardMenuItems;