"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.catchAndReturnBoomErrors = void 0;
exports.createSavedObjectsStreamFromNdJson = createSavedObjectsStreamFromNdJson;
exports.isKibanaRequest = isKibanaRequest;
exports.logWarnOnExternalRequest = logWarnOnExternalRequest;
exports.throwIfAnyTypeNotVisibleByAPI = throwIfAnyTypeNotVisibleByAPI;
exports.throwIfTypeNotVisibleByAPI = throwIfTypeNotVisibleByAPI;
exports.throwOnGloballyHiddenTypes = throwOnGloballyHiddenTypes;
exports.throwOnHttpHiddenTypes = throwOnHttpHiddenTypes;
exports.validateObjects = validateObjects;
exports.validateTypes = validateTypes;
var _utils = require("@kbn/utils");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

async function createSavedObjectsStreamFromNdJson(ndJsonStream) {
  const savedObjects = await (0, _utils.createPromiseFromStreams)([ndJsonStream, (0, _utils.createSplitStream)('\n'), (0, _utils.createMapStream)(str => {
    if (str && str.trim() !== '') {
      return JSON.parse(str);
    }
  }), (0, _utils.createFilterStream)(obj => !!obj && obj.exportedCount === undefined), (0, _utils.createConcatStream)([])]);
  return (0, _utils.createListStream)(savedObjects);
}
function validateTypes(types, supportedTypes) {
  const invalidTypes = types.filter(t => !supportedTypes.includes(t));
  if (invalidTypes.length) {
    return `Trying to export non-exportable type(s): ${invalidTypes.join(', ')}`;
  }
}
function validateObjects(objects, supportedTypes) {
  const invalidObjects = objects.filter(obj => !supportedTypes.includes(obj.type));
  if (invalidObjects.length) {
    return `Trying to export object(s) with non-exportable types: ${invalidObjects.map(obj => `${obj.type}:${obj.id}`).join(', ')}`;
  }
}

/**
 * Catches errors thrown by saved object route handlers and returns an error
 * with the payload and statusCode of the boom error.
 *
 * This is very close to the core `router.handleLegacyErrors` except that it
 * throws internal errors (statusCode: 500) so that the internal error's
 * message get logged by Core.
 *
 * TODO: Remove once https://github.com/elastic/kibana/issues/65291 is fixed.
 */
const catchAndReturnBoomErrors = handler => {
  return async (context, request, response) => {
    try {
      return await handler(context, request, response);
    } catch (e) {
      if (_boom.default.isBoom(e) && e.output.statusCode !== 500) {
        return response.customError({
          body: e.output.payload,
          statusCode: e.output.statusCode,
          headers: e.output.headers
        });
      }
      throw e;
    }
  };
};

/**
 *
 * @param {string[]} exposedVisibleTypes all registered types with hidden:false and hiddenFromHttpApis:false|undefined
 * @param {string[]} typesToCheck saved object types provided to the httpApi request
 */
exports.catchAndReturnBoomErrors = catchAndReturnBoomErrors;
function throwOnGloballyHiddenTypes(allHttpApisVisibleTypes, typesToCheck) {
  if (!typesToCheck.length) {
    return;
  }
  const denyRequestForTypes = typesToCheck.filter(type => !allHttpApisVisibleTypes.includes(type));
  if (denyRequestForTypes.length > 0) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(`Unsupported saved object type(s): ${denyRequestForTypes.join(', ')}`);
  }
}
/**
 * @param {string[]} unsupportedTypes saved object types registered with hidden=false and hiddenFromHttpApis=true
 */

function throwOnHttpHiddenTypes(unsupportedTypes) {
  if (unsupportedTypes.length > 0) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(`Unsupported saved object type(s): ${unsupportedTypes.join(', ')}`);
  }
}
/**
 * @param {string[]} type saved object type
 * @param {ISavedObjectTypeRegistry} registry the saved object type registry
 */

function throwIfTypeNotVisibleByAPI(type, registry) {
  if (!type) return;
  const fullType = registry.getType(type);
  if (!(fullType !== null && fullType !== void 0 && fullType.hidden) && fullType !== null && fullType !== void 0 && fullType.hiddenFromHttpApis) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
  }
}
function throwIfAnyTypeNotVisibleByAPI(typesToCheck, registry) {
  const unsupportedTypes = typesToCheck.filter(tname => {
    const fullType = registry.getType(tname);
    if (!(fullType !== null && fullType !== void 0 && fullType.hidden) && fullType !== null && fullType !== void 0 && fullType.hiddenFromHttpApis) {
      return fullType.name;
    }
  });
  if (unsupportedTypes.length > 0) {
    throwOnHttpHiddenTypes(unsupportedTypes);
  }
}
function isKibanaRequest({
  headers
}) {
  // The presence of these two request headers gives us a good indication that this is a first-party request from the Kibana client.
  // We can't be 100% certain, but this is a reasonable attempt.
  return headers && headers['kbn-version'] && headers.referer;
}
/**
 * Only log a warning when the request is internal
 * Allows us to silence the logs for development
 *  @internal
 */
function logWarnOnExternalRequest(params) {
  const {
    method,
    path,
    req,
    logger
  } = params;
  if (!isKibanaRequest(req)) {
    logger.warn(`The ${method} saved object API ${path} is deprecated.`);
  }
}