"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.init = void 0;
var _lodash = require("lodash");
var Either = _interopRequireWildcard(require("fp-ts/lib/Either"));
var _retry_state = require("../../../model/retry_state");
var _helpers = require("../../../model/helpers");
var _actions = require("../../actions");
var _utils = require("../../utils");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const init = (state, res, context) => {
  var _currentMappings$_met;
  if (Either.isLeft(res)) {
    const left = res.left;
    if ((0, _actions.isTypeof)(left, 'incompatible_cluster_routing_allocation')) {
      const retryErrorMessage = `[${left.type}] Incompatible Elasticsearch cluster settings detected. Remove the persistent and transient Elasticsearch cluster setting 'cluster.routing.allocation.enable' or set it to a value of 'all' to allow migrations to proceed. Refer to ${context.migrationDocLinks.routingAllocationDisabled} for more information on how to resolve the issue.`;
      return (0, _retry_state.delayRetryState)(state, retryErrorMessage, context.maxRetryAttempts);
    } else {
      return (0, _helpers.throwBadResponse)(state, left);
    }
  }
  const types = context.types.map(type => context.typeRegistry.getType(type));
  const logs = [...state.logs];
  const indices = res.right;
  const currentIndex = (0, _utils.getCurrentIndex)(indices, context.indexPrefix);

  // No indices were found, likely because it is the first time Kibana boots.
  // In that case, we just create the index.
  if (!currentIndex) {
    return {
      ...state,
      logs,
      controlState: 'CREATE_TARGET_INDEX',
      currentIndex: `${context.indexPrefix}_1`,
      indexMappings: (0, _utils.buildIndexMappings)({
        types
      })
    };
  }

  // Index was found. This is the standard scenario, we check the model versions
  // compatibility before going further.
  const currentMappings = indices[currentIndex].mappings;
  const versionCheck = (0, _utils.checkVersionCompatibility)({
    mappings: currentMappings,
    types,
    source: 'mappingVersions',
    deletedTypes: context.deletedTypes
  });
  logs.push({
    level: 'info',
    message: `Mappings model version check result: ${versionCheck.status}`
  });
  const aliases = Object.keys(indices[currentIndex].aliases);
  const aliasActions = (0, _utils.getAliasActions)({
    existingAliases: aliases,
    currentIndex,
    indexPrefix: context.indexPrefix,
    kibanaVersion: context.kibanaVersion
  });
  // cloning as we may be mutating it in later stages.
  const currentIndexMeta = (0, _lodash.cloneDeep)(currentMappings._meta);
  const commonState = {
    logs,
    currentIndex,
    currentIndexMeta,
    aliases,
    aliasActions,
    previousMappings: currentMappings
  };
  switch (versionCheck.status) {
    // app version is greater than the index mapping version.
    // scenario of an upgrade: we need to update the mappings
    case 'greater':
      const additiveMappingChanges = (0, _utils.generateAdditiveMappingDiff)({
        types,
        meta: (_currentMappings$_met = currentMappings._meta) !== null && _currentMappings$_met !== void 0 ? _currentMappings$_met : {},
        deletedTypes: context.deletedTypes
      });
      return {
        ...state,
        controlState: 'UPDATE_INDEX_MAPPINGS',
        ...commonState,
        additiveMappingChanges,
        newIndexCreation: false
      };
    // app version and index mapping version are the same.
    // either application upgrade without model change, or a simple reboot on the same version.
    // In that case we jump directly to alias update
    case 'equal':
      return {
        ...state,
        controlState: aliasActions.length ? 'UPDATE_ALIASES' : 'INDEX_STATE_UPDATE_DONE',
        ...commonState,
        newIndexCreation: false
      };
    // app version is lower than the index mapping version.
    // likely a rollback scenario - unsupported for the initial implementation
    case 'lesser':
      return {
        ...state,
        controlState: 'FATAL',
        reason: 'Downgrading model version is currently unsupported',
        logs
      };
    // conflicts: version for some types are greater, some are lower
    // shouldn't occur in any normal scenario - cannot recover
    case 'conflict':
    default:
      return {
        ...state,
        controlState: 'FATAL',
        reason: 'Model version conflict: inconsistent higher/lower versions',
        logs
      };
  }
};
exports.init = init;