"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildTempIndexMap = buildTempIndexMap;
exports.createBatches = createBatches;
var Either = _interopRequireWildcard(require("fp-ts/lib/Either"));
var _helpers = require("./helpers");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Build a relationship of temporary index names for each SO type, e.g.:
 *  'cases': '.kibana_cases_8.8.0_reindex_temp'
 *  'task': '.kibana_task_manager_8.8.0_reindex_temp'
 *   ...
 *
 * @param indexTypesMap information about which types are stored in each index
 * @param kibanaVersion the target version of the indices
 */
function buildTempIndexMap(indexTypesMap, kibanaVersion) {
  return Object.entries(indexTypesMap || {}).reduce((acc, [indexAlias, types]) => {
    const tempIndex = (0, _helpers.getTempIndexName)(indexAlias, kibanaVersion) + '_alias';
    types.forEach(type => {
      acc[type] = tempIndex;
    });
    return acc;
  }, {});
}

/**
 * Creates batches of documents to be used by the bulk API. Each batch will
 * have a request body content length that's <= maxBatchSizeBytes
 */
function createBatches({
  documents,
  corruptDocumentIds = [],
  transformErrors = [],
  maxBatchSizeBytes,
  typeIndexMap
}) {
  /* To build up the NDJSON request body we construct an array of objects like:
   * [
   *   {"index": ...}
   *   {"title": "my saved object"}
   *   {"delete": ...}
   *   {"delete": ...}
   *   ...
   * ]
   * For indexing operations, createBulkIndexOperationTuple
   * returns a tuple of the form [{operation, id}, {document}]
   * Thus, for batch size calculations, we must take into account
   * that this tuple's surrounding brackets `[]` won't be present in the NDJSON
   */
  const BRACKETS_BYTES = 2;
  /* Each document in the NDJSON (including the last one) needs to be
   * terminated by a newline, so we need to account for an extra newline
   * character
   */
  const NDJSON_NEW_LINE_BYTES = 1;
  const BASE_DELETE_OPERATION_SIZE = Buffer.byteLength(JSON.stringify((0, _helpers.createBulkDeleteOperationBody)('')), 'utf8');
  const batches = [[]];
  let currBatch = 0;
  let currBatchSizeBytes = 0;

  // group operations in batches of at most maxBatchSize
  const assignToBatch = (operation, operationSizeBytes) => {
    operationSizeBytes += NDJSON_NEW_LINE_BYTES;
    if (operationSizeBytes > maxBatchSizeBytes) {
      // the current operation (+ payload) does not even fit a single batch, fail!
      return false;
    } else if (currBatchSizeBytes + operationSizeBytes <= maxBatchSizeBytes) {
      batches[currBatch].push(operation);
      currBatchSizeBytes = currBatchSizeBytes + operationSizeBytes;
    } else {
      currBatch++;
      batches[currBatch] = [operation];
      currBatchSizeBytes = operationSizeBytes;
    }
    return true;
  };

  // create index (update) operations for all transformed documents
  for (const document of documents) {
    const bulkIndexOperationBody = (0, _helpers.createBulkIndexOperationTuple)(document, typeIndexMap);
    // take into account that this tuple's surrounding brackets `[]` won't be present in the NDJSON
    const docSizeBytes = Buffer.byteLength(JSON.stringify(bulkIndexOperationBody), 'utf8') - BRACKETS_BYTES;
    if (!assignToBatch(bulkIndexOperationBody, docSizeBytes)) {
      return Either.left({
        documentId: document._id,
        type: 'document_exceeds_batch_size_bytes',
        docSizeBytes,
        maxBatchSizeBytes
      });
    }
  }

  // create delete operations for all corrupt documents + transform errors
  const unwantedDocumentIds = [...corruptDocumentIds, ...transformErrors.map(({
    rawId: documentId
  }) => documentId)];
  for (const documentId of unwantedDocumentIds) {
    const bulkDeleteOperationBody = (0, _helpers.createBulkDeleteOperationBody)(documentId);
    const docSizeBytes = BASE_DELETE_OPERATION_SIZE + Buffer.byteLength(documentId, 'utf8');
    if (!assignToBatch(bulkDeleteOperationBody, docSizeBytes)) {
      return Either.left({
        documentId,
        type: 'document_exceeds_batch_size_bytes',
        docSizeBytes,
        maxBatchSizeBytes
      });
    }
  }
  return Either.right(batches);
}