"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DocumentUpgradePipeline = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _semver = _interopRequireDefault(require("semver"));
var _types = require("../types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function isGreater(a, b) {
  return !!a && (!b || _semver.default.gt(a, b));
}
class DocumentUpgradePipeline {
  constructor({
    document,
    migrations,
    kibanaVersion,
    convertNamespaceTypes,
    targetTypeVersion
  }) {
    var _migrations$document$;
    (0, _defineProperty2.default)(this, "additionalDocs", []);
    (0, _defineProperty2.default)(this, "document", void 0);
    (0, _defineProperty2.default)(this, "originalDoc", void 0);
    (0, _defineProperty2.default)(this, "migrations", void 0);
    (0, _defineProperty2.default)(this, "kibanaVersion", void 0);
    (0, _defineProperty2.default)(this, "convertNamespaceTypes", void 0);
    (0, _defineProperty2.default)(this, "targetTypeVersion", void 0);
    this.originalDoc = document;
    this.document = (0, _lodash.cloneDeep)(document);
    this.migrations = migrations;
    this.kibanaVersion = kibanaVersion;
    this.convertNamespaceTypes = convertNamespaceTypes;
    this.targetTypeVersion = targetTypeVersion || ((_migrations$document$ = migrations[document.type]) === null || _migrations$document$ === void 0 ? void 0 : _migrations$document$.latestVersion.migrate);
  }
  *getPipeline() {
    while (this.hasPendingTransforms()) {
      for (const transform of this.getPendingTransforms()) {
        yield transform;
        if (this.document.type !== this.originalDoc.type) {
          throw new Error(`Changing a document's type during a migration is not supported.`);
        }
      }
    }
  }
  hasPendingTransforms() {
    var _this$migrations$type;
    const {
      coreMigrationVersion,
      typeMigrationVersion,
      type
    } = this.document;
    const latestVersion = (_this$migrations$type = this.migrations[type]) === null || _this$migrations$type === void 0 ? void 0 : _this$migrations$type.latestVersion;
    if (isGreater(latestVersion === null || latestVersion === void 0 ? void 0 : latestVersion.core, coreMigrationVersion)) {
      return true;
    }
    if (typeMigrationVersion == null) {
      return false;
    }
    return isGreater(this.targetTypeVersion, typeMigrationVersion) || this.convertNamespaceTypes && isGreater(latestVersion === null || latestVersion === void 0 ? void 0 : latestVersion.convert, typeMigrationVersion) || this.convertNamespaceTypes && isGreater(latestVersion === null || latestVersion === void 0 ? void 0 : latestVersion.reference, coreMigrationVersion);
  }
  getPendingTransforms() {
    const {
      transforms
    } = this.migrations[this.document.type];
    return transforms.filter(transform => this.isPendingTransform(transform));
  }
  isPendingTransform({
    transformType,
    version
  }) {
    var _this$migrations$type2;
    const {
      coreMigrationVersion,
      typeMigrationVersion,
      type
    } = this.document;
    const latestVersion = (_this$migrations$type2 = this.migrations[type]) === null || _this$migrations$type2 === void 0 ? void 0 : _this$migrations$type2.latestVersion;
    switch (transformType) {
      case _types.TransformType.Core:
        return isGreater(version, coreMigrationVersion);
      case _types.TransformType.Reference:
        return (this.convertNamespaceTypes || isGreater(latestVersion.core, coreMigrationVersion)) && isGreater(version, coreMigrationVersion);
      case _types.TransformType.Convert:
        return typeMigrationVersion != null && this.convertNamespaceTypes && isGreater(version, typeMigrationVersion);
      case _types.TransformType.Migrate:
        return typeMigrationVersion != null && isGreater(version, typeMigrationVersion) && _semver.default.lte(version, this.targetTypeVersion);
    }
  }

  /**
   * Verifies that the document version is not greater than the version supported by Kibana.
   * If we have a document with some version and no migrations available for this type,
   * the document belongs to a future version.
   */
  assertCompatibility() {
    var _this$migrations$type3, _this$migrations$type4;
    const {
      id,
      type,
      typeMigrationVersion: currentVersion
    } = this.document;
    const latestVersion = (0, _utils.maxVersion)((_this$migrations$type3 = this.migrations[type]) === null || _this$migrations$type3 === void 0 ? void 0 : _this$migrations$type3.latestVersion.migrate, (_this$migrations$type4 = this.migrations[type]) === null || _this$migrations$type4 === void 0 ? void 0 : _this$migrations$type4.latestVersion.convert);
    if (isGreater(currentVersion, latestVersion)) {
      throw _boom.default.badData(`Document "${id}" belongs to a more recent version of Kibana [${currentVersion}] when the last known version is [${latestVersion}].`, this.document);
    }
  }

  /**
   * Transforms that remove or downgrade `typeMigrationVersion` properties are not allowed,
   * as this could get us into an infinite loop. So, we explicitly check for that here.
   */
  assertUpgrade({
    transformType,
    version
  }, previousVersion) {
    if (_utils.coreVersionTransformTypes.includes(transformType)) {
      return;
    }
    const {
      typeMigrationVersion: currentVersion,
      type
    } = this.document;
    if (isGreater(previousVersion, currentVersion)) {
      throw new Error(`Migration "${type} v${version}" attempted to downgrade "typeMigrationVersion" from ${previousVersion} to ${currentVersion}.`);
    }
  }
  ensureVersion({
    coreMigrationVersion: currentCoreMigrationVersion,
    typeMigrationVersion: currentTypeMigrationVersion,
    ...document
  }) {
    var _this$migrations$type5;
    const {
      type
    } = document;
    const latestVersion = (_this$migrations$type5 = this.migrations[type]) === null || _this$migrations$type5 === void 0 ? void 0 : _this$migrations$type5.latestVersion;
    const coreMigrationVersion = currentCoreMigrationVersion || (0, _utils.maxVersion)(latestVersion === null || latestVersion === void 0 ? void 0 : latestVersion.core, latestVersion === null || latestVersion === void 0 ? void 0 : latestVersion.reference);
    const typeMigrationVersion = currentTypeMigrationVersion || (0, _utils.maxVersion)(latestVersion === null || latestVersion === void 0 ? void 0 : latestVersion.migrate, latestVersion === null || latestVersion === void 0 ? void 0 : latestVersion.convert);
    return {
      ...document,
      ...(coreMigrationVersion ? {
        coreMigrationVersion
      } : {}),
      ...(typeMigrationVersion ? {
        typeMigrationVersion
      } : {})
    };
  }
  run() {
    (0, _utils.assertValidCoreVersion)({
      document: this.document,
      kibanaVersion: this.kibanaVersion
    });
    for (const transform of this.getPipeline()) {
      const {
        typeMigrationVersion: previousVersion
      } = this.document;
      const {
        additionalDocs,
        transformedDoc
      } = transform.transform(this.document);
      this.document = transformedDoc;
      this.additionalDocs.push(...additionalDocs.map(document => this.ensureVersion(document)));
      this.assertUpgrade(transform, previousVersion);
      this.document = (0, _utils.applyVersion)({
        document: this.document,
        transform
      });
    }
    this.assertCompatibility();
    this.document = this.ensureVersion(this.document);
    return {
      document: this.document,
      additionalDocs: this.additionalDocs
    };
  }
}
exports.DocumentUpgradePipeline = DocumentUpgradePipeline;