"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DocumentMigrator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("./pipelines/utils");
var _build_active_migrations = require("./build_active_migrations");
var _pipelines = require("./pipelines");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * A concrete implementation of the VersionedTransformer interface.
 */
class DocumentMigrator {
  /**
   * Creates an instance of DocumentMigrator.
   *
   * @param {DocumentMigratorOptions} options
   * @prop {string} kibanaVersion - The current version of Kibana
   * @prop {SavedObjectTypeRegistry} typeRegistry - The type registry to get type migrations from
   * @prop {string} convertVersion - The version of Kibana in which documents can be converted to multi-namespace types
   * @prop {Logger} log - The migration logger
   */
  constructor(_options) {
    (0, _defineProperty2.default)(this, "options", void 0);
    (0, _defineProperty2.default)(this, "migrations", void 0);
    /**
     * Prepares active migrations and document transformer function.
     */
    (0, _defineProperty2.default)(this, "prepareMigrations", () => {
      const {
        typeRegistry,
        kibanaVersion,
        log,
        convertVersion
      } = this.options;
      this.migrations = (0, _build_active_migrations.buildActiveMigrations)({
        typeRegistry,
        kibanaVersion,
        log,
        convertVersion
      });
    });
    /**
     * Migrates a document to the latest version.
     */
    (0, _defineProperty2.default)(this, "migrate", doc => {
      const {
        document
      } = this.transform(doc);
      return document;
    });
    /**
     * Migrates a document to the latest version and applies type conversions if applicable. Also returns any additional document(s) that may
     * have been created during the transformation process.
     */
    (0, _defineProperty2.default)(this, "migrateAndConvert", doc => {
      const {
        document,
        additionalDocs
      } = this.transform(doc, {
        convertNamespaceTypes: true
      });
      return [document, ...additionalDocs];
    });
    (0, _defineProperty2.default)(this, "transformDown", (doc, options) => {
      if (!this.migrations) {
        throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
      }
      const pipeline = new _pipelines.DocumentDowngradePipeline({
        document: doc,
        typeTransforms: this.migrations[doc.type],
        kibanaVersion: this.options.kibanaVersion,
        targetTypeVersion: options.targetTypeVersion
      });
      const {
        document
      } = pipeline.run();
      return document;
    });
    this.options = _options;
  }

  /**
   * Gets the latest version of each migrate-able property.
   */
  get migrationVersion() {
    if (!this.migrations) {
      throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
    }
    return Object.entries(this.migrations).reduce((acc, [prop, {
      latestVersion
    }]) => {
      // some migration objects won't have a latest migration version (they only contain reference transforms that are applied from other types)
      const latestMigrationVersion = (0, _utils.maxVersion)(latestVersion.migrate, latestVersion.convert);
      if (latestMigrationVersion) {
        return {
          ...acc,
          [prop]: latestMigrationVersion
        };
      }
      return acc;
    }, {});
  }
  transform(doc, {
    convertNamespaceTypes = false
  } = {}) {
    if (!this.migrations) {
      throw new Error('Migrations are not ready. Make sure prepareMigrations is called first.');
    }
    const pipeline = new _pipelines.DocumentUpgradePipeline({
      document: doc,
      migrations: this.migrations,
      kibanaVersion: this.options.kibanaVersion,
      convertNamespaceTypes
    });
    const {
      document,
      additionalDocs
    } = pipeline.run();
    return {
      document,
      additionalDocs
    };
  }
}
exports.DocumentMigrator = DocumentMigrator;