"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedObjectsRepository = exports.DEFAULT_RETRY_COUNT = exports.DEFAULT_REFRESH_SETTING = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var esKuery = _interopRequireWildcard(require("@kbn/es-query"));
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _pMap = _interopRequireDefault(require("p-map"));
var _point_in_time_finder = require("./point_in_time_finder");
var _repository_es_client = require("./repository_es_client");
var _search_dsl = require("./search_dsl");
var _included_fields = require("./included_fields");
var _internal_bulk_resolve = require("./internal_bulk_resolve");
var _filter_utils = require("./filter_utils");
var _aggregations = require("./aggregations");
var _internal_utils = require("./internal_utils");
var _collect_multi_namespace_references = require("./collect_multi_namespace_references");
var _update_objects_spaces = require("./update_objects_spaces");
var _preflight_check_for_create = require("./preflight_check_for_create");
var _legacy_url_aliases = require("./legacy_url_aliases");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const DEFAULT_REFRESH_SETTING = 'wait_for';
exports.DEFAULT_REFRESH_SETTING = DEFAULT_REFRESH_SETTING;
const DEFAULT_RETRY_COUNT = 3;
exports.DEFAULT_RETRY_COUNT = DEFAULT_RETRY_COUNT;
const MAX_CONCURRENT_ALIAS_DELETIONS = 10;

/**
 * @internal
 */

function isMgetDoc(doc) {
  return Boolean(doc && 'found' in doc);
}

/**
 * Saved Objects Respositiry - the client entry point for saved object manipulation.
 *
 * The SOR calls the Elasticsearch client and leverages extension implementations to
 * support spaces, security, and encryption features.
 *
 * @public
 */
class SavedObjectsRepository {
  /**
   * A factory function for creating SavedObjectRepository instances.
   *
   * @internalRemarks
   * Tests are located in ./repository_create_repository.test.ts
   *
   * @internal
   */
  static createRepository(migrator, typeRegistry, indexName, client, logger, includedHiddenTypes = [], extensions, /** The injectedConstructor is only used for unit testing */
  injectedConstructor = SavedObjectsRepository) {
    const mappings = migrator.getActiveMappings();
    const allTypes = typeRegistry.getAllTypes().map(t => t.name);
    const serializer = new _coreSavedObjectsBaseServerInternal.SavedObjectsSerializer(typeRegistry);
    const visibleTypes = allTypes.filter(type => !typeRegistry.isHidden(type));
    const missingTypeMappings = includedHiddenTypes.filter(type => !allTypes.includes(type));
    if (missingTypeMappings.length > 0) {
      throw new Error(`Missing mappings for saved objects types: '${missingTypeMappings.join(', ')}'`);
    }
    const allowedTypes = [...new Set(visibleTypes.concat(includedHiddenTypes))];
    return new injectedConstructor({
      index: indexName,
      migrator,
      mappings,
      typeRegistry,
      serializer,
      allowedTypes,
      client,
      logger,
      extensions
    });
  }
  constructor(options) {
    (0, _defineProperty2.default)(this, "_migrator", void 0);
    (0, _defineProperty2.default)(this, "_index", void 0);
    (0, _defineProperty2.default)(this, "_mappings", void 0);
    (0, _defineProperty2.default)(this, "_registry", void 0);
    (0, _defineProperty2.default)(this, "_allowedTypes", void 0);
    (0, _defineProperty2.default)(this, "typeValidatorMap", {});
    (0, _defineProperty2.default)(this, "client", void 0);
    (0, _defineProperty2.default)(this, "_encryptionExtension", void 0);
    (0, _defineProperty2.default)(this, "_securityExtension", void 0);
    (0, _defineProperty2.default)(this, "_spacesExtension", void 0);
    (0, _defineProperty2.default)(this, "_serializer", void 0);
    (0, _defineProperty2.default)(this, "_logger", void 0);
    const {
      index,
      mappings,
      client,
      typeRegistry,
      serializer,
      migrator,
      allowedTypes = [],
      logger,
      extensions
    } = options;

    // It's important that we migrate documents / mark them as up-to-date
    // prior to writing them to the index. Otherwise, we'll cause unnecessary
    // index migrations to run at Kibana startup, and those will probably fail
    // due to invalidly versioned documents in the index.
    //
    // The migrator performs double-duty, and validates the documents prior
    // to returning them.
    this._migrator = migrator;
    this._index = index;
    this._mappings = mappings;
    this._registry = typeRegistry;
    this.client = (0, _repository_es_client.createRepositoryEsClient)(client);
    if (allowedTypes.length === 0) {
      throw new Error('Empty or missing types for saved object repository!');
    }
    this._allowedTypes = allowedTypes;
    this._serializer = serializer;
    this._logger = logger;
    this._encryptionExtension = extensions === null || extensions === void 0 ? void 0 : extensions.encryptionExtension;
    this._securityExtension = extensions === null || extensions === void 0 ? void 0 : extensions.securityExtension;
    this._spacesExtension = extensions === null || extensions === void 0 ? void 0 : extensions.spacesExtension;
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.create}
   */
  async create(type, attributes, options = {}) {
    var _this$_securityExtens, _preflightResult$exis, _preflightResult3, _preflightResult3$exi, _preflightResult3$exi2, _preflightResult4;
    const namespace = this.getCurrentNamespace(options.namespace);
    const {
      migrationVersion,
      coreMigrationVersion,
      typeMigrationVersion,
      managed,
      overwrite = false,
      references = [],
      refresh = DEFAULT_REFRESH_SETTING,
      initialNamespaces,
      version
    } = options;
    const {
      migrationVersionCompatibility
    } = options;
    if (!this._allowedTypes.includes(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
    }
    const id = this.getValidId(type, options.id, options.version, options.overwrite);
    this.validateInitialNamespaces(type, initialNamespaces);
    this.validateOriginId(type, options);
    const time = (0, _internal_utils.getCurrentTime)();
    let savedObjectNamespace;
    let savedObjectNamespaces;
    let existingOriginId;
    const namespaceString = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace);
    let preflightResult;
    if (this._registry.isSingleNamespace(type)) {
      savedObjectNamespace = initialNamespaces ? (0, _internal_utils.normalizeNamespace)(initialNamespaces[0]) : namespace;
    } else if (this._registry.isMultiNamespace(type)) {
      var _preflightResult, _preflightResult2, _preflightResult2$exi, _preflightResult2$exi2;
      if (options.id) {
        // we will overwrite a multi-namespace saved object if it exists; if that happens, ensure we preserve its included namespaces
        // note: this check throws an error if the object is found but does not exist in this namespace
        preflightResult = (await (0, _preflight_check_for_create.preflightCheckForCreate)({
          registry: this._registry,
          client: this.client,
          serializer: this._serializer,
          getIndexForType: this.getIndexForType.bind(this),
          createPointInTimeFinder: this.createPointInTimeFinder.bind(this),
          objects: [{
            type,
            id,
            overwrite,
            namespaces: initialNamespaces !== null && initialNamespaces !== void 0 ? initialNamespaces : [namespaceString]
          }]
        }))[0];
      }
      savedObjectNamespaces = initialNamespaces || getSavedObjectNamespaces(namespace, (_preflightResult = preflightResult) === null || _preflightResult === void 0 ? void 0 : _preflightResult.existingDocument);
      existingOriginId = (_preflightResult2 = preflightResult) === null || _preflightResult2 === void 0 ? void 0 : (_preflightResult2$exi = _preflightResult2.existingDocument) === null || _preflightResult2$exi === void 0 ? void 0 : (_preflightResult2$exi2 = _preflightResult2$exi._source) === null || _preflightResult2$exi2 === void 0 ? void 0 : _preflightResult2$exi2.originId;
    }
    const authorizationResult = await ((_this$_securityExtens = this._securityExtension) === null || _this$_securityExtens === void 0 ? void 0 : _this$_securityExtens.authorizeCreate({
      namespace,
      object: {
        type,
        id,
        initialNamespaces,
        existingNamespaces: (_preflightResult$exis = (_preflightResult3 = preflightResult) === null || _preflightResult3 === void 0 ? void 0 : (_preflightResult3$exi = _preflightResult3.existingDocument) === null || _preflightResult3$exi === void 0 ? void 0 : (_preflightResult3$exi2 = _preflightResult3$exi._source) === null || _preflightResult3$exi2 === void 0 ? void 0 : _preflightResult3$exi2.namespaces) !== null && _preflightResult$exis !== void 0 ? _preflightResult$exis : []
      }
    }));
    if ((_preflightResult4 = preflightResult) !== null && _preflightResult4 !== void 0 && _preflightResult4.error) {
      // This intentionally occurs _after_ the authZ enforcement (which may throw a 403 error earlier)
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id);
    }

    // 1. If the originId has been *explicitly set* in the options (defined or undefined), respect that.
    // 2. Otherwise, preserve the originId of the existing object that is being overwritten, if any.
    const originId = Object.keys(options).includes('originId') ? options.originId : existingOriginId;
    const migrated = this._migrator.migrateDocument({
      id,
      type,
      ...(savedObjectNamespace && {
        namespace: savedObjectNamespace
      }),
      ...(savedObjectNamespaces && {
        namespaces: savedObjectNamespaces
      }),
      originId,
      attributes: await this.optionallyEncryptAttributes(type, id, savedObjectNamespace,
      // if single namespace type, this is the first in initialNamespaces. If multi-namespace type this is options.namespace/current namespace.
      attributes),
      migrationVersion,
      coreMigrationVersion,
      typeMigrationVersion,
      managed: (0, _internal_utils.setManaged)({
        optionsManaged: managed
      }),
      created_at: time,
      updated_at: time,
      ...(Array.isArray(references) && {
        references
      })
    });

    /**
     * If a validation has been registered for this type, we run it against the migrated attributes.
     * This is an imperfect solution because malformed attributes could have already caused the
     * migration to fail, but it's the best we can do without devising a way to run validations
     * inside the migration algorithm itself.
     */
    this.validateObjectForCreate(type, migrated);
    const raw = this._serializer.savedObjectToRaw(migrated);
    const requestParams = {
      id: raw._id,
      index: this.getIndexForType(type),
      refresh,
      body: raw._source,
      ...(overwrite && version ? (0, _coreSavedObjectsBaseServerInternal.decodeRequestVersion)(version) : {}),
      require_alias: true
    };
    const {
      body,
      statusCode,
      headers
    } = id && overwrite ? await this.client.index(requestParams, {
      meta: true
    }) : await this.client.create(requestParams, {
      meta: true
    });

    // throw if we can't verify a 404 response is from Elasticsearch
    if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode,
      headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(id, type);
    }
    return this.optionallyDecryptAndRedactSingleResult(this._rawToSavedObject({
      ...raw,
      ...body
    }, {
      migrationVersionCompatibility
    }), authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, attributes);
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.bulkCreate}
   */
  async bulkCreate(objects, options = {}) {
    var _this$_securityExtens2;
    const namespace = this.getCurrentNamespace(options.namespace);
    const {
      migrationVersionCompatibility,
      overwrite = false,
      refresh = DEFAULT_REFRESH_SETTING,
      managed: optionsManaged
    } = options;
    const time = (0, _internal_utils.getCurrentTime)();
    let preflightCheckIndexCounter = 0;
    const expectedResults = objects.map(object => {
      const {
        type,
        id: requestId,
        initialNamespaces,
        version,
        managed
      } = object;
      let error;
      let id = ''; // Assign to make TS happy, the ID will be validated (or randomly generated if needed) during getValidId below
      const objectManaged = managed;
      if (!this._allowedTypes.includes(type)) {
        error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
      } else {
        try {
          id = this.getValidId(type, requestId, version, overwrite);
          this.validateInitialNamespaces(type, initialNamespaces);
          this.validateOriginId(type, object);
        } catch (e) {
          error = e;
        }
      }
      if (error) {
        return {
          tag: 'Left',
          value: {
            id: requestId,
            type,
            error: errorContent(error)
          }
        };
      }
      const method = requestId && overwrite ? 'index' : 'create';
      const requiresNamespacesCheck = requestId && this._registry.isMultiNamespace(type);
      return {
        tag: 'Right',
        value: {
          method,
          object: {
            ...object,
            id,
            managed: (0, _internal_utils.setManaged)({
              optionsManaged,
              objectManaged
            })
          },
          ...(requiresNamespacesCheck && {
            preflightCheckIndex: preflightCheckIndexCounter++
          })
        }
      };
    });
    const validObjects = expectedResults.filter(_internal_utils.isRight);
    if (validObjects.length === 0) {
      // We only have error results; return early to avoid potentially trying authZ checks for 0 types which would result in an exception.
      return {
        // Technically the returned array should only contain SavedObject results, but for errors this is not true (we cast to 'unknown' below)
        saved_objects: expectedResults.map(({
          value
        }) => value)
      };
    }
    const namespaceString = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace);
    const preflightCheckObjects = validObjects.filter(({
      value
    }) => value.preflightCheckIndex !== undefined).map(({
      value
    }) => {
      const {
        type,
        id,
        initialNamespaces
      } = value.object;
      const namespaces = initialNamespaces !== null && initialNamespaces !== void 0 ? initialNamespaces : [namespaceString];
      return {
        type,
        id,
        overwrite,
        namespaces
      };
    });
    const preflightCheckResponse = await (0, _preflight_check_for_create.preflightCheckForCreate)({
      registry: this._registry,
      client: this.client,
      serializer: this._serializer,
      getIndexForType: this.getIndexForType.bind(this),
      createPointInTimeFinder: this.createPointInTimeFinder.bind(this),
      objects: preflightCheckObjects
    });
    const authObjects = validObjects.map(element => {
      var _preflightResult$exis2, _preflightResult$exis3;
      const {
        object,
        preflightCheckIndex: index
      } = element.value;
      const preflightResult = index !== undefined ? preflightCheckResponse[index] : undefined;
      return {
        type: object.type,
        id: object.id,
        initialNamespaces: object.initialNamespaces,
        existingNamespaces: (_preflightResult$exis2 = preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$exis3 = preflightResult.existingDocument) === null || _preflightResult$exis3 === void 0 ? void 0 : _preflightResult$exis3._source.namespaces) !== null && _preflightResult$exis2 !== void 0 ? _preflightResult$exis2 : []
      };
    });
    const authorizationResult = await ((_this$_securityExtens2 = this._securityExtension) === null || _this$_securityExtens2 === void 0 ? void 0 : _this$_securityExtens2.authorizeBulkCreate({
      namespace,
      objects: authObjects
    }));
    let bulkRequestIndexCounter = 0;
    const bulkCreateParams = [];
    const expectedBulkResults = await Promise.all(expectedResults.map(async expectedBulkGetResult => {
      if ((0, _internal_utils.isLeft)(expectedBulkGetResult)) {
        return expectedBulkGetResult;
      }
      let savedObjectNamespace;
      let savedObjectNamespaces;
      let existingOriginId;
      let versionProperties;
      const {
        preflightCheckIndex,
        object: {
          initialNamespaces,
          version,
          ...object
        },
        method
      } = expectedBulkGetResult.value;
      if (preflightCheckIndex !== undefined) {
        var _existingDocument$_so;
        const preflightResult = preflightCheckResponse[preflightCheckIndex];
        const {
          type,
          id,
          existingDocument,
          error
        } = preflightResult;
        if (error) {
          const {
            metadata
          } = error;
          return {
            tag: 'Left',
            value: {
              id,
              type,
              error: {
                ...errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id)),
                ...(metadata && {
                  metadata
                })
              }
            }
          };
        }
        savedObjectNamespaces = initialNamespaces || getSavedObjectNamespaces(namespace, existingDocument);
        versionProperties = (0, _internal_utils.getExpectedVersionProperties)(version);
        existingOriginId = existingDocument === null || existingDocument === void 0 ? void 0 : (_existingDocument$_so = existingDocument._source) === null || _existingDocument$_so === void 0 ? void 0 : _existingDocument$_so.originId;
      } else {
        if (this._registry.isSingleNamespace(object.type)) {
          savedObjectNamespace = initialNamespaces ? (0, _internal_utils.normalizeNamespace)(initialNamespaces[0]) : namespace;
        } else if (this._registry.isMultiNamespace(object.type)) {
          savedObjectNamespaces = initialNamespaces || getSavedObjectNamespaces(namespace);
        }
        versionProperties = (0, _internal_utils.getExpectedVersionProperties)(version);
      }

      // 1. If the originId has been *explicitly set* in the options (defined or undefined), respect that.
      // 2. Otherwise, preserve the originId of the existing object that is being overwritten, if any.
      const originId = Object.keys(object).includes('originId') ? object.originId : existingOriginId;
      const migrated = this._migrator.migrateDocument({
        id: object.id,
        type: object.type,
        attributes: await this.optionallyEncryptAttributes(object.type, object.id, savedObjectNamespace,
        // only used for multi-namespace object types
        object.attributes),
        migrationVersion: object.migrationVersion,
        coreMigrationVersion: object.coreMigrationVersion,
        typeMigrationVersion: object.typeMigrationVersion,
        ...(savedObjectNamespace && {
          namespace: savedObjectNamespace
        }),
        ...(savedObjectNamespaces && {
          namespaces: savedObjectNamespaces
        }),
        managed: (0, _internal_utils.setManaged)({
          optionsManaged,
          objectManaged: object.managed
        }),
        updated_at: time,
        created_at: time,
        references: object.references || [],
        originId
      });

      /**
       * If a validation has been registered for this type, we run it against the migrated attributes.
       * This is an imperfect solution because malformed attributes could have already caused the
       * migration to fail, but it's the best we can do without devising a way to run validations
       * inside the migration algorithm itself.
       */
      try {
        this.validateObjectForCreate(object.type, migrated);
      } catch (error) {
        return {
          tag: 'Left',
          value: {
            id: object.id,
            type: object.type,
            error
          }
        };
      }
      const expectedResult = {
        esRequestIndex: bulkRequestIndexCounter++,
        requestedId: object.id,
        rawMigratedDoc: this._serializer.savedObjectToRaw(migrated)
      };
      bulkCreateParams.push({
        [method]: {
          _id: expectedResult.rawMigratedDoc._id,
          _index: this.getIndexForType(object.type),
          ...(overwrite && versionProperties)
        }
      }, expectedResult.rawMigratedDoc._source);
      return {
        tag: 'Right',
        value: expectedResult
      };
    }));
    const bulkResponse = bulkCreateParams.length ? await this.client.bulk({
      refresh,
      require_alias: true,
      body: bulkCreateParams
    }) : undefined;
    const result = {
      saved_objects: expectedBulkResults.map(expectedResult => {
        var _bulkResponse$items$e;
        if ((0, _internal_utils.isLeft)(expectedResult)) {
          return expectedResult.value;
        }
        const {
          requestedId,
          rawMigratedDoc,
          esRequestIndex
        } = expectedResult.value;
        const rawResponse = Object.values((_bulkResponse$items$e = bulkResponse === null || bulkResponse === void 0 ? void 0 : bulkResponse.items[esRequestIndex]) !== null && _bulkResponse$items$e !== void 0 ? _bulkResponse$items$e : {})[0];
        const error = (0, _internal_utils.getBulkOperationError)(rawMigratedDoc._source.type, requestedId, rawResponse);
        if (error) {
          return {
            type: rawMigratedDoc._source.type,
            id: requestedId,
            error
          };
        }

        // When method == 'index' the bulkResponse doesn't include the indexed
        // _source so we return rawMigratedDoc but have to spread the latest
        // _seq_no and _primary_term values from the rawResponse.
        return this._rawToSavedObject({
          ...rawMigratedDoc,
          ...{
            _seq_no: rawResponse._seq_no,
            _primary_term: rawResponse._primary_term
          }
        }, {
          migrationVersionCompatibility
        });
      })
    };
    return this.optionallyDecryptAndRedactBulkResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, objects);
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.checkConflicts}
   */
  async checkConflicts(objects = [], options = {}) {
    var _this$_securityExtens3;
    const namespace = this.getCurrentNamespace(options.namespace);
    if (objects.length === 0) {
      return {
        errors: []
      };
    }
    let bulkGetRequestIndexCounter = 0;
    const expectedBulkGetResults = objects.map(object => {
      const {
        type,
        id
      } = object;
      if (!this._allowedTypes.includes(type)) {
        return {
          tag: 'Left',
          value: {
            id,
            type,
            error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type))
          }
        };
      }
      return {
        tag: 'Right',
        value: {
          type,
          id,
          esRequestIndex: bulkGetRequestIndexCounter++
        }
      };
    });
    const validObjects = expectedBulkGetResults.filter(_internal_utils.isRight);
    await ((_this$_securityExtens3 = this._securityExtension) === null || _this$_securityExtens3 === void 0 ? void 0 : _this$_securityExtens3.authorizeCheckConflicts({
      namespace,
      objects: validObjects.map(element => ({
        type: element.value.type,
        id: element.value.id
      }))
    }));
    const bulkGetDocs = validObjects.map(({
      value: {
        type,
        id
      }
    }) => ({
      _id: this._serializer.generateRawId(namespace, type, id),
      _index: this.getIndexForType(type),
      _source: {
        includes: ['type', 'namespaces']
      }
    }));
    const bulkGetResponse = bulkGetDocs.length ? await this.client.mget({
      body: {
        docs: bulkGetDocs
      }
    }, {
      ignore: [404],
      meta: true
    }) : undefined;
    // throw if we can't verify a 404 response is from Elasticsearch
    if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode: bulkGetResponse.statusCode,
      headers: bulkGetResponse.headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
    }
    const errors = [];
    expectedBulkGetResults.forEach(expectedResult => {
      if ((0, _internal_utils.isLeft)(expectedResult)) {
        errors.push(expectedResult.value);
        return;
      }
      const {
        type,
        id,
        esRequestIndex
      } = expectedResult.value;
      const doc = bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[esRequestIndex];
      if (isMgetDoc(doc) && doc.found) {
        errors.push({
          id,
          type,
          error: {
            ...errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id)),
            // @ts-expect-error MultiGetHit._source is optional
            ...(!this.rawDocExistsInNamespace(doc, namespace) && {
              metadata: {
                isNotOverwritable: true
              }
            })
          }
        });
      }
    });
    return {
      errors
    };
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.delete}
   */
  async delete(type, id, options = {}) {
    var _this$_securityExtens4;
    const namespace = this.getCurrentNamespace(options.namespace);
    if (!this._allowedTypes.includes(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    const {
      refresh = DEFAULT_REFRESH_SETTING,
      force
    } = options;

    // we don't need to pass existing namespaces in because we're only concerned with authorizing
    // the current space. This saves us from performing the preflight check if we're unauthorized
    await ((_this$_securityExtens4 = this._securityExtension) === null || _this$_securityExtens4 === void 0 ? void 0 : _this$_securityExtens4.authorizeDelete({
      namespace,
      object: {
        type,
        id
      }
    }));
    const rawId = this._serializer.generateRawId(namespace, type, id);
    let preflightResult;
    if (this._registry.isMultiNamespace(type)) {
      var _preflightResult$save;
      // note: this check throws an error if the object is found but does not exist in this namespace
      preflightResult = await this.preflightCheckNamespaces({
        type,
        id,
        namespace
      });
      if (preflightResult.checkResult === 'found_outside_namespace' || preflightResult.checkResult === 'not_found') {
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
      }
      const existingNamespaces = (_preflightResult$save = preflightResult.savedObjectNamespaces) !== null && _preflightResult$save !== void 0 ? _preflightResult$save : [];
      if (!force && (existingNamespaces.length > 1 || existingNamespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING))) {
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('Unable to delete saved object that exists in multiple namespaces, use the `force` option to delete it anyway');
      }
    }
    const {
      body,
      statusCode,
      headers
    } = await this.client.delete({
      id: rawId,
      index: this.getIndexForType(type),
      ...(0, _internal_utils.getExpectedVersionProperties)(undefined),
      refresh
    }, {
      ignore: [404],
      meta: true
    });
    if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode,
      headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(type, id);
    }
    const deleted = body.result === 'deleted';
    if (deleted) {
      var _preflightResult5;
      const namespaces = (_preflightResult5 = preflightResult) === null || _preflightResult5 === void 0 ? void 0 : _preflightResult5.savedObjectNamespaces;
      if (namespaces) {
        // This is a multi-namespace object type, and it might have legacy URL aliases that need to be deleted.
        await (0, _legacy_url_aliases.deleteLegacyUrlAliases)({
          mappings: this._mappings,
          registry: this._registry,
          client: this.client,
          getIndexForType: this.getIndexForType.bind(this),
          type,
          id,
          ...(namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING) ? {
            namespaces: [],
            deleteBehavior: 'exclusive'
          } // delete legacy URL aliases for this type/ID for all spaces
          : {
            namespaces,
            deleteBehavior: 'inclusive'
          }) // delete legacy URL aliases for this type/ID for these specific spaces
        }).catch(err => {
          // The object has already been deleted, but we caught an error when attempting to delete aliases.
          // A consumer cannot attempt to delete the object again, so just log the error and swallow it.
          this._logger.error(`Unable to delete aliases when deleting an object: ${err.message}`);
        });
      }
      return {};
    }
    const deleteDocNotFound = body.result === 'not_found';
    // @ts-expect-error @elastic/elasticsearch doesn't declare error on DeleteResponse
    const deleteIndexNotFound = body.error && body.error.type === 'index_not_found_exception';
    if (deleteDocNotFound || deleteIndexNotFound) {
      // see "404s from missing index" above
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    throw new Error(`Unexpected Elasticsearch DELETE response: ${JSON.stringify({
      type,
      id,
      response: {
        body,
        statusCode
      }
    })}`);
  }

  /**
   * Performs initial checks on object type validity and flags multi-namespace objects for preflight checks by adding an `esRequestIndex`
   * @param objects SavedObjectsBulkDeleteObject[]
   * @returns array BulkDeleteExpectedBulkGetResult[]
   * @internal
   */
  presortObjectsByNamespaceType(objects) {
    let bulkGetRequestIndexCounter = 0;
    return objects.map(object => {
      const {
        type,
        id
      } = object;
      if (!this._allowedTypes.includes(type)) {
        return {
          tag: 'Left',
          value: {
            id,
            type,
            error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type))
          }
        };
      }
      const requiresNamespacesCheck = this._registry.isMultiNamespace(type);
      return {
        tag: 'Right',
        value: {
          type,
          id,
          ...(requiresNamespacesCheck && {
            esRequestIndex: bulkGetRequestIndexCounter++
          })
        }
      };
    });
  }

  /**
   * Fetch multi-namespace saved objects
   * @returns MgetResponse
   * @notes multi-namespace objects shared to more than one space require special handling. We fetch these docs to retrieve their namespaces.
   * @internal
   */
  async preflightCheckForBulkDelete(params) {
    const {
      expectedBulkGetResults,
      namespace
    } = params;
    const bulkGetMultiNamespaceDocs = expectedBulkGetResults.filter(_internal_utils.isRight).filter(({
      value
    }) => value.esRequestIndex !== undefined).map(({
      value: {
        type,
        id
      }
    }) => ({
      _id: this._serializer.generateRawId(namespace, type, id),
      _index: this.getIndexForType(type),
      _source: ['type', 'namespaces']
    }));
    const bulkGetMultiNamespaceDocsResponse = bulkGetMultiNamespaceDocs.length ? await this.client.mget({
      body: {
        docs: bulkGetMultiNamespaceDocs
      }
    }, {
      ignore: [404],
      meta: true
    }) : undefined;
    // fail fast if we can't verify a 404 response is from Elasticsearch
    if (bulkGetMultiNamespaceDocsResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode: bulkGetMultiNamespaceDocsResponse.statusCode,
      headers: bulkGetMultiNamespaceDocsResponse.headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
    }
    return bulkGetMultiNamespaceDocsResponse;
  }

  /**
   * @returns array of objects sorted by expected delete success or failure result
   * @internal
   */
  getExpectedBulkDeleteMultiNamespaceDocsResults(params) {
    const {
      expectedBulkGetResults,
      multiNamespaceDocsResponse,
      namespace,
      force
    } = params;
    let indexCounter = 0;
    const expectedBulkDeleteMultiNamespaceDocsResults = expectedBulkGetResults.map(expectedBulkGetResult => {
      if ((0, _internal_utils.isLeft)(expectedBulkGetResult)) {
        return {
          ...expectedBulkGetResult
        };
      }
      const {
        esRequestIndex: esBulkGetRequestIndex,
        id,
        type
      } = expectedBulkGetResult.value;
      let namespaces;
      if (esBulkGetRequestIndex !== undefined) {
        var _source$namespaces;
        const indexFound = (multiNamespaceDocsResponse === null || multiNamespaceDocsResponse === void 0 ? void 0 : multiNamespaceDocsResponse.statusCode) !== 404;
        const actualResult = indexFound ? multiNamespaceDocsResponse === null || multiNamespaceDocsResponse === void 0 ? void 0 : multiNamespaceDocsResponse.body.docs[esBulkGetRequestIndex] : undefined;
        const docFound = indexFound && isMgetDoc(actualResult) && actualResult.found;

        // return an error if the doc isn't found at all or the doc doesn't exist in the namespaces
        if (!docFound) {
          return {
            tag: 'Left',
            value: {
              id,
              type,
              error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
            }
          };
        }
        // the following check should be redundant since we're retrieving the docs from elasticsearch but we check just to make sure
        // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
        if (!this.rawDocExistsInNamespace(actualResult, namespace)) {
          return {
            tag: 'Left',
            value: {
              id,
              type,
              error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
            }
          };
        }
        // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
        namespaces = (_source$namespaces = actualResult._source.namespaces) !== null && _source$namespaces !== void 0 ? _source$namespaces : [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)];
        const useForce = force && force === true;
        // the document is shared to more than one space and can only be deleted by force.
        if (!useForce && (namespaces.length > 1 || namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING))) {
          return {
            tag: 'Left',
            value: {
              success: false,
              id,
              type,
              error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('Unable to delete saved object that exists in multiple namespaces, use the `force` option to delete it anyway'))
            }
          };
        }
      }
      // contains all objects that passed initial preflight checks, including single namespace objects that skipped the mget call
      // single namespace objects will have namespaces:undefined
      const expectedResult = {
        type,
        id,
        namespaces,
        esRequestIndex: indexCounter++
      };
      return {
        tag: 'Right',
        value: expectedResult
      };
    });
    return expectedBulkDeleteMultiNamespaceDocsResults;
  }

  /**
   *  {@inheritDoc ISavedObjectsRepository.bulkDelete}
   */
  async bulkDelete(objects, options = {}) {
    const {
      refresh = DEFAULT_REFRESH_SETTING,
      force
    } = options;
    const namespace = this.getCurrentNamespace(options.namespace);
    const expectedBulkGetResults = this.presortObjectsByNamespaceType(objects);
    if (expectedBulkGetResults.length === 0) {
      return {
        statuses: []
      };
    }
    const multiNamespaceDocsResponse = await this.preflightCheckForBulkDelete({
      expectedBulkGetResults,
      namespace
    });

    // First round of filtering (Left: object doesn't exist/doesn't exist in namespace, Right: good to proceed)
    const expectedBulkDeleteMultiNamespaceDocsResults = this.getExpectedBulkDeleteMultiNamespaceDocsResults({
      expectedBulkGetResults,
      multiNamespaceDocsResponse,
      namespace,
      force
    });
    if (this._securityExtension) {
      // Perform Auth Check (on both L/R, we'll deal with that later)
      const authObjects = expectedBulkDeleteMultiNamespaceDocsResults.map(element => {
        var _preflightResult$_sou, _preflightResult$_sou2;
        const index = element.value.esRequestIndex;
        const {
          type,
          id
        } = element.value;
        const preflightResult = index !== undefined ? multiNamespaceDocsResponse === null || multiNamespaceDocsResponse === void 0 ? void 0 : multiNamespaceDocsResponse.body.docs[index] : undefined;
        return {
          type,
          id,
          // @ts-expect-error MultiGetHit._source is optional
          existingNamespaces: (_preflightResult$_sou = preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$_sou2 = preflightResult._source) === null || _preflightResult$_sou2 === void 0 ? void 0 : _preflightResult$_sou2.namespaces) !== null && _preflightResult$_sou !== void 0 ? _preflightResult$_sou : []
        };
      });
      await this._securityExtension.authorizeBulkDelete({
        namespace,
        objects: authObjects
      });
    }

    // Filter valid objects
    const validObjects = expectedBulkDeleteMultiNamespaceDocsResults.filter(_internal_utils.isRight);
    if (validObjects.length === 0) {
      // We only have error results; return early to avoid potentially trying authZ checks for 0 types which would result in an exception.
      const savedObjects = expectedBulkDeleteMultiNamespaceDocsResults.filter(_internal_utils.isLeft).map(expectedResult => {
        return {
          ...expectedResult.value,
          success: false
        };
      });
      return {
        statuses: [...savedObjects]
      };
    }

    // Create the bulkDeleteParams
    const bulkDeleteParams = [];
    validObjects.map(expectedResult => {
      bulkDeleteParams.push({
        delete: {
          _id: this._serializer.generateRawId(namespace, expectedResult.value.type, expectedResult.value.id),
          _index: this.getIndexForType(expectedResult.value.type),
          ...(0, _internal_utils.getExpectedVersionProperties)(undefined)
        }
      });
    });
    const bulkDeleteResponse = bulkDeleteParams.length ? await this.client.bulk({
      refresh,
      body: bulkDeleteParams,
      require_alias: true
    }) : undefined;

    // extracted to ensure consistency in the error results returned
    let errorResult;
    const objectsToDeleteAliasesFor = [];
    const savedObjects = expectedBulkDeleteMultiNamespaceDocsResults.map(expectedResult => {
      if ((0, _internal_utils.isLeft)(expectedResult)) {
        return {
          ...expectedResult.value,
          success: false
        };
      }
      const {
        type,
        id,
        namespaces,
        esRequestIndex: esBulkDeleteRequestIndex
      } = expectedResult.value;
      // we assume this wouldn't happen but is needed to ensure type consistency
      if (bulkDeleteResponse === undefined) {
        throw new Error(`Unexpected error in bulkDelete saved objects: bulkDeleteResponse is undefined`);
      }
      const rawResponse = Object.values(bulkDeleteResponse.items[esBulkDeleteRequestIndex])[0];
      const error = (0, _internal_utils.getBulkOperationError)(type, id, rawResponse);
      if (error) {
        errorResult = {
          success: false,
          type,
          id,
          error
        };
        return errorResult;
      }
      if (rawResponse.result === 'not_found') {
        errorResult = {
          success: false,
          type,
          id,
          error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
        };
        return errorResult;
      }
      if (rawResponse.result === 'deleted') {
        // `namespaces` should only exist in the expectedResult.value if the type is multi-namespace.
        if (namespaces) {
          objectsToDeleteAliasesFor.push({
            type,
            id,
            ...(namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING) ? {
              namespaces: [],
              deleteBehavior: 'exclusive'
            } : {
              namespaces,
              deleteBehavior: 'inclusive'
            })
          });
        }
      }
      const successfulResult = {
        success: true,
        id,
        type
      };
      return successfulResult;
    });

    // Delete aliases if necessary, ensuring we don't have too many concurrent operations running.
    const mapper = async ({
      type,
      id,
      namespaces,
      deleteBehavior
    }) => {
      await (0, _legacy_url_aliases.deleteLegacyUrlAliases)({
        mappings: this._mappings,
        registry: this._registry,
        client: this.client,
        getIndexForType: this.getIndexForType.bind(this),
        type,
        id,
        namespaces,
        deleteBehavior
      }).catch(err => {
        this._logger.error(`Unable to delete aliases when deleting an object: ${err.message}`);
      });
    };
    await (0, _pMap.default)(objectsToDeleteAliasesFor, mapper, {
      concurrency: MAX_CONCURRENT_ALIAS_DELETIONS
    });
    return {
      statuses: [...savedObjects]
    };
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.deleteByNamespace}
   */
  async deleteByNamespace(namespace, options = {}) {
    // This is not exposed on the SOC; authorization and audit logging is handled by the Spaces plugin
    if (!namespace || typeof namespace !== 'string' || namespace === '*') {
      throw new TypeError(`namespace is required, and must be a string that is not equal to '*'`);
    }
    const allTypes = Object.keys((0, _coreSavedObjectsBaseServerInternal.getRootPropertiesObjects)(this._mappings));
    const typesToUpdate = [...allTypes.filter(type => !this._registry.isNamespaceAgnostic(type)), _coreSavedObjectsBaseServerInternal.LEGACY_URL_ALIAS_TYPE];

    // Construct kueryNode to filter legacy URL aliases (these space-agnostic objects do not use root-level "namespace/s" fields)
    const {
      buildNode
    } = esKuery.nodeTypes.function;
    const match1 = buildNode('is', `${_coreSavedObjectsBaseServerInternal.LEGACY_URL_ALIAS_TYPE}.targetNamespace`, namespace);
    const match2 = buildNode('not', buildNode('is', 'type', _coreSavedObjectsBaseServerInternal.LEGACY_URL_ALIAS_TYPE));
    const kueryNode = buildNode('or', [match1, match2]);
    const {
      body,
      statusCode,
      headers
    } = await this.client.updateByQuery({
      index: this.getIndicesForTypes(typesToUpdate),
      refresh: options.refresh,
      body: {
        script: {
          source: `
              if (!ctx._source.containsKey('namespaces')) {
                ctx.op = "delete";
              } else {
                ctx._source['namespaces'].removeAll(Collections.singleton(params['namespace']));
                if (ctx._source['namespaces'].empty) {
                  ctx.op = "delete";
                }
              }
            `,
          lang: 'painless',
          params: {
            namespace
          }
        },
        conflicts: 'proceed',
        ...(0, _search_dsl.getSearchDsl)(this._mappings, this._registry, {
          namespaces: [namespace],
          type: typesToUpdate,
          kueryNode
        })
      }
    }, {
      ignore: [404],
      meta: true
    });
    // throw if we can't verify a 404 response is from Elasticsearch
    if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode,
      headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
    }
    return body;
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.find}
   */
  async find(options, internalOptions = {}) {
    var _this$_securityExtens5, _authorizationResult3;
    let namespaces;
    const {
      disableExtensions
    } = internalOptions;
    if (disableExtensions || !this._spacesExtension) {
      var _options$namespaces;
      namespaces = (_options$namespaces = options.namespaces) !== null && _options$namespaces !== void 0 ? _options$namespaces : [_coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING];
      // If the consumer specified `namespaces: []`, throw a Bad Request error
      if (namespaces.length === 0) throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.namespaces cannot be an empty array');
    }
    const {
      search,
      defaultSearchOperator = 'OR',
      searchFields,
      rootSearchFields,
      hasReference,
      hasReferenceOperator,
      hasNoReference,
      hasNoReferenceOperator,
      page = _coreSavedObjectsUtilsServer.FIND_DEFAULT_PAGE,
      perPage = _coreSavedObjectsUtilsServer.FIND_DEFAULT_PER_PAGE,
      pit,
      searchAfter,
      sortField,
      sortOrder,
      fields,
      type,
      filter,
      preference,
      aggs,
      migrationVersionCompatibility
    } = options;
    if (!type) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.type must be a string or an array of strings');
    } else if (preference !== null && preference !== void 0 && preference.length && pit) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.preference must be excluded when options.pit is used');
    }
    const types = Array.isArray(type) ? type : [type];
    const allowedTypes = types.filter(t => this._allowedTypes.includes(t));
    if (allowedTypes.length === 0) {
      return _coreSavedObjectsUtilsServer.SavedObjectsUtils.createEmptyFindResponse(options);
    }
    if (searchFields && !Array.isArray(searchFields)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.searchFields must be an array');
    }
    if (fields && !Array.isArray(fields)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.fields must be an array');
    }
    let kueryNode;
    if (filter) {
      try {
        kueryNode = (0, _filter_utils.validateConvertFilterToKueryNode)(allowedTypes, filter, this._mappings);
      } catch (e) {
        if (e.name === 'KQLSyntaxError') {
          throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(`KQLSyntaxError: ${e.message}`);
        } else {
          throw e;
        }
      }
    }
    let aggsObject;
    if (aggs) {
      try {
        aggsObject = (0, _aggregations.validateAndConvertAggregations)(allowedTypes, aggs, this._mappings);
      } catch (e) {
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(`Invalid aggregation: ${e.message}`);
      }
    }
    if (!disableExtensions && this._spacesExtension) {
      try {
        namespaces = await this._spacesExtension.getSearchableNamespaces(options.namespaces);
      } catch (err) {
        if (_boom.default.isBoom(err) && err.output.payload.statusCode === 403) {
          // The user is not authorized to access any space, return an empty response.
          return _coreSavedObjectsUtilsServer.SavedObjectsUtils.createEmptyFindResponse(options);
        }
        throw err;
      }
      if (namespaces.length === 0) {
        // The user is authorized to access *at least one space*, but not any of the spaces they requested; return an empty response.
        return _coreSavedObjectsUtilsServer.SavedObjectsUtils.createEmptyFindResponse(options);
      }
    }

    // We have to first perform an initial authorization check so that we can construct the search DSL accordingly
    const spacesToAuthorize = new Set(namespaces);
    const typesToAuthorize = new Set(types);
    let typeToNamespacesMap;
    let authorizationResult;
    if (!disableExtensions && this._securityExtension) {
      var _authorizationResult, _authorizationResult2;
      authorizationResult = await this._securityExtension.authorizeFind({
        namespaces: spacesToAuthorize,
        types: typesToAuthorize
      });
      if (((_authorizationResult = authorizationResult) === null || _authorizationResult === void 0 ? void 0 : _authorizationResult.status) === 'unauthorized') {
        // If the user is unauthorized to find *anything* they requested, return an empty response
        return _coreSavedObjectsUtilsServer.SavedObjectsUtils.createEmptyFindResponse(options);
      }
      if (((_authorizationResult2 = authorizationResult) === null || _authorizationResult2 === void 0 ? void 0 : _authorizationResult2.status) === 'partially_authorized') {
        typeToNamespacesMap = new Map();
        for (const [objType, entry] of authorizationResult.typeMap) {
          if (!entry.find) continue;
          // This ensures that the query DSL can filter only for object types that the user is authorized to access for a given space
          const {
            authorizedSpaces,
            isGloballyAuthorized
          } = entry.find;
          typeToNamespacesMap.set(objType, isGloballyAuthorized ? namespaces : authorizedSpaces);
        }
      }
    }
    const esOptions = {
      // If `pit` is provided, we drop the `index`, otherwise ES returns 400.
      index: pit ? undefined : this.getIndicesForTypes(allowedTypes),
      // If `searchAfter` is provided, we drop `from` as it will not be used for pagination.
      from: searchAfter ? undefined : perPage * (page - 1),
      _source: (0, _included_fields.includedFields)(allowedTypes, fields),
      preference,
      rest_total_hits_as_int: true,
      size: perPage,
      body: {
        size: perPage,
        seq_no_primary_term: true,
        from: perPage * (page - 1),
        _source: (0, _included_fields.includedFields)(allowedTypes, fields),
        ...(aggsObject ? {
          aggs: aggsObject
        } : {}),
        ...(0, _search_dsl.getSearchDsl)(this._mappings, this._registry, {
          search,
          defaultSearchOperator,
          searchFields,
          pit,
          rootSearchFields,
          type: allowedTypes,
          searchAfter,
          sortField,
          sortOrder,
          namespaces,
          typeToNamespacesMap,
          // If defined, this takes precedence over the `type` and `namespaces` fields
          hasReference,
          hasReferenceOperator,
          hasNoReference,
          hasNoReferenceOperator,
          kueryNode
        })
      }
    };
    const {
      body,
      statusCode,
      headers
    } = await this.client.search(esOptions, {
      ignore: [404],
      meta: true
    });
    if (statusCode === 404) {
      if (!(0, _coreElasticsearchServerInternal.isSupportedEsServer)(headers)) {
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
      }
      // 404 is only possible here if the index is missing, which
      // we don't want to leak, see "404s from missing index" above
      return _coreSavedObjectsUtilsServer.SavedObjectsUtils.createEmptyFindResponse(options);
    }
    const result = {
      ...(body.aggregations ? {
        aggregations: body.aggregations
      } : {}),
      page,
      per_page: perPage,
      total: body.hits.total,
      saved_objects: body.hits.hits.map(hit => ({
        // @ts-expect-error @elastic/elasticsearch _source is optional
        ...this._rawToSavedObject(hit, {
          migrationVersionCompatibility
        }),
        score: hit._score,
        sort: hit.sort
      })),
      pit_id: body.pit_id
    };
    if (disableExtensions) {
      return result;
    }

    // Now that we have a full set of results with all existing namespaces for each object,
    // we need an updated authorization type map to pass on to the redact method
    const redactTypeMap = await ((_this$_securityExtens5 = this._securityExtension) === null || _this$_securityExtens5 === void 0 ? void 0 : _this$_securityExtens5.getFindRedactTypeMap({
      previouslyCheckedNamespaces: spacesToAuthorize,
      objects: result.saved_objects.map(obj => {
        var _obj$namespaces;
        return {
          type: obj.type,
          id: obj.id,
          existingNamespaces: (_obj$namespaces = obj.namespaces) !== null && _obj$namespaces !== void 0 ? _obj$namespaces : []
        };
      })
    }));
    return this.optionallyDecryptAndRedactBulkResult(result, redactTypeMap !== null && redactTypeMap !== void 0 ? redactTypeMap : (_authorizationResult3 = authorizationResult) === null || _authorizationResult3 === void 0 ? void 0 : _authorizationResult3.typeMap // If the redact type map is valid, use that one; otherwise, fall back to the authorization check
    );
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.bulkGet}
   */
  async bulkGet(objects = [], options = {}) {
    var _this$_securityExtens6;
    const namespace = this.getCurrentNamespace(options.namespace);
    const {
      migrationVersionCompatibility
    } = options;
    if (objects.length === 0) {
      return {
        saved_objects: []
      };
    }
    let availableSpacesPromise;
    const getAvailableSpaces = async spacesExtension => {
      if (!availableSpacesPromise) {
        availableSpacesPromise = spacesExtension.getSearchableNamespaces([_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING]).catch(err => {
          if (_boom.default.isBoom(err) && err.output.payload.statusCode === 403) {
            // the user doesn't have access to any spaces; return the current space ID and allow the SOR authZ check to fail
            return [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)];
          } else {
            throw err;
          }
        });
      }
      return availableSpacesPromise;
    };
    let bulkGetRequestIndexCounter = 0;
    const expectedBulkGetResults = await Promise.all(objects.map(async object => {
      var _namespaces;
      const {
        type,
        id,
        fields
      } = object;
      let error;
      if (!this._allowedTypes.includes(type)) {
        error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
      } else {
        try {
          this.validateObjectNamespaces(type, id, object.namespaces);
        } catch (e) {
          error = e;
        }
      }
      if (error) {
        return {
          tag: 'Left',
          value: {
            id,
            type,
            error: errorContent(error)
          }
        };
      }
      let namespaces = object.namespaces;
      if (this._spacesExtension && (_namespaces = namespaces) !== null && _namespaces !== void 0 && _namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING)) {
        namespaces = await getAvailableSpaces(this._spacesExtension);
      }
      return {
        tag: 'Right',
        value: {
          type,
          id,
          fields,
          namespaces,
          esRequestIndex: bulkGetRequestIndexCounter++
        }
      };
    }));
    const validObjects = expectedBulkGetResults.filter(_internal_utils.isRight);
    if (validObjects.length === 0) {
      // We only have error results; return early to avoid potentially trying authZ checks for 0 types which would result in an exception.
      return {
        // Technically the returned array should only contain SavedObject results, but for errors this is not true (we cast to 'any' below)
        saved_objects: expectedBulkGetResults.map(({
          value
        }) => value)
      };
    }
    const getNamespaceId = namespaces => namespaces !== undefined ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceStringToId(namespaces[0]) : namespace;
    const bulkGetDocs = validObjects.map(({
      value: {
        type,
        id,
        fields,
        namespaces
      }
    }) => ({
      _id: this._serializer.generateRawId(getNamespaceId(namespaces), type, id),
      // the namespace prefix is only used for single-namespace object types
      _index: this.getIndexForType(type),
      _source: {
        includes: (0, _included_fields.includedFields)(type, fields)
      }
    }));
    const bulkGetResponse = bulkGetDocs.length ? await this.client.mget({
      body: {
        docs: bulkGetDocs
      }
    }, {
      ignore: [404],
      meta: true
    }) : undefined;
    // fail fast if we can't verify a 404 is from Elasticsearch
    if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode: bulkGetResponse.statusCode,
      headers: bulkGetResponse.headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
    }
    const authObjects = [];
    const result = {
      saved_objects: expectedBulkGetResults.map(expectedResult => {
        var _doc$_source$namespac, _doc$_source;
        if ((0, _internal_utils.isLeft)(expectedResult)) {
          const {
            type,
            id
          } = expectedResult.value;
          authObjects.push({
            type,
            id,
            existingNamespaces: [],
            error: true
          });
          return expectedResult.value;
        }
        const {
          type,
          id,
          // set to default namespaces value for `rawDocExistsInNamespaces` check below
          namespaces = [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)],
          esRequestIndex
        } = expectedResult.value;
        const doc = bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[esRequestIndex];

        // @ts-expect-error MultiGetHit._source is optional
        const docNotFound = !(doc !== null && doc !== void 0 && doc.found) || !this.rawDocExistsInNamespaces(doc, namespaces);
        authObjects.push({
          type,
          id,
          objectNamespaces: namespaces,
          // @ts-expect-error MultiGetHit._source is optional
          existingNamespaces: (_doc$_source$namespac = doc === null || doc === void 0 ? void 0 : (_doc$_source = doc._source) === null || _doc$_source === void 0 ? void 0 : _doc$_source.namespaces) !== null && _doc$_source$namespac !== void 0 ? _doc$_source$namespac : [],
          error: docNotFound
        });
        if (docNotFound) {
          return {
            id,
            type,
            error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
          };
        }

        // @ts-expect-error MultiGetHit._source is optional
        return (0, _internal_utils.getSavedObjectFromSource)(this._registry, type, id, doc, {
          migrationVersionCompatibility
        });
      })
    };
    const authorizationResult = await ((_this$_securityExtens6 = this._securityExtension) === null || _this$_securityExtens6 === void 0 ? void 0 : _this$_securityExtens6.authorizeBulkGet({
      namespace,
      objects: authObjects
    }));
    return this.optionallyDecryptAndRedactBulkResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap);
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.bulkResolve}
   */
  async bulkResolve(objects, options = {}) {
    const namespace = this.getCurrentNamespace(options.namespace);
    const {
      resolved_objects: bulkResults
    } = await (0, _internal_bulk_resolve.internalBulkResolve)({
      registry: this._registry,
      allowedTypes: this._allowedTypes,
      client: this.client,
      serializer: this._serializer,
      getIndexForType: this.getIndexForType.bind(this),
      incrementCounterInternal: this.incrementCounterInternal.bind(this),
      encryptionExtension: this._encryptionExtension,
      securityExtension: this._securityExtension,
      objects,
      options: {
        ...options,
        namespace
      }
    });
    const resolvedObjects = bulkResults.map(result => {
      // extract payloads from saved object errors
      if ((0, _internal_bulk_resolve.isBulkResolveError)(result)) {
        const errorResult = result;
        const {
          type,
          id,
          error
        } = errorResult;
        return {
          saved_object: {
            type,
            id,
            error: errorContent(error)
          },
          outcome: 'exactMatch'
        };
      }
      return result;
    });
    return {
      resolved_objects: resolvedObjects
    };
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.get}
   */
  async get(type, id, options = {}) {
    var _this$_securityExtens7, _body$_source$namespa, _body$_source;
    const namespace = this.getCurrentNamespace(options.namespace);
    const {
      migrationVersionCompatibility
    } = options;
    if (!this._allowedTypes.includes(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    const {
      body,
      statusCode,
      headers
    } = await this.client.get({
      id: this._serializer.generateRawId(namespace, type, id),
      index: this.getIndexForType(type)
    }, {
      ignore: [404],
      meta: true
    });
    const indexNotFound = statusCode === 404;
    // check if we have the elasticsearch header when index is not found and, if we do, ensure it is from Elasticsearch
    if (indexNotFound && !(0, _coreElasticsearchServerInternal.isSupportedEsServer)(headers)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(type, id);
    }
    const objectNotFound = !isFoundGetResponse(body) || indexNotFound || !this.rawDocExistsInNamespace(body, namespace);
    const authorizationResult = await ((_this$_securityExtens7 = this._securityExtension) === null || _this$_securityExtens7 === void 0 ? void 0 : _this$_securityExtens7.authorizeGet({
      namespace,
      object: {
        type,
        id,
        existingNamespaces: (_body$_source$namespa = body === null || body === void 0 ? void 0 : (_body$_source = body._source) === null || _body$_source === void 0 ? void 0 : _body$_source.namespaces) !== null && _body$_source$namespa !== void 0 ? _body$_source$namespa : []
      },
      objectNotFound
    }));
    if (objectNotFound) {
      // see "404s from missing index" above
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    const result = (0, _internal_utils.getSavedObjectFromSource)(this._registry, type, id, body, {
      migrationVersionCompatibility
    });
    return this.optionallyDecryptAndRedactSingleResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap);
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.resolve}
   */
  async resolve(type, id, options = {}) {
    const namespace = this.getCurrentNamespace(options.namespace);
    const {
      resolved_objects: bulkResults
    } = await (0, _internal_bulk_resolve.internalBulkResolve)({
      registry: this._registry,
      allowedTypes: this._allowedTypes,
      client: this.client,
      serializer: this._serializer,
      getIndexForType: this.getIndexForType.bind(this),
      incrementCounterInternal: this.incrementCounterInternal.bind(this),
      encryptionExtension: this._encryptionExtension,
      securityExtension: this._securityExtension,
      objects: [{
        type,
        id
      }],
      options: {
        ...options,
        namespace
      }
    });
    const [result] = bulkResults;
    if ((0, _internal_bulk_resolve.isBulkResolveError)(result)) {
      throw result.error;
    }
    return result;
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.update}
   */
  async update(type, id, attributes, options = {}) {
    var _preflightResult$save2, _preflightResult6, _this$_securityExtens8, _preflightResult7, _preflightResult8, _preflightResult9, _body$get$_source, _body$get;
    const namespace = this.getCurrentNamespace(options.namespace);
    if (!this._allowedTypes.includes(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    if (!id) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('id cannot be empty'); // prevent potentially upserting a saved object with an empty ID
    }

    const {
      version,
      references,
      upsert,
      refresh = DEFAULT_REFRESH_SETTING,
      retryOnConflict = version ? 0 : DEFAULT_RETRY_COUNT
    } = options;
    let preflightResult;
    if (this._registry.isMultiNamespace(type)) {
      preflightResult = await this.preflightCheckNamespaces({
        type,
        id,
        namespace
      });
    }
    const existingNamespaces = (_preflightResult$save2 = (_preflightResult6 = preflightResult) === null || _preflightResult6 === void 0 ? void 0 : _preflightResult6.savedObjectNamespaces) !== null && _preflightResult$save2 !== void 0 ? _preflightResult$save2 : [];
    const authorizationResult = await ((_this$_securityExtens8 = this._securityExtension) === null || _this$_securityExtens8 === void 0 ? void 0 : _this$_securityExtens8.authorizeUpdate({
      namespace,
      object: {
        type,
        id,
        existingNamespaces
      }
    }));
    if (((_preflightResult7 = preflightResult) === null || _preflightResult7 === void 0 ? void 0 : _preflightResult7.checkResult) === 'found_outside_namespace' || !upsert && ((_preflightResult8 = preflightResult) === null || _preflightResult8 === void 0 ? void 0 : _preflightResult8.checkResult) === 'not_found') {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    if (upsert && ((_preflightResult9 = preflightResult) === null || _preflightResult9 === void 0 ? void 0 : _preflightResult9.checkResult) === 'not_found') {
      // If an upsert would result in the creation of a new object, we need to check for alias conflicts too.
      // This takes an extra round trip to Elasticsearch, but this won't happen often.
      // TODO: improve performance by combining these into a single preflight check
      await this.preflightCheckForUpsertAliasConflict(type, id, namespace);
    }
    const time = (0, _internal_utils.getCurrentTime)();
    let rawUpsert;
    // don't include upsert if the object already exists; ES doesn't allow upsert in combination with version properties
    if (upsert && (!preflightResult || preflightResult.checkResult === 'not_found')) {
      let savedObjectNamespace;
      let savedObjectNamespaces;
      if (this._registry.isSingleNamespace(type) && namespace) {
        savedObjectNamespace = namespace;
      } else if (this._registry.isMultiNamespace(type)) {
        savedObjectNamespaces = preflightResult.savedObjectNamespaces;
      }
      const migrated = this._migrator.migrateDocument({
        id,
        type,
        ...(savedObjectNamespace && {
          namespace: savedObjectNamespace
        }),
        ...(savedObjectNamespaces && {
          namespaces: savedObjectNamespaces
        }),
        attributes: {
          ...(await this.optionallyEncryptAttributes(type, id, namespace, upsert))
        },
        updated_at: time
      });
      rawUpsert = this._serializer.savedObjectToRaw(migrated);
    }
    const doc = {
      [type]: await this.optionallyEncryptAttributes(type, id, namespace, attributes),
      updated_at: time,
      ...(Array.isArray(references) && {
        references
      })
    };
    const body = await this.client.update({
      id: this._serializer.generateRawId(namespace, type, id),
      index: this.getIndexForType(type),
      ...(0, _internal_utils.getExpectedVersionProperties)(version),
      refresh,
      retry_on_conflict: retryOnConflict,
      body: {
        doc,
        ...(rawUpsert && {
          upsert: rawUpsert._source
        })
      },
      _source_includes: ['namespace', 'namespaces', 'originId'],
      require_alias: true
    }).catch(err => {
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isEsUnavailableError(err)) {
        throw err;
      }
      if (_coreSavedObjectsServer.SavedObjectsErrorHelpers.isNotFoundError(err)) {
        // see "404s from missing index" above
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
      }
      throw err;
    });
    const {
      originId
    } = (_body$get$_source = (_body$get = body.get) === null || _body$get === void 0 ? void 0 : _body$get._source) !== null && _body$get$_source !== void 0 ? _body$get$_source : {};
    let namespaces = [];
    if (!this._registry.isNamespaceAgnostic(type)) {
      var _body$get$_source$nam, _body$get2, _body$get3;
      namespaces = (_body$get$_source$nam = (_body$get2 = body.get) === null || _body$get2 === void 0 ? void 0 : _body$get2._source.namespaces) !== null && _body$get$_source$nam !== void 0 ? _body$get$_source$nam : [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString((_body$get3 = body.get) === null || _body$get3 === void 0 ? void 0 : _body$get3._source.namespace)];
    }
    const result = {
      id,
      type,
      updated_at: time,
      version: (0, _coreSavedObjectsBaseServerInternal.encodeHitVersion)(body),
      namespaces,
      ...(originId && {
        originId
      }),
      references,
      attributes
    };
    return this.optionallyDecryptAndRedactSingleResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, attributes);
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.collectMultiNamespaceReferences}
   */
  async collectMultiNamespaceReferences(objects, options = {}) {
    const namespace = this.getCurrentNamespace(options.namespace);
    return (0, _collect_multi_namespace_references.collectMultiNamespaceReferences)({
      registry: this._registry,
      allowedTypes: this._allowedTypes,
      client: this.client,
      serializer: this._serializer,
      getIndexForType: this.getIndexForType.bind(this),
      createPointInTimeFinder: this.createPointInTimeFinder.bind(this),
      securityExtension: this._securityExtension,
      objects,
      options: {
        ...options,
        namespace
      }
    });
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.updateObjectsSpaces}
   */
  async updateObjectsSpaces(objects, spacesToAdd, spacesToRemove, options = {}) {
    const namespace = this.getCurrentNamespace(options.namespace);
    return (0, _update_objects_spaces.updateObjectsSpaces)({
      mappings: this._mappings,
      registry: this._registry,
      allowedTypes: this._allowedTypes,
      client: this.client,
      serializer: this._serializer,
      logger: this._logger,
      getIndexForType: this.getIndexForType.bind(this),
      securityExtension: this._securityExtension,
      objects,
      spacesToAdd,
      spacesToRemove,
      options: {
        ...options,
        namespace
      }
    });
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.bulkUpdate}
   */
  async bulkUpdate(objects, options = {}) {
    var _this$_securityExtens9;
    const namespace = this.getCurrentNamespace(options.namespace);
    const time = (0, _internal_utils.getCurrentTime)();
    let bulkGetRequestIndexCounter = 0;
    const expectedBulkGetResults = objects.map(object => {
      const {
        type,
        id,
        attributes,
        references,
        version,
        namespace: objectNamespace
      } = object;
      let error;
      if (!this._allowedTypes.includes(type)) {
        error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
      } else {
        try {
          if (objectNamespace === _coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING) {
            error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"namespace" cannot be "*"');
          }
        } catch (e) {
          error = e;
        }
      }
      if (error) {
        return {
          tag: 'Left',
          value: {
            id,
            type,
            error: errorContent(error)
          }
        };
      }
      const documentToSave = {
        [type]: attributes,
        updated_at: time,
        ...(Array.isArray(references) && {
          references
        })
      };
      const requiresNamespacesCheck = this._registry.isMultiNamespace(object.type);
      return {
        tag: 'Right',
        value: {
          type,
          id,
          version,
          documentToSave,
          objectNamespace,
          ...(requiresNamespacesCheck && {
            esRequestIndex: bulkGetRequestIndexCounter++
          })
        }
      };
    });
    const validObjects = expectedBulkGetResults.filter(_internal_utils.isRight);
    if (validObjects.length === 0) {
      // We only have error results; return early to avoid potentially trying authZ checks for 0 types which would result in an exception.
      return {
        // Technically the returned array should only contain SavedObject results, but for errors this is not true (we cast to 'any' below)
        saved_objects: expectedBulkGetResults.map(({
          value
        }) => value)
      };
    }

    // `objectNamespace` is a namespace string, while `namespace` is a namespace ID.
    // The object namespace string, if defined, will supersede the operation's namespace ID.
    const namespaceString = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace);
    const getNamespaceId = objectNamespace => objectNamespace !== undefined ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceStringToId(objectNamespace) : namespace;
    const getNamespaceString = objectNamespace => objectNamespace !== null && objectNamespace !== void 0 ? objectNamespace : namespaceString;
    const bulkGetDocs = validObjects.filter(({
      value
    }) => value.esRequestIndex !== undefined).map(({
      value: {
        type,
        id,
        objectNamespace
      }
    }) => ({
      _id: this._serializer.generateRawId(getNamespaceId(objectNamespace), type, id),
      _index: this.getIndexForType(type),
      _source: ['type', 'namespaces']
    }));
    const bulkGetResponse = bulkGetDocs.length ? await this.client.mget({
      body: {
        docs: bulkGetDocs
      }
    }, {
      ignore: [404],
      meta: true
    }) : undefined;
    // fail fast if we can't verify a 404 response is from Elasticsearch
    if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode: bulkGetResponse.statusCode,
      headers: bulkGetResponse.headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
    }
    const authObjects = validObjects.map(element => {
      var _preflightResult$_sou3, _preflightResult$_sou4;
      const {
        type,
        id,
        objectNamespace,
        esRequestIndex: index
      } = element.value;
      const preflightResult = index !== undefined ? bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[index] : undefined;
      return {
        type,
        id,
        objectNamespace,
        // @ts-expect-error MultiGetHit._source is optional
        existingNamespaces: (_preflightResult$_sou3 = preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$_sou4 = preflightResult._source) === null || _preflightResult$_sou4 === void 0 ? void 0 : _preflightResult$_sou4.namespaces) !== null && _preflightResult$_sou3 !== void 0 ? _preflightResult$_sou3 : []
      };
    });
    const authorizationResult = await ((_this$_securityExtens9 = this._securityExtension) === null || _this$_securityExtens9 === void 0 ? void 0 : _this$_securityExtens9.authorizeBulkUpdate({
      namespace,
      objects: authObjects
    }));
    let bulkUpdateRequestIndexCounter = 0;
    const bulkUpdateParams = [];
    const expectedBulkUpdateResults = await Promise.all(expectedBulkGetResults.map(async expectedBulkGetResult => {
      if ((0, _internal_utils.isLeft)(expectedBulkGetResult)) {
        return expectedBulkGetResult;
      }
      const {
        esRequestIndex,
        id,
        type,
        version,
        documentToSave,
        objectNamespace
      } = expectedBulkGetResult.value;
      let namespaces;
      let versionProperties;
      if (esRequestIndex !== undefined) {
        var _source$namespaces2;
        const indexFound = (bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.statusCode) !== 404;
        const actualResult = indexFound ? bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[esRequestIndex] : undefined;
        const docFound = indexFound && isMgetDoc(actualResult) && actualResult.found;
        if (!docFound ||
        // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
        !this.rawDocExistsInNamespace(actualResult, getNamespaceId(objectNamespace))) {
          return {
            tag: 'Left',
            value: {
              id,
              type,
              error: errorContent(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
            }
          };
        }
        // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
        namespaces = (_source$namespaces2 = actualResult._source.namespaces) !== null && _source$namespaces2 !== void 0 ? _source$namespaces2 : [
        // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
        _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(actualResult._source.namespace)];
        versionProperties = (0, _internal_utils.getExpectedVersionProperties)(version);
      } else {
        if (this._registry.isSingleNamespace(type)) {
          // if `objectNamespace` is undefined, fall back to `options.namespace`
          namespaces = [getNamespaceString(objectNamespace)];
        }
        versionProperties = (0, _internal_utils.getExpectedVersionProperties)(version);
      }
      const expectedResult = {
        type,
        id,
        namespaces,
        esRequestIndex: bulkUpdateRequestIndexCounter++,
        documentToSave: expectedBulkGetResult.value.documentToSave
      };
      bulkUpdateParams.push({
        update: {
          _id: this._serializer.generateRawId(getNamespaceId(objectNamespace), type, id),
          _index: this.getIndexForType(type),
          ...versionProperties
        }
      }, {
        doc: {
          ...documentToSave,
          [type]: await this.optionallyEncryptAttributes(type, id, objectNamespace || namespace, documentToSave[type])
        }
      });
      return {
        tag: 'Right',
        value: expectedResult
      };
    }));
    const {
      refresh = DEFAULT_REFRESH_SETTING
    } = options;
    const bulkUpdateResponse = bulkUpdateParams.length ? await this.client.bulk({
      refresh,
      body: bulkUpdateParams,
      _source_includes: ['originId'],
      require_alias: true
    }) : undefined;
    const result = {
      saved_objects: expectedBulkUpdateResults.map(expectedResult => {
        var _bulkUpdateResponse$i;
        if ((0, _internal_utils.isLeft)(expectedResult)) {
          return expectedResult.value;
        }
        const {
          type,
          id,
          namespaces,
          documentToSave,
          esRequestIndex
        } = expectedResult.value;
        const response = (_bulkUpdateResponse$i = bulkUpdateResponse === null || bulkUpdateResponse === void 0 ? void 0 : bulkUpdateResponse.items[esRequestIndex]) !== null && _bulkUpdateResponse$i !== void 0 ? _bulkUpdateResponse$i : {};
        const rawResponse = Object.values(response)[0];
        const error = (0, _internal_utils.getBulkOperationError)(type, id, rawResponse);
        if (error) {
          return {
            type,
            id,
            error
          };
        }

        // When a bulk update operation is completed, any fields specified in `_sourceIncludes` will be found in the "get" value of the
        // returned object. We need to retrieve the `originId` if it exists so we can return it to the consumer.
        const {
          _seq_no: seqNo,
          _primary_term: primaryTerm,
          get
        } = rawResponse;

        // eslint-disable-next-line @typescript-eslint/naming-convention
        const {
          [type]: attributes,
          references,
          updated_at
        } = documentToSave;
        const {
          originId
        } = get._source;
        return {
          id,
          type,
          ...(namespaces && {
            namespaces
          }),
          ...(originId && {
            originId
          }),
          updated_at,
          version: (0, _coreSavedObjectsBaseServerInternal.encodeVersion)(seqNo, primaryTerm),
          attributes,
          references
        };
      })
    };
    return this.optionallyDecryptAndRedactBulkResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, objects);
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.removeReferencesTo}
   */
  async removeReferencesTo(type, id, options = {}) {
    var _this$_securityExtens10, _body$failures;
    const namespace = this.getCurrentNamespace(options.namespace);
    const {
      refresh = true
    } = options;
    await ((_this$_securityExtens10 = this._securityExtension) === null || _this$_securityExtens10 === void 0 ? void 0 : _this$_securityExtens10.authorizeRemoveReferences({
      namespace,
      object: {
        type,
        id
      }
    }));
    const allTypes = this._registry.getAllTypes().map(t => t.name);

    // we need to target all SO indices as all types of objects may have references to the given SO.
    const targetIndices = this.getIndicesForTypes(allTypes);
    const {
      body,
      statusCode,
      headers
    } = await this.client.updateByQuery({
      index: targetIndices,
      refresh,
      body: {
        script: {
          source: `
              if (ctx._source.containsKey('references')) {
                def items_to_remove = [];
                for (item in ctx._source.references) {
                  if ( (item['type'] == params['type']) && (item['id'] == params['id']) ) {
                    items_to_remove.add(item);
                  }
                }
                ctx._source.references.removeAll(items_to_remove);
              }
            `,
          params: {
            type,
            id
          },
          lang: 'painless'
        },
        conflicts: 'proceed',
        ...(0, _search_dsl.getSearchDsl)(this._mappings, this._registry, {
          namespaces: namespace ? [namespace] : undefined,
          type: allTypes,
          hasReference: {
            type,
            id
          }
        })
      }
    }, {
      ignore: [404],
      meta: true
    });
    // fail fast if we can't verify a 404 is from Elasticsearch
    if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode,
      headers
    })) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(type, id);
    }
    if ((_body$failures = body.failures) !== null && _body$failures !== void 0 && _body$failures.length) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id, `${body.failures.length} references could not be removed`);
    }
    return {
      updated: body.updated
    };
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.incrementCounter}
   */
  async incrementCounter(type, id, counterFields, options) {
    // This is not exposed on the SOC, there are no authorization or audit logging checks
    if (typeof type !== 'string') {
      throw new Error('"type" argument must be a string');
    }
    const isArrayOfCounterFields = Array.isArray(counterFields) && counterFields.every(field => typeof field === 'string' || (0, _lodash.isObject)(field) && typeof field.fieldName === 'string');
    if (!isArrayOfCounterFields) {
      throw new Error('"counterFields" argument must be of type Array<string | { incrementBy?: number; fieldName: string }>');
    }
    if (!this._allowedTypes.includes(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type);
    }
    return this.incrementCounterInternal(type, id, counterFields, options);
  }

  /** @internal incrementCounter function that is used internally and bypasses validation checks. */
  async incrementCounterInternal(type, id, counterFields, options = {}) {
    var _body$get$_source2, _body$get4, _body$get$_source$ref, _body$get5, _body$get6;
    const {
      migrationVersion,
      typeMigrationVersion,
      refresh = DEFAULT_REFRESH_SETTING,
      initialize = false,
      upsertAttributes,
      managed
    } = options;
    if (!id) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('id cannot be empty'); // prevent potentially upserting a saved object with an empty ID
    }

    const normalizedCounterFields = counterFields.map(counterField => {
      /**
       * no counterField configs provided, instead a field name string was passed.
       * ie `incrementCounter(so_type, id, ['my_field_name'])`
       * Using the default of incrementing by 1
       */
      if (typeof counterField === 'string') {
        return {
          fieldName: counterField,
          incrementBy: initialize ? 0 : 1
        };
      }
      const {
        incrementBy = 1,
        fieldName
      } = counterField;
      return {
        fieldName,
        incrementBy: initialize ? 0 : incrementBy
      };
    });
    const namespace = (0, _internal_utils.normalizeNamespace)(options.namespace);
    const time = (0, _internal_utils.getCurrentTime)();
    let savedObjectNamespace;
    let savedObjectNamespaces;
    if (this._registry.isSingleNamespace(type) && namespace) {
      savedObjectNamespace = namespace;
    } else if (this._registry.isMultiNamespace(type)) {
      // note: this check throws an error if the object is found but does not exist in this namespace
      const preflightResult = await this.preflightCheckNamespaces({
        type,
        id,
        namespace
      });
      if (preflightResult.checkResult === 'found_outside_namespace') {
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id);
      }
      if (preflightResult.checkResult === 'not_found') {
        // If an upsert would result in the creation of a new object, we need to check for alias conflicts too.
        // This takes an extra round trip to Elasticsearch, but this won't happen often.
        // TODO: improve performance by combining these into a single preflight check
        await this.preflightCheckForUpsertAliasConflict(type, id, namespace);
      }
      savedObjectNamespaces = preflightResult.savedObjectNamespaces;
    }

    // attributes: { [counterFieldName]: incrementBy },
    const migrated = this._migrator.migrateDocument({
      id,
      type,
      ...(savedObjectNamespace && {
        namespace: savedObjectNamespace
      }),
      ...(savedObjectNamespaces && {
        namespaces: savedObjectNamespaces
      }),
      attributes: {
        ...(upsertAttributes !== null && upsertAttributes !== void 0 ? upsertAttributes : {}),
        ...normalizedCounterFields.reduce((acc, counterField) => {
          const {
            fieldName,
            incrementBy
          } = counterField;
          acc[fieldName] = incrementBy;
          return acc;
        }, {})
      },
      migrationVersion,
      typeMigrationVersion,
      managed,
      updated_at: time
    });
    const raw = this._serializer.savedObjectToRaw(migrated);
    const body = await this.client.update({
      id: raw._id,
      index: this.getIndexForType(type),
      refresh,
      require_alias: true,
      _source: true,
      body: {
        script: {
          source: `
              for (int i = 0; i < params.counterFieldNames.length; i++) {
                def counterFieldName = params.counterFieldNames[i];
                def count = params.counts[i];

                if (ctx._source[params.type][counterFieldName] == null) {
                  ctx._source[params.type][counterFieldName] = count;
                }
                else {
                  ctx._source[params.type][counterFieldName] += count;
                }
              }
              ctx._source.updated_at = params.time;
            `,
          lang: 'painless',
          params: {
            counts: normalizedCounterFields.map(normalizedCounterField => normalizedCounterField.incrementBy),
            counterFieldNames: normalizedCounterFields.map(normalizedCounterField => normalizedCounterField.fieldName),
            time,
            type
          }
        },
        upsert: raw._source
      }
    });
    const {
      originId
    } = (_body$get$_source2 = (_body$get4 = body.get) === null || _body$get4 === void 0 ? void 0 : _body$get4._source) !== null && _body$get$_source2 !== void 0 ? _body$get$_source2 : {};
    return {
      id,
      type,
      ...(savedObjectNamespaces && {
        namespaces: savedObjectNamespaces
      }),
      ...(originId && {
        originId
      }),
      updated_at: time,
      references: (_body$get$_source$ref = (_body$get5 = body.get) === null || _body$get5 === void 0 ? void 0 : _body$get5._source.references) !== null && _body$get$_source$ref !== void 0 ? _body$get$_source$ref : [],
      version: (0, _coreSavedObjectsBaseServerInternal.encodeHitVersion)(body),
      attributes: (_body$get6 = body.get) === null || _body$get6 === void 0 ? void 0 : _body$get6._source[type],
      ...(managed && {
        managed
      })
    };
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.openPointInTimeForType}
   */
  async openPointInTimeForType(type, options = {}, internalOptions = {}) {
    const {
      disableExtensions
    } = internalOptions;
    let namespaces;
    if (disableExtensions || !this._spacesExtension) {
      var _options$namespaces2;
      namespaces = (_options$namespaces2 = options.namespaces) !== null && _options$namespaces2 !== void 0 ? _options$namespaces2 : [_coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING];
      // If the consumer specified `namespaces: []`, throw a Bad Request error
      if (namespaces.length === 0) throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.namespaces cannot be an empty array');
    }
    const {
      keepAlive = '5m',
      preference
    } = options;
    const types = Array.isArray(type) ? type : [type];
    const allowedTypes = types.filter(t => this._allowedTypes.includes(t));
    if (allowedTypes.length === 0) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError();
    }
    if (!disableExtensions && this._spacesExtension) {
      try {
        namespaces = await this._spacesExtension.getSearchableNamespaces(options.namespaces);
      } catch (err) {
        if (_boom.default.isBoom(err) && err.output.payload.statusCode === 403) {
          // The user is not authorized to access any space, throw a bad request error.
          throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError();
        }
        throw err;
      }
      if (namespaces.length === 0) {
        // The user is authorized to access *at least one space*, but not any of the spaces they requested; throw a bad request error.
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError();
      }
    }
    if (!disableExtensions && this._securityExtension) {
      await this._securityExtension.authorizeOpenPointInTime({
        namespaces: new Set(namespaces),
        types: new Set(types)
      });
    }
    const esOptions = {
      index: this.getIndicesForTypes(allowedTypes),
      keep_alive: keepAlive,
      ...(preference ? {
        preference
      } : {})
    };
    const {
      body,
      statusCode,
      headers
    } = await this.client.openPointInTime(esOptions, {
      ignore: [404],
      meta: true
    });
    if (statusCode === 404) {
      if (!(0, _coreElasticsearchServerInternal.isSupportedEsServer)(headers)) {
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
      } else {
        throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError();
      }
    }
    return {
      id: body.id
    };
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.closePointInTime}
   */
  async closePointInTime(id, options, internalOptions = {}) {
    const {
      disableExtensions
    } = internalOptions;
    if (!disableExtensions && this._securityExtension) {
      this._securityExtension.auditClosePointInTime();
    }
    return await this.client.closePointInTime({
      body: {
        id
      }
    });
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.createPointInTimeFinder}
   */
  createPointInTimeFinder(findOptions, dependencies, internalOptions) {
    return new _point_in_time_finder.PointInTimeFinder(findOptions, {
      logger: this._logger,
      client: this,
      ...dependencies,
      internalOptions
    });
  }

  /**
   * {@inheritDoc ISavedObjectsRepository.getCurrentNamespace}
   */
  getCurrentNamespace(namespace) {
    if (this._spacesExtension) {
      return this._spacesExtension.getCurrentNamespace(namespace);
    }
    return (0, _internal_utils.normalizeNamespace)(namespace);
  }

  /**
   * Returns index specified by the given type or the default index
   *
   * @param type - the type
   */
  getIndexForType(type) {
    return (0, _coreSavedObjectsBaseServerInternal.getIndexForType)({
      type,
      defaultIndex: this._index,
      typeRegistry: this._registry,
      kibanaVersion: this._migrator.kibanaVersion
    });
  }

  /**
   * Returns an array of indices as specified in `this._registry` for each of the
   * given `types`. If any of the types don't have an associated index, the
   * default index `this._index` will be included.
   *
   * @param types The types whose indices should be retrieved
   */
  getIndicesForTypes(types) {
    return unique(types.map(t => this.getIndexForType(t)));
  }
  _rawToSavedObject(raw, options) {
    const savedObject = this._serializer.rawToSavedObject(raw, options);
    const {
      namespace,
      type
    } = savedObject;
    if (this._registry.isSingleNamespace(type)) {
      savedObject.namespaces = [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)];
    }
    return (0, _lodash.omit)(savedObject, ['namespace']);
  }
  rawDocExistsInNamespaces(raw, namespaces) {
    return (0, _internal_utils.rawDocExistsInNamespaces)(this._registry, raw, namespaces);
  }
  rawDocExistsInNamespace(raw, namespace) {
    return (0, _internal_utils.rawDocExistsInNamespace)(this._registry, raw, namespace);
  }

  /**
   * Pre-flight check to ensure that a multi-namespace object exists in the current namespace.
   */
  async preflightCheckNamespaces({
    type,
    id,
    namespace,
    initialNamespaces
  }) {
    if (!this._registry.isMultiNamespace(type)) {
      throw new Error(`Cannot make preflight get request for non-multi-namespace type '${type}'.`);
    }
    const {
      body,
      statusCode,
      headers
    } = await this.client.get({
      id: this._serializer.generateRawId(undefined, type, id),
      index: this.getIndexForType(type)
    }, {
      ignore: [404],
      meta: true
    });
    const namespaces = initialNamespaces !== null && initialNamespaces !== void 0 ? initialNamespaces : [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)];
    const indexFound = statusCode !== 404;
    if (indexFound && isFoundGetResponse(body)) {
      if (!this.rawDocExistsInNamespaces(body, namespaces)) {
        return {
          checkResult: 'found_outside_namespace'
        };
      }
      return {
        checkResult: 'found_in_namespace',
        savedObjectNamespaces: initialNamespaces !== null && initialNamespaces !== void 0 ? initialNamespaces : getSavedObjectNamespaces(namespace, body),
        rawDocSource: body
      };
    } else if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
      statusCode,
      headers
    })) {
      // checking if the 404 is from Elasticsearch
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    return {
      checkResult: 'not_found',
      savedObjectNamespaces: initialNamespaces !== null && initialNamespaces !== void 0 ? initialNamespaces : getSavedObjectNamespaces(namespace)
    };
  }

  /**
   * Pre-flight check to ensure that an upsert which would create a new object does not result in an alias conflict.
   */
  async preflightCheckForUpsertAliasConflict(type, id, namespace) {
    const namespaceString = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace);
    const [{
      error
    }] = await (0, _preflight_check_for_create.preflightCheckForCreate)({
      registry: this._registry,
      client: this.client,
      serializer: this._serializer,
      getIndexForType: this.getIndexForType.bind(this),
      createPointInTimeFinder: this.createPointInTimeFinder.bind(this),
      objects: [{
        type,
        id,
        namespaces: [namespaceString]
      }]
    });
    if ((error === null || error === void 0 ? void 0 : error.type) === 'aliasConflict') {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id);
    }
    // any other error from this check does not matter
  }

  /** The `initialNamespaces` field (create, bulkCreate) is used to create an object in an initial set of spaces. */
  validateInitialNamespaces(type, initialNamespaces) {
    if (!initialNamespaces) {
      return;
    }
    if (this._registry.isNamespaceAgnostic(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"initialNamespaces" cannot be used on space-agnostic types');
    } else if (!initialNamespaces.length) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"initialNamespaces" must be a non-empty array of strings');
    } else if (!this._registry.isShareable(type) && (initialNamespaces.length > 1 || initialNamespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING))) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"initialNamespaces" can only specify a single space when used with space-isolated types');
    }
  }

  /** The object-specific `namespaces` field (bulkGet) is used to check if an object exists in any of a given number of spaces. */
  validateObjectNamespaces(type, id, namespaces) {
    if (!namespaces) {
      return;
    }
    if (this._registry.isNamespaceAgnostic(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"namespaces" cannot be used on space-agnostic types');
    } else if (!namespaces.length) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    } else if (!this._registry.isShareable(type) && (namespaces.length > 1 || namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING))) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"namespaces" can only specify a single space when used with space-isolated types');
    }
  }

  /** Validate a migrated doc against the registered saved object type's schema. */
  validateObjectForCreate(type, doc) {
    if (!this._registry.getType(type)) {
      return;
    }
    const validator = this.getTypeValidator(type);
    try {
      validator.validate(doc, this._migrator.kibanaVersion);
    } catch (error) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(error.message);
    }
  }
  getTypeValidator(type) {
    if (!this.typeValidatorMap[type]) {
      var _schemas;
      const savedObjectType = this._registry.getType(type);
      this.typeValidatorMap[type] = new _coreSavedObjectsBaseServerInternal.SavedObjectsTypeValidator({
        logger: this._logger.get('type-validator'),
        type,
        validationMap: (_schemas = savedObjectType.schemas) !== null && _schemas !== void 0 ? _schemas : {},
        defaultVersion: this._migrator.kibanaVersion
      });
    }
    return this.typeValidatorMap[type];
  }

  /** This is used when objects are created. */
  validateOriginId(type, objectOrOptions) {
    if (Object.keys(objectOrOptions).includes('originId') && !this._registry.isMultiNamespace(type)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"originId" can only be set for multi-namespace object types');
    }
  }

  /**
   * Saved objects with encrypted attributes should have IDs that are hard to guess, especially since IDs are part of the AAD used during
   * encryption, that's why we control them within this function and don't allow consumers to specify their own IDs directly for encryptable
   * types unless overwriting the original document.
   */
  getValidId(type, id, version, overwrite) {
    var _this$_encryptionExte;
    if (!((_this$_encryptionExte = this._encryptionExtension) !== null && _this$_encryptionExte !== void 0 && _this$_encryptionExte.isEncryptableType(type))) {
      return id || _coreSavedObjectsUtilsServer.SavedObjectsUtils.generateId();
    }
    if (!id) {
      return _coreSavedObjectsUtilsServer.SavedObjectsUtils.generateId();
    }
    // only allow a specified ID if we're overwriting an existing ESO with a Version
    // this helps us ensure that the document really was previously created using ESO
    // and not being used to get around the specified ID limitation
    const canSpecifyID = overwrite && version || _coreSavedObjectsUtilsServer.SavedObjectsUtils.isRandomId(id);
    if (!canSpecifyID) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('Predefined IDs are not allowed for saved objects with encrypted attributes unless the ID is a UUID.');
    }
    return id;
  }
  async optionallyEncryptAttributes(type, id, namespaceOrNamespaces, attributes) {
    var _this$_encryptionExte2;
    if (!((_this$_encryptionExte2 = this._encryptionExtension) !== null && _this$_encryptionExte2 !== void 0 && _this$_encryptionExte2.isEncryptableType(type))) {
      return attributes;
    }
    const namespace = Array.isArray(namespaceOrNamespaces) ? namespaceOrNamespaces[0] : namespaceOrNamespaces;
    const descriptor = {
      type,
      id,
      namespace
    };
    return this._encryptionExtension.encryptAttributes(descriptor, attributes);
  }
  async optionallyDecryptAndRedactSingleResult(object, typeMap, originalAttributes) {
    var _this$_encryptionExte3;
    if ((_this$_encryptionExte3 = this._encryptionExtension) !== null && _this$_encryptionExte3 !== void 0 && _this$_encryptionExte3.isEncryptableType(object.type)) {
      object = await this._encryptionExtension.decryptOrStripResponseAttributes(object, originalAttributes);
    }
    if (typeMap) {
      return this._securityExtension.redactNamespaces({
        typeMap,
        savedObject: object
      });
    }
    return object;
  }
  async optionallyDecryptAndRedactBulkResult(response, typeMap, originalObjects) {
    const modifiedObjects = await Promise.all(response.saved_objects.map(async (object, index) => {
      if (object.error) {
        // If the bulk operation failed, the object will not have an attributes field at all, it will have an error field instead.
        // In this case, don't attempt to decrypt, just return the object.
        return object;
      }
      const originalAttributes = originalObjects === null || originalObjects === void 0 ? void 0 : originalObjects[index].attributes;
      return await this.optionallyDecryptAndRedactSingleResult(object, typeMap, originalAttributes);
    }));
    return {
      ...response,
      saved_objects: modifiedObjects
    };
  }
}

/**
 * Returns a string array of namespaces for a given saved object. If the saved object is undefined, the result is an array that contains the
 * current namespace. Value may be undefined if an existing saved object has no namespaces attribute; this should not happen in normal
 * operations, but it is possible if the Elasticsearch document is manually modified.
 *
 * @param namespace The current namespace.
 * @param document Optional existing saved object that was obtained in a preflight operation.
 */
exports.SavedObjectsRepository = SavedObjectsRepository;
function getSavedObjectNamespaces(namespace, document) {
  if (document) {
    var _document$_source;
    return (_document$_source = document._source) === null || _document$_source === void 0 ? void 0 : _document$_source.namespaces;
  }
  return [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)];
}

/**
 * Extracts the contents of a decorated error to return the attributes for bulk operations.
 */
const errorContent = error => error.output.payload;
const unique = array => [...new Set(array)];

/**
 * Type and type guard function for converting a possibly not existent doc to an existent doc.
 */

const isFoundGetResponse = doc => doc.found;