"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSendCurrentRequest = void 0;
var _i18n = require("@kbn/i18n");
var _react = require("react");
var _shared_imports = require("../../../shared_imports");
var _history = require("../../../services/history");
var _editor_registry = require("../../contexts/editor_context/editor_registry");
var _contexts = require("../../contexts");
var _storage_quota_error = require("../../components/storage_quota_error");
var _send_request = require("./send_request");
var _track = require("./track");
var _utils = require("../../../lib/utils");
var _services = require("../../../services");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const useSendCurrentRequest = () => {
  const {
    services: {
      history,
      settings,
      notifications,
      trackUiMetric,
      http,
      autocompleteInfo,
      storage
    },
    theme$
  } = (0, _contexts.useServicesContext)();
  const dispatch = (0, _contexts.useRequestActionContext)();
  return (0, _react.useCallback)(async () => {
    try {
      const editor = _editor_registry.instance.getInputEditor();
      const variables = storage.get(_services.StorageKeys.VARIABLES, _constants.DEFAULT_VARIABLES);
      let requests = await editor.getRequestsInRange();
      requests = (0, _utils.replaceVariables)(requests, variables);
      if (!requests.length) {
        notifications.toasts.add(_i18n.i18n.translate('console.notification.error.noRequestSelectedTitle', {
          defaultMessage: 'No request selected. Select a request by placing the cursor inside it.'
        }));
        return;
      }
      dispatch({
        type: 'sendRequest',
        payload: undefined
      });

      // Fire and forget
      setTimeout(() => (0, _track.track)(requests, editor, trackUiMetric), 0);
      const results = await (0, _send_request.sendRequest)({
        http,
        requests
      });
      let saveToHistoryError;
      const {
        isHistoryEnabled
      } = settings.toJSON();
      if (isHistoryEnabled) {
        results.forEach(({
          request: {
            path,
            method,
            data
          }
        }) => {
          try {
            history.addToHistory(path, method, data);
          } catch (e) {
            // Grab only the first error
            if (!saveToHistoryError) {
              saveToHistoryError = e;
            }
          }
        });
      }
      if (saveToHistoryError) {
        const errorTitle = _i18n.i18n.translate('console.notification.error.couldNotSaveRequestTitle', {
          defaultMessage: 'Could not save request to Console history.'
        });
        if ((0, _history.isQuotaExceededError)(saveToHistoryError)) {
          const toast = notifications.toasts.addWarning({
            title: _i18n.i18n.translate('console.notification.error.historyQuotaReachedMessage', {
              defaultMessage: 'Request history is full. Clear the console history or disable saving new requests.'
            }),
            text: (0, _shared_imports.toMountPoint)((0, _storage_quota_error.StorageQuotaError)({
              onClearHistory: () => {
                history.clearHistory();
                notifications.toasts.remove(toast);
              },
              onDisableSavingToHistory: () => {
                settings.setIsHistoryEnabled(false);
                notifications.toasts.remove(toast);
              }
            }), {
              theme$
            })
          });
        } else {
          // Best effort, but still notify the user.
          notifications.toasts.addError(saveToHistoryError, {
            title: errorTitle
          });
        }
      }
      const {
        polling
      } = settings.toJSON();
      if (polling) {
        // If the user has submitted a request against ES, something in the fields, indices, aliases,
        // or templates may have changed, so we'll need to update this data. Assume that if
        // the user disables polling they're trying to optimize performance or otherwise
        // preserve resources, so they won't want this request sent either.
        autocompleteInfo.retrieve(settings, settings.getAutocomplete());
      }
      dispatch({
        type: 'requestSuccess',
        payload: {
          data: results
        }
      });
    } catch (e) {
      if (e !== null && e !== void 0 && e.response) {
        dispatch({
          type: 'requestFail',
          payload: e
        });
      } else {
        dispatch({
          type: 'requestFail',
          payload: undefined
        });
        notifications.toasts.addError(e, {
          title: _i18n.i18n.translate('console.notification.error.unknownErrorTitle', {
            defaultMessage: 'Unknown Request Error'
          })
        });
      }
    }
  }, [storage, dispatch, http, settings, notifications.toasts, trackUiMetric, history, theme$, autocompleteInfo]);
};
exports.useSendCurrentRequest = useSendCurrentRequest;