"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineGetComplianceDashboardRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _configSchema = require("@kbn/config-schema");
var _get_safe_posture_type_runtime_mapping = require("../../../common/runtime_mappings/get_safe_posture_type_runtime_mapping");
var _constants = require("../../../common/constants");
var _get_grouped_findings_evaluation = require("./get_grouped_findings_evaluation");
var _get_clusters = require("./get_clusters");
var _get_stats = require("./get_stats");
var _get_trends = require("./get_trends");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getClustersTrends = (clustersWithoutTrends, trends) => clustersWithoutTrends.map(cluster => ({
  ...cluster,
  trend: trends.map(({
    timestamp,
    clusters: clustersTrendData
  }) => ({
    timestamp,
    ...clustersTrendData[cluster.meta.assetIdentifierId]
  }))
}));
const getSummaryTrend = trends => trends.map(({
  timestamp,
  summary
}) => ({
  timestamp,
  ...summary
}));
const queryParamsSchema = {
  params: _configSchema.schema.object({
    policy_template: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.CSPM_POLICY_TEMPLATE), _configSchema.schema.literal(_constants.KSPM_POLICY_TEMPLATE)])
  })
};
const defineGetComplianceDashboardRoute = router => router.get({
  path: _constants.STATS_ROUTE_PATH,
  validate: queryParamsSchema,
  options: {
    tags: ['access:cloud-security-posture-read']
  }
}, async (context, request, response) => {
  const cspContext = await context.csp;
  try {
    const esClient = cspContext.esClient.asCurrentUser;
    const {
      id: pitId
    } = await esClient.openPointInTime({
      index: _constants.LATEST_FINDINGS_INDEX_DEFAULT_NS,
      keep_alive: '30s'
    });
    const policyTemplate = request.params.policy_template;

    // runtime mappings create the `safe_posture_type` field, which equals to `kspm` or `cspm` based on the value and existence of the `posture_type` field which was introduced at 8.7
    // the `query` is then being passed to our getter functions to filter per posture type even for older findings before 8.7
    const runtimeMappings = (0, _get_safe_posture_type_runtime_mapping.getSafePostureTypeRuntimeMapping)();
    const query = {
      bool: {
        filter: [{
          term: {
            safe_posture_type: policyTemplate
          }
        }]
      }
    };
    const [stats, groupedFindingsEvaluation, clustersWithoutTrends, trends] = await Promise.all([(0, _get_stats.getStats)(esClient, query, pitId, runtimeMappings), (0, _get_grouped_findings_evaluation.getGroupedFindingsEvaluation)(esClient, query, pitId, runtimeMappings), (0, _get_clusters.getClusters)(esClient, query, pitId, runtimeMappings), (0, _get_trends.getTrends)(esClient, policyTemplate)]);

    // Try closing the PIT, if it fails we can safely ignore the error since it closes itself after the keep alive
    //   ends. Not waiting on the promise returned from the `closePointInTime` call to avoid delaying the request
    esClient.closePointInTime({
      id: pitId
    }).catch(err => {
      cspContext.logger.warn(`Could not close PIT for stats endpoint: ${err}`);
    });
    const clusters = getClustersTrends(clustersWithoutTrends, trends);
    const trend = getSummaryTrend(trends);
    const body = {
      stats,
      groupedFindingsEvaluation,
      clusters,
      trend
    };
    return response.ok({
      body
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    cspContext.logger.error(`Error while fetching CSP stats: ${err}`);
    return response.customError({
      body: {
        message: error.message
      },
      statusCode: error.statusCode
    });
  }
});
exports.defineGetComplianceDashboardRoute = defineGetComplianceDashboardRoute;