"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFindingsByResource = exports.getFindingsByResourceAggQuery = void 0;
var _reactQuery = require("@tanstack/react-query");
var _rxjs = require("rxjs");
var _get_belongs_to_runtime_mapping = require("../../../../common/runtime_mappings/get_belongs_to_runtime_mapping");
var _constants = require("../../../common/constants");
var _use_kibana = require("../../../common/hooks/use_kibana");
var _show_error_toast = require("../../../common/utils/show_error_toast");
var _utils = require("../utils/utils");
var _constants2 = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Maximum number of grouped findings, default limit in elasticsearch is set to 65,536 (ref: https://www.elastic.co/guide/en/elasticsearch/reference/current/search-settings.html#search-settings-max-buckets)
const MAX_BUCKETS = 60 * 1000;
const getFindingsByResourceAggQuery = ({
  query,
  sortDirection
}) => ({
  index: _constants2.CSP_LATEST_FINDINGS_DATA_VIEW,
  query,
  size: 0,
  runtime_mappings: (0, _get_belongs_to_runtime_mapping.getBelongsToRuntimeMapping)(),
  aggs: {
    ...(0, _utils.getFindingsCountAggQuery)(),
    resource_total: {
      cardinality: {
        field: 'resource.id'
      }
    },
    resources: {
      terms: {
        field: 'resource.id',
        size: MAX_BUCKETS
      },
      aggs: {
        name: {
          terms: {
            field: 'resource.name',
            size: 1
          }
        },
        subtype: {
          terms: {
            field: 'resource.sub_type',
            size: 1
          }
        },
        benchmarkName: {
          terms: {
            field: 'rule.benchmark.name'
          }
        },
        cis_sections: {
          terms: {
            field: 'rule.section'
          }
        },
        failed_findings: {
          filter: {
            term: {
              'result.evaluation': 'failed'
            }
          }
        },
        passed_findings: {
          filter: {
            term: {
              'result.evaluation': 'passed'
            }
          }
        },
        // this field is runtime generated
        belongs_to: {
          terms: {
            field: 'belongs_to',
            size: 1
          }
        },
        compliance_score: {
          bucket_script: {
            buckets_path: {
              passed: 'passed_findings>_count',
              failed: 'failed_findings>_count'
            },
            script: 'params.passed / (params.passed + params.failed)'
          }
        },
        sort_by_compliance_score: {
          bucket_sort: {
            size: _constants.MAX_FINDINGS_TO_LOAD,
            sort: [{
              compliance_score: {
                order: sortDirection
              },
              _count: {
                order: 'desc'
              },
              _key: {
                order: 'asc'
              }
            }]
          }
        }
      }
    }
  },
  ignore_unavailable: false
});
exports.getFindingsByResourceAggQuery = getFindingsByResourceAggQuery;
const getFirstKey = buckets => {
  if (!!Array.isArray(buckets) && !!buckets.length) return buckets[0].key;
};
const getKeysList = buckets => {
  if (!!Array.isArray(buckets) && !!buckets.length) return buckets.map(v => v.key);
};
const createFindingsByResource = resource => ({
  resource_id: resource.key,
  ['resource.name']: getFirstKey(resource.name.buckets),
  ['resource.sub_type']: getFirstKey(resource.subtype.buckets),
  ['rule.section']: getKeysList(resource.cis_sections.buckets),
  ['rule.benchmark.name']: getFirstKey(resource.benchmarkName.buckets),
  belongs_to: getFirstKey(resource.belongs_to.buckets),
  compliance_score: resource.compliance_score.value,
  findings: {
    failed_findings: resource.failed_findings.doc_count,
    normalized: resource.doc_count > 0 ? resource.failed_findings.doc_count / resource.doc_count : 0,
    total_findings: resource.doc_count,
    passed_findings: resource.passed_findings.doc_count
  }
});
const useFindingsByResource = options => {
  const {
    data,
    notifications: {
      toasts
    }
  } = (0, _use_kibana.useKibana)().services;
  const params = {
    ...options
  };
  return (0, _reactQuery.useQuery)(['csp_findings_resource', {
    params
  }], async () => {
    const {
      rawResponse: {
        aggregations
      }
    } = await (0, _rxjs.lastValueFrom)(data.search.search({
      params: getFindingsByResourceAggQuery(params)
    }));
    if (!aggregations) throw new Error('Failed to aggregate by, missing resource id');
    if (!Array.isArray(aggregations.resources.buckets) || !Array.isArray(aggregations.count.buckets)) throw new Error('Failed to group by, missing resource id');
    const page = aggregations.resources.buckets.map(createFindingsByResource);
    return {
      page,
      total: aggregations.resource_total.value,
      count: (0, _utils.getAggregationCount)(aggregations.count.buckets)
    };
  }, {
    enabled: options.enabled,
    keepPreviousData: true,
    onError: err => (0, _show_error_toast.showErrorToast)(toasts, err)
  });
};
exports.useFindingsByResource = useFindingsByResource;